	# ICBN parameters
# epsGuess:  allowed violations for an edge to be included
# epsStop: stop EM if likelihood difference below epsStop
# nIter: Number of iterations in the annealing run
# tInit: Initial temperature in the annealing run
# tAlpha: Temperature for preselecting propositions in the annealing run
# tFactor: cooling factor in the annealing run
# maxIter: Maximal number of iterations in isotone
estimate_icbn_model <- function(X, Y, epsGuess = 0.05, epsStop=1.0, nIter=100, tInit = 100, tAlpha = 0.1, tFactor=1, maxIter=200 ) {
	outputPath = NA
	icbnModel <- doAnneal(X, Y, outputPath, tInit=tInit, nIter=nIter, loggingFlag=FALSE, tAlpha=tAlpha, epsStop=epsStop,
			epsGuess=epsGuess, tFactor=tFactor, maxIter=maxIter, plotFlag=FALSE, saveFlag=FALSE)
	
	
	 lattice <- cbind(icbnModel$H[icbnModel$G > 0,], icbnModel$mus)
	 
	 list(lattice=lattice, poset=icbnModel$P)
}

write_poset <- function(poset, base_dir, cbn_name) {
  filename <- paste(base_dir, cbn_name, ".poset", sep='')
  write.table(ncol(poset), file = filename, sep = " ", row.names = FALSE, col.names = FALSE) 
  if(ncol(poset) > 1 & sum(poset) > 0)
  {
    edges <- matrix(convert2EdgeList(poset), ncol = 2, byrow=TRUE)
    write.table(edges, file = filename, sep = " ", row.names = FALSE, col.names = FALSE, append=TRUE)
  } 
  write.table(0, file = filename, sep = " ", row.names = FALSE, col.names = FALSE, append=TRUE) 
}

write_pat_file <- function(events, base_dir, cbn_name) {
  events <- cbind(1, events)
  pat_file = paste(base_dir, cbn_name, ".pat", sep='')
  
  write.table(t(dim(events)), file = pat_file, sep = " ", col.names = FALSE, row.names=FALSE)
  write.table(events, file = pat_file, sep = " ", col.names = FALSE, row.names=FALSE, append=TRUE)
  
}

est_ctcbn_on_events <- function(X, poset, temp_path, ctcbn_folder) {
  cbn_name = "temp"
  base_dir = paste(temp_path, cbn_name, "/", sep='')
  dir.create(paste(base_dir, cbn_name, "/", sep='') , showWarnings = FALSE, recursive = TRUE)
  write_poset(poset, base_dir, cbn_name) 
  write_pat_file(X, base_dir, cbn_name)
  
  pre_file_name <- paste(base_dir, cbn_name, sep='')
  
  d = system(paste(ctcbn_folder, "h-cbn -f ", pre_file_name, " -w", sep = '') )
  
  lambdas_est = unlist(read.table(paste(pre_file_name, ".lambda", sep='')))
  
  unlink(paste(base_dir, sep=''), recursive = TRUE, force = TRUE)
  
  lambdas_est[2:length(lambdas_est)]
}


generate_input_files_for_ode_model <- function(X, Y, mutation_names, ctcbn_folder, temp_path, ...) {

	fit = estimate_icbn_model(X, Y, ...)
	lambdas = est_ctcbn_on_events(X, fit$poset, temp_path, ctcbn_folder)
	list(poset = fit$poset, lattice=fit$lattice, lambdas = lambdas, mutation_names=mutation_names, data=cbind(X,logrf=Y))
}

save_cbn_model <- function(cbn_model, cbn_name, output_path) {
	dir.create(output_path, showWarnings = FALSE, recursive = TRUE)
	write_poset(cbn_model$poset, output_path, cbn_name)
	write.table(cbn_model$lattice, file = paste(output_path, cbn_name, ".lattice", sep=''), sep = " ", row.names = FALSE, col.names = FALSE)
	write.table(cbn_model$lambdas, file = paste(output_path, cbn_name, ".lambdas", sep=''), sep = " ", row.names = FALSE, col.names = FALSE)
	write.table(cbn_model$mutation_names, file = paste(output_path, cbn_name, ".mutations", sep=''), sep = " ", row.names = FALSE, col.names = FALSE)
	write.table(cbn_model$data, file = paste(output_path, cbn_name, ".data", sep=''), sep = " ", row.names = FALSE, col.names = FALSE)
}

