%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%------------------------------------------------------------------------
%% DESCRIPTION:
%%%------------------------------------------------------------------------
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% This function simulates viral dynamics under indinavir monotherapy. The
% inputs for the code include fitness costs and resistance factors for
% different mutant genotypes and drug efficacy. The code computes the time
% profiles of the mutant genotypes, mutations and selective advantages.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% For reference, please cite

% Estimating HIV-1 Fitness Characteristics from Cross-sectional Genotype
% Data, Sathej Gopalakrishnan, Hesam Montazeri, Stephan Menz, Niko
% Beerenwinkel and Wilhelm Huisinga (Submitted)

% Details of the viral dynamics model and mutation models found there-in

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function IDV_Simulation

clear all;
close all
clc


epsilon = 0.90; %Drug efficacy of ZDV on wild-type
DrugConcFold = epsilon/(1-epsilon); % C/IC50 for the wild-type assuming an Emax kind of model

% -------------------------------------------------------------------------
% Average Waiting times for each site of mutation computed from CT-CBN
% -------------------------------------------------------------------------

lambda = textread('IDV_lambdas');
T_data = [1/lambda(5) + 1/lambda(1); 1/lambda(2) + 1/lambda(3); 1/lambda(3); 1/lambda(3) + 1/lambda(4); 1/lambda(5)];
[min_T_data index_min_T_data] = min(T_data);
T_stat = T_data./min_T_data; % Normalize by shortest time

% -------------------------------------------------------------------------

% Key to numbering of mutations

% 1 : M46I
% 2 : I54V
% 3 : A71V
% 4 : V82A
% 5 : L90M

% -------------------------------------------------------------------------
%                              Mutant Genotypes
% -------------------------------------------------------------------------

% Mut1  : {5}
% Mut2  : {1,5}
% Mut3  : {3,5}
% Mut4  : {3}
% Mut5  : {2,3}
% Mut6  : {3,4}
% Mut7  : {2,3,4}
% Mut8  : {2,3,5}
% Mut9  : {3,4,5}
% Mut10 : {2,3,4,5}
% Mut11 : {1,3,5}
% Mut12 : {1,3,4,5}
% Mut13 : {1,2,3,5}
% Mut14 : {1,2,3,4,5}

% Resistance factors estimated from ICBN model

%     1.0000  : WT
%     3.8905  : Mut1
%    19.0546  : Mut2
%    12.0226  : Mut3
%     2.0893  : Mut4
%    19.4984  : Mut5
%    19.4984  : Mut6
%    19.4984  : Mut7
%    33.1131  : Mut8
%    33.1131  : Mut9
%    33.1131  : Mut10
%    19.0546  : Mut11
%    46.7735  : Mut12
%    46.7735  : Mut13
%    46.7735  : Mut14

% Resistances re-defined to enter the model appropriately

par.Res = [1;1/3.8905;1/19.0546;1/12.0226;1/2.0893;1/19.4984; 1/19.4984; 1/19.4984; 1/33.1131; 1/33.1131; 1/33.1131; 1/19.0546; 1/46.7735; 1/46.7735; 1/46.7735];
par.Res = par.Res./par.Res(1); % Normalize so that Res(WT) = 1
Res = par.Res;

% Vector of fitness costs

s = [0
    0.359519887014450
    0.655066826595293
    0.312601704833754
    0.453220598938908
    0.493440498603359
    0.420438100908267
    0.578910930799280
    0.278840425530371
    0.386767650778210
    0.284064322311880
    0.440140021876466
    0.312873756227845
    0.151788841851837
    0.042593640872153];

% Fitness costs averaged across number of sites of mutation

s_avg = [(s(2) + s(5))/2 ; (s(3) + s(4) + s(6) +s(7))/4; (s(8) + s(9) +s(10)+s(12))/4 ; (s(11) + s(13) + s(14))/3; s(15)];

% -------------------------------------------------------------------------
% -------------Infection Model Parameters (in vivo) -----------------------
% -------------------------------------------------------------------------

% T-cells

par.lambda_T     = 2e9;
par.delta_T      = 0.02;
par.delta_PIC_T  = 0.35;
par.beta_T       = 8e-12;
par.rho_rev      = 0.33;
par.CL_T         = par.beta_T* (1/par.rho_rev - 1);
par.delta_T1     = 0.02;
par.k_T          = 0.35;
par.delta_T2     = 1;
par.CL_VI        = 23;
par.CL_VNI       = 23;
par.b            = 1;
par.q            = 1;
par.rho_PR       = 0.67;
par.Nhat_T       = 1000;
par.N_T_WT       = par.b*par.q*par.rho_PR*par.Nhat_T;
par.N_T_NI_WT    = par.Nhat_T - par.N_T_WT;

% Macrophages

par.lambda_M    = 6.9e7;
par.beta_M      = 1e-14;
par.Nhat_M      = 100;
par.delta_M     = 0.0069;
par.delta_M1    = 0.0069;
par.delta_M2    = 0.09;
par.k_M         = 0.07;
par.delta_PIC_M = 0.0035;
par.N_M_WT      = par.b*par.q*par.rho_PR*par.Nhat_M;
par.N_M_NI_WT   = par.Nhat_M - par.N_M_WT;
par.CL_M        = par.beta_M* (1/par.rho_rev - 1);

% Latent cells

par.p           = 8e-6;
par.alpha       = 1e-3;
par.delta_TL    = 1e-4;

% -------------------------------------------------------------------------
%      Mutation Probabilities, similar as in Stilianakis et al, 1998
% -------------------------------------------------------------------------

par.mu0        = 3.4e-5;
par.mu1        = par.mu0/3;  % Probability of G -> A mutations
par.mu2        = par.mu1/2;  % Probability of other non-transversion mutations
par.mu3        = par.mu1/10; % Probability of transversion mutations


% -------------------------------------------------------------------------
% -------------- Pre-treatment Steadystate Computation---------------------
% -------------------------------------------------------------------------

% Species ordered as (TU,MU,T1_WT,T2_WT,TL_WT,M1_WT,M2_WT,VI_WT,VNI_WT,T1_Mut1,T2_Mut1,VI_Mut1,...,T1_Mutn,T2_Mutn,VI_Mutn,VNI_Mut1,VNI_Mut2,...,VNI_Mutn)

tspan = [0 5000];
Xinit = zeros(7*length(Res)+2,1);
Xinit(1,1) = par.lambda_T/par.delta_T; % Initial TU
Xinit(2,1) = par.lambda_M/par.delta_M; % Initial MU
Xinit(8,1) = 1; % Initial infecting virus

par.epsilon = 0; % No drug

par.N_T    = zeros(length(Res),1);
par.N_T_NI = zeros(length(Res),1);
par.N_M    = zeros(length(Res),1);
par.N_M_NI = zeros(length(Res),1);

par.N_T    = (1-s).*par.N_T_WT;
par.N_T_NI = par.Nhat_T - par.N_T;
par.N_M    = (1-s).*par.N_M_WT;
par.N_M_NI = par.Nhat_M - par.N_M;

[t Xss] = ode15s(@Kleist_Menz_Huisinga_ODE,tspan,Xinit,[],par);

% -------------------------------------------------------------------------
% ------------------- AFTER DRUG TREATMENT WITH IDV -----------------------
% -------------------------------------------------------------------------

par.epsilon = epsilon;

par.N_T    = zeros(length(Res),1);
par.N_T_NI = zeros(length(Res),1);
par.N_M    = zeros(length(Res),1);
par.N_M_NI = zeros(length(Res),1);

F = DrugConcFold.*par.Res;

par.N_T(:,1)    = par.N_T_WT.*(1-s)./(par.rho_PR + (1 - par.rho_PR)./(1./(1+F)));
par.N_T_NI(:,1) = par.Nhat_T - par.N_T;
par.N_M(:,1)    = par.N_M_WT.*(1-s)./(par.rho_PR + (1 - par.rho_PR)./(1./(1+F)));
par.N_M_NI(:,1) = par.Nhat_M - par.N_M;

% -------------------------------------------------------------------------
%                               Solve the ODE system
% -------------------------------------------------------------------------

tspan = [0:1:1000];
[t X] = ode15s(@Kleist_Menz_Huisinga_ODE,tspan,Xss(end,:),[],par);

% -------------------------------------------------------------------------
% Compute Selective Advantages of different mutants relative to WT
% (normalized to 1) and also to compare amongst specific mutants
% -------------------------------------------------------------------------

DrugEff_fac_eta    = 1./(par.rho_PR + (1-par.rho_PR)./(1./(1+F))); % 1 - eta
SelectiveAdvantage = ((1- s(2:15)).*(DrugEff_fac_eta(2:15)))./(DrugEff_fac_eta(1));

SA_avg = [1;(SelectiveAdvantage(1) + SelectiveAdvantage(4))/2 ; (SelectiveAdvantage(2) + SelectiveAdvantage(3) + SelectiveAdvantage(5) +SelectiveAdvantage(6))/4; (SelectiveAdvantage(7) + SelectiveAdvantage(8) +SelectiveAdvantage(9)+SelectiveAdvantage(11))/4 ; (SelectiveAdvantage(10) + SelectiveAdvantage(12) + SelectiveAdvantage(13))/3; SelectiveAdvantage(14)];
V_tot = X(:,8) + X(:,9) + X(:,15) + X(:,21) + X(:,27) + X(:,33) + X(:,39) + X(:,45) + X(:,51) + X(:,57) + X(:,63) + X(:,69) + X(:,75) + X(:,81) + X(:,87) + X(:,93) + X(:,94) + X(:,95) + X(:,96) + X(:,97) + X(:,98) + X(:,99) + X(:,100) + X(:,101) + X(:,102)  + X(:,103) + X(:,104) + X(:,105) + X(:,106) + X(:,107);

figure(1)
semilogy(t,V_tot,'b-');xlim([0 270]);ylim([1e6 1e12]);
xlabel('Time (days)'); ylabel('Viral load (in numbers)');

% -------------------------------------------------------------------------
%                Compute abundance of different mutations
% -------------------------------------------------------------------------

Mutation1 = X(:,21) + X(:,75) + X(:,81) + X(:,87) + X(:,93) + X(:,95) + X(:,104) + X(:,105) + X(:,106) + X(:,107);
Mutation2 = X(:,39) + X(:,51) + X(:,57)+ X(:,69) + X(:,87) + X(:,93) + X(:,98) + X(:,100) + X(:,101) + X(:,103) + X(:,106) + X(:,107);
Mutation3 = X(:,27) + X(:,33) + X(:,39) + X(:,45) + X(:,51)+ X(:,57) + X(:,63) + X(:,69) + X(:,75) + X(:,81) + X(:,87) + X(:,93) + X(:,96) + X(:,97) + X(:,98) + X(:,99) + X(:,100) + X(:,101) + X(:,102) + X(:,103) + X(:,104) + X(:,105) + X(:,106) + X(:,107);
Mutation4 = X(:,45) + X(:,51) + X(:,63) + X(:,69) + X(:,81) + X(:,93) + X(:,99) + X(:,100) + X(:,102) + X(:,103) + X(:,105) + X(:,107);
Mutation5 = X(:,15) + X(:,21) + X(:,27) + X(:,57) + X(:,63) + X(:,69) + X(:,75) + X(:,81) + X(:,87) + X(:,93) + X(:,94) + X(:,95) + X(:,96) + X(:,101) + X(:,102) + X(:,103) + X(:,104) + X(:,105) + X(:,106) + X(:,107);

% -------------------------------------------------------------------------
%           Compute fraction of abundance of different mutations
% -------------------------------------------------------------------------

TotalLoad = X(:,8) + X(:,9) + X(:,15) + X(:,21) + X(:,27) + X(:,33) + X(:,39) + X(:,45) + X(:,51) + X(:,57) + X(:,63) + X(:,69) + X(:,75) + X(:,81) + X(:,87) + X(:,93) + X(:,94) + X(:,95) + X(:,96) + X(:,97) + X(:,98) + X(:,99) + X(:,100) + X(:,101) + X(:,102)  + X(:,103) + X(:,104) + X(:,105) + X(:,106) + X(:,107);
Mutation1_fraction = Mutation1./TotalLoad;
Mutation2_fraction = Mutation2./TotalLoad;
Mutation3_fraction = Mutation3./TotalLoad;
Mutation4_fraction = Mutation4./TotalLoad;
Mutation5_fraction = Mutation5./TotalLoad;

%--------------------------------------------------------------------------
%----------------- Computing mechanistic waiting times---------------------
% Detection criteria: Total viral load > 500 copies of RNA per ml and a
% mutation constitutes >20% of the total viral abundance
%--------------------------------------------------------------------------
%--------------------------------------------------------------------------

% Mutation 1

for i = 1:length(tspan)
    if (Mutation1_fraction(i) > 0.2) && (V_tot(i) > 1.2e8)
        Rebound_Mutation1 = tspan(i);
        break;
    end
end

% Mutation 2

for i = 1:length(tspan)
    if (Mutation2_fraction(i) > 0.2) && (V_tot(i) > 1.2e8)
        Rebound_Mutation2 = tspan(i);
        break;
    end
end

% Mutation 3

for i = 1:length(tspan)
    if (Mutation3_fraction(i) > 0.2) && (V_tot(i) > 1.2e8)
        Rebound_Mutation3 = tspan(i);
        break;
    end
end

% Mutation 4

for i = 1:length(tspan)
    if (Mutation4_fraction(i) > 0.2) && (V_tot(i) > 1.2e8)
        Rebound_Mutation4 = tspan(i);
        break;
    end
end

% Mutation 5

for i = 1:length(tspan)
    if (Mutation5_fraction(i) > 0.2) && (V_tot(i) > 1.2e8)
        Rebound_Mutation5 = tspan(i);
        break;
    end
end

% -------------------------------------------------------------------------
%  Computing waiting time to observe a mutation at each of the sites from
%                                  the model
% -------------------------------------------------------------------------

T_pred = zeros(5,1);
T_pred(1) = Rebound_Mutation1;
T_pred(2) = Rebound_Mutation2;
T_pred(3) = Rebound_Mutation3;
T_pred(4) = Rebound_Mutation4;
T_pred(5) = Rebound_Mutation5;

T_pred = T_pred./T_pred(index_min_T_data); % Normalize by smallest T_pred, as in data
T_mech = T_pred;

% Compare mechanistic and statistical waiting times

WaitingTimes = [T_stat T_mech]

end



function dX = Kleist_Menz_Huisinga_ODE(t,X,par)

lambda_T       = par.lambda_T;
delta_T        = par.delta_T;
delta_PIC_T    = par.delta_PIC_T;
delta_T1       = par.delta_T1;
beta_T         = par.beta_T;
CL_T           = par.CL_T;
k_T            = par.k_T;
delta_T2       = par.delta_T2;
CL_VI          = par.CL_VI;
CL_VNI         = par.CL_VNI;
Nhat_T         = par.Nhat_T;

lambda_M       = par.lambda_M;
delta_M        = par.delta_M;
delta_PIC_M    = par.delta_PIC_M;
delta_M1       = par.delta_M1;
beta_M         = par.beta_M;
CL_M           = par.CL_M;
k_M            = par.k_M;
delta_M2       = par.delta_M2;
Nhat_M         = par.Nhat_M;

p              = par.p;
alpha          = par.alpha;
delta_TL       = par.delta_TL;

Res= par.Res;

mu0 = par.mu0;
mu1 = par.mu1;
mu2 = par.mu2;
mu3 = par.mu3;

N_T    = par.N_T;
N_T_NI = par.N_T_NI;
N_M    = par.N_M;
N_M_NI = par.N_M_NI;


dX = zeros(7*length(Res) + 2,1);
TU      = X(1);
MU      = X(2);
T1_WT   = X(3);
T2_WT   = X(4);
TL_WT   = X(5);
M1_WT   = X(6);
M2_WT   = X(7);
VI_WT   = X(8);
VNI_WT  = X(9);

T1_Mut  = zeros(length(Res)-1,1);
T2_Mut  = zeros(length(Res)-1,1);
TL_Mut  = zeros(length(Res)-1,1);
M1_Mut  = zeros(length(Res)-1,1);
M2_Mut  = zeros(length(Res)-1,1);
VI_Mut  = zeros(length(Res)-1,1);
VNI_Mut = zeros(length(Res)-1,1);

j = 1;
for i = 1:length(Res)-1
    T1_Mut(i) = X(9+j);
    T2_Mut(i) = X(10+j);
    TL_Mut(i) = X(11+j);
    M1_Mut(i) = X(12+j);
    M2_Mut(i) = X(13+j);
    VI_Mut(i) = X(14+j);
    j = j+6;
end

% Upto X(93) defined above, i.e., VI_Mut(14) = X(93)

j = 94;
for i = 1:length(Res)-1
    VNI_Mut(i) = X(j);
    j = j+1;
end

T1 = [T1_WT; T1_Mut(1);T1_Mut(2);T1_Mut(3);T1_Mut(4);T1_Mut(5);T1_Mut(6);T1_Mut(7);T1_Mut(8);T1_Mut(9);T1_Mut(10);T1_Mut(11);T1_Mut(12);T1_Mut(13);T1_Mut(14)];
T2 = [T2_WT; T2_Mut(1);T2_Mut(2);T2_Mut(3);T2_Mut(4);T2_Mut(5);T2_Mut(6);T2_Mut(7);T2_Mut(8);T2_Mut(9);T2_Mut(10);T2_Mut(11);T2_Mut(12);T2_Mut(13);T2_Mut(14)];
TL = [TL_WT; TL_Mut(1);TL_Mut(2);TL_Mut(3);TL_Mut(4);TL_Mut(5);TL_Mut(6);TL_Mut(7);TL_Mut(8);TL_Mut(9);TL_Mut(10);TL_Mut(11);TL_Mut(12);TL_Mut(13);TL_Mut(14)];
M1 = [M1_WT; M1_Mut(1);M1_Mut(2);M1_Mut(3);M1_Mut(4);M1_Mut(5);M1_Mut(6);M1_Mut(7);M1_Mut(8);M1_Mut(9);M1_Mut(10);M1_Mut(11);M1_Mut(12);M1_Mut(13);M1_Mut(14)];
M2 = [M2_WT; M2_Mut(1);M2_Mut(2);M2_Mut(3);M2_Mut(4);M2_Mut(5);M2_Mut(6);M2_Mut(7);M2_Mut(8);M2_Mut(9);M2_Mut(10);M2_Mut(11);M2_Mut(12);M2_Mut(13);M2_Mut(14)];
VI = [VI_WT; VI_Mut(1);VI_Mut(2);VI_Mut(3);VI_Mut(4);VI_Mut(5);VI_Mut(6);VI_Mut(7);VI_Mut(8);VI_Mut(9);VI_Mut(10);VI_Mut(11);VI_Mut(12);VI_Mut(13);VI_Mut(14)];
VNI = [VNI_WT; VNI_Mut(1);VNI_Mut(2);VNI_Mut(3);VNI_Mut(4);VNI_Mut(5);VNI_Mut(6);VNI_Mut(7);VNI_Mut(8);VNI_Mut(9);VNI_Mut(10);VNI_Mut(11);VNI_Mut(12);VNI_Mut(13);VNI_Mut(14)];

dTU     =  lambda_T + delta_PIC_T*sum(T1) - delta_T*TU - TU*(sum(beta_T.*VI));
dMU     =  lambda_M + delta_PIC_M*sum(M1) - delta_M*MU - MU*(sum(beta_M.*VI));
dT1_WT  =  beta_T*VI_WT*TU - (delta_T1 + delta_PIC_T + k_T)*T1_WT;
dM1_WT  =  beta_M*VI_WT*MU - (delta_M1 + delta_PIC_M + k_M)*M1_WT;
dVI_WT  =  N_T(1)*T2_WT + N_M(1)*M2_WT - CL_VI*VI_WT - (CL_T*TU + beta_T*TU + CL_M*MU + beta_M*MU)*VI_WT;
dVNI_WT =  N_T_NI(1)*T2_WT + N_M_NI(1)*M2_WT - CL_VNI*VNI_WT;

for i = 2:length(Res)
    dT1_Mut(i-1) = beta_T*VI_Mut(i-1)*TU - (delta_T1 + delta_PIC_T + k_T)*T1_Mut(i-1);
    dM1_Mut(i-1) = beta_M*VI_Mut(i-1)*MU - (delta_M1 + delta_PIC_M + k_M)*M1_Mut(i-1);
    dVI_Mut(i-1) = N_T(i)*T2_Mut(i-1) + N_M(i)*M2_Mut(i-1) - CL_VI*VI_Mut(i-1) - (CL_T*TU + beta_T*TU + CL_M*MU + beta_M*MU)*VI_Mut(i-1);
    dVNI_Mut(i-1)=  N_T_NI(i)*T2_Mut(i-1) + N_M_NI(i)*M2_Mut(i-1) - CL_VNI*VNI_Mut(i-1);
end

% The following code segment describing production of T2, M2 and TL cells
% can be generalized (see code for ZDV monotherapy simulation, here it is
% coded manually to clarify)
    
dT2_WT      = (1-p)*((1-mu0)^5*k_T*T1_WT      + k_T*mu1*(1-mu0)^4*T1_Mut(1)     + k_T*mu2*(1-mu0)^4*T1_Mut(4))                              + alpha*TL_WT      - delta_T2*T2_WT;
dT2_Mut(1)  = (1-p)*((1-mu0)^5*k_T*T1_Mut(1)     + k_T*T1_WT*mu3*(1-mu0)^4         + k_T*mu2*(1-mu0)^4*T1_Mut(2) + k_T*mu2*(1-mu0)^4*T1_Mut(3)) + alpha*TL_Mut(1)  - delta_T2*T2_Mut(1);
dT2_Mut(2)  = (1-p)*((1-mu0)^5*k_T*T1_Mut(2)     + k_T*T1_Mut(1)*mu1*(1-mu0)^4     + k_T*mu2*(1-mu0)^4*T1_Mut(11))                             + alpha*TL_Mut(2) - delta_T2*T2_Mut(2);
dT2_Mut(3)  = (1-p)*((1-mu0)^5*k_T*T1_Mut(3)     + k_T*T1_Mut(1)*mu2*(1-mu0)^4     + k_T*mu3*(1-mu0)^4*T1_Mut(4) + k_T*mu1*(1-mu0)^4*T1_Mut(8) + k_T*mu2*(1-mu0)^4*T1_Mut(9) + k_T*mu2*(1-mu0)^4*T1_Mut(11)) + alpha*TL_Mut(3) - delta_T2*T2_Mut(3);
dT2_Mut(4)  = (1-p)*((1-mu0)^5*k_T*T1_Mut(4)     + k_T*T1_WT*mu2*(1-mu0)^4         + k_T*T1_Mut(3)*mu1*(1-mu0)^4 + k_T*mu1*(1-mu0)^4*T1_Mut(5) + k_T*mu2*(1-mu0)^4*T1_Mut(6)) + alpha*TL_Mut(4) - delta_T2*T2_Mut(4);
dT2_Mut(5)  = (1-p)*((1-mu0)^5*k_T*T1_Mut(5)     + k_T*mu2*(1-mu0)^4*T1_Mut(4)     + k_T*mu2*(1-mu0)^4*T1_Mut(7) + k_T*mu1*(1-mu0)^4*T1_Mut(8)) + alpha*TL_Mut(5) - delta_T2*T2_Mut(5);
dT2_Mut(6)  = (1-p)*((1-mu0)^5*k_T*T1_Mut(6)     + k_T*T1_Mut(4)*mu2*(1-mu0)^4     + k_T*mu1*(1-mu0)^4*T1_Mut(7) + k_T*mu1*(1-mu0)^4*T1_Mut(9)) + alpha*TL_Mut(6) - delta_T2*T2_Mut(6);
dT2_Mut(7)  = (1-p)*((1-mu0)^5*k_T*T1_Mut(7)     + k_T*T1_Mut(5)*mu2*(1-mu0)^4     + k_T*T1_Mut(6)*mu2*(1-mu0)^4 + k_T*mu1*(1-mu0)^4*T1_Mut(10)) + alpha*TL_Mut(7) - delta_T2*T2_Mut(7);
dT2_Mut(8)  = (1-p)*((1-mu0)^5*k_T*T1_Mut(8)     + k_T*T1_Mut(3)*mu2*(1-mu0)^4     + k_T*T1_Mut(5)*mu3*(1-mu0)^4 + k_T*mu2*(1-mu0)^4*T1_Mut(10) + k_T*mu2*(1-mu0)^4*T1_Mut(13)) + alpha*TL_Mut(8) - delta_T2*T2_Mut(8);
dT2_Mut(9)  = (1-p)*((1-mu0)^5*k_T*T1_Mut(9)     + k_T*T1_Mut(3)*mu2*(1-mu0)^4     + k_T*T1_Mut(6)*mu3*(1-mu0)^4 + k_T*mu1*(1-mu0)^4*T1_Mut(10) + k_T*mu2*(1-mu0)^4*T1_Mut(12)) + alpha*TL_Mut(9) - delta_T2*T2_Mut(9);
dT2_Mut(10) = (1-p)*((1-mu0)^5*k_T*T1_Mut(10)   + k_T*T1_Mut(7)*mu3*(1-mu0)^4     + k_T*T1_Mut(8)*mu2*(1-mu0)^4 + k_T*mu2*(1-mu0)^4*T1_Mut(9) + k_T*mu2*(1-mu0)^4*T1_Mut(14)) + alpha*TL_Mut(10) - delta_T2*T2_Mut(10);
dT2_Mut(11) = (1-p)*((1-mu0)^5*k_T*T1_Mut(11)   + k_T*T1_Mut(2)*mu2*(1-mu0)^4     + k_T*T1_Mut(3)*mu1*(1-mu0)^4 + k_T*mu2*(1-mu0)^4*T1_Mut(12) + k_T*mu1*(1-mu0)^4*T1_Mut(13)) + alpha*TL_Mut(11) - delta_T2*T2_Mut(11);
dT2_Mut(12) = (1-p)*((1-mu0)^5*k_T*T1_Mut(12)   + k_T*mu1*(1-mu0)^4*T1_Mut(9)     + k_T*T1_Mut(11)*mu2*(1-mu0)^4 + k_T*mu1*(1-mu0)^4*T1_Mut(14)) + alpha*TL_Mut(12) - delta_T2*T2_Mut(12);
dT2_Mut(13) = (1-p)*((1-mu0)^5*k_T*T1_Mut(13)   + k_T*T1_Mut(8)*mu1*(1-mu0)^4     + k_T*T1_Mut(11)*mu2*(1-mu0)^4 + k_T*mu2*(1-mu0)^4*T1_Mut(14)) + alpha*TL_Mut(13) - delta_T2*T2_Mut(13);
dT2_Mut(14) = (1-p)*((1-mu0)^5*k_T*T1_Mut(14)   + k_T*T1_Mut(10)*mu1*(1-mu0)^4 + k_T*T1_Mut(12)*mu2*(1-mu0)^4    + k_T*mu2*(1-mu0)^4*T1_Mut(13)) + alpha*TL_Mut(14) - delta_T2*T2_Mut(14);

dTL_WT      = p*((1-mu0)^5*k_T*T1_WT      + k_T*mu1*(1-mu0)^4*T1_Mut(1)     + k_T*mu2*(1-mu0)^4*T1_Mut(4))                              - alpha*TL_WT      - delta_TL*TL_WT;
dTL_Mut(1)  = p*((1-mu0)^5*k_T*T1_Mut(1)     + k_T*T1_WT*mu3*(1-mu0)^4         + k_T*mu2*(1-mu0)^4*T1_Mut(2) + k_T*mu2*(1-mu0)^4*T1_Mut(3)) - alpha*TL_Mut(1)  - delta_TL*TL_Mut(1);
dTL_Mut(2)  = p*((1-mu0)^5*k_T*T1_Mut(2)     + k_T*T1_Mut(1)*mu1*(1-mu0)^4     + k_T*mu2*(1-mu0)^4*T1_Mut(11))                             - alpha*TL_Mut(2) - delta_TL*TL_Mut(2);
dTL_Mut(3)  = p*((1-mu0)^5*k_T*T1_Mut(3)     + k_T*T1_Mut(1)*mu2*(1-mu0)^4     + k_T*mu3*(1-mu0)^4*T1_Mut(4) + k_T*mu1*(1-mu0)^4*T1_Mut(8) + k_T*mu2*(1-mu0)^4*T1_Mut(9) + k_T*mu2*(1-mu0)^4*T1_Mut(11)) - alpha*TL_Mut(3) - delta_TL*TL_Mut(3);
dTL_Mut(4)  = p*((1-mu0)^5*k_T*T1_Mut(4)     + k_T*T1_WT*mu2*(1-mu0)^4         + k_T*T1_Mut(3)*mu1*(1-mu0)^4 + k_T*mu1*(1-mu0)^4*T1_Mut(5) + k_T*mu2*(1-mu0)^4*T1_Mut(6)) - alpha*TL_Mut(4) - delta_TL*TL_Mut(4);
dTL_Mut(5)  = p*((1-mu0)^5*k_T*T1_Mut(5)     + k_T*mu2*(1-mu0)^4*T1_Mut(4)     + k_T*mu2*(1-mu0)^4*T1_Mut(7) + k_T*mu1*(1-mu0)^4*T1_Mut(8)) - alpha*TL_Mut(5) - delta_TL*TL_Mut(5);
dTL_Mut(6)  = p*((1-mu0)^5*k_T*T1_Mut(6)     + k_T*T1_Mut(4)*mu2*(1-mu0)^4     + k_T*mu1*(1-mu0)^4*T1_Mut(7) + k_T*mu1*(1-mu0)^4*T1_Mut(9)) - alpha*TL_Mut(6) - delta_TL*TL_Mut(6);
dTL_Mut(7)  = p*((1-mu0)^5*k_T*T1_Mut(7)     + k_T*T1_Mut(5)*mu2*(1-mu0)^4     + k_T*T1_Mut(6)*mu2*(1-mu0)^4 + k_T*mu1*(1-mu0)^4*T1_Mut(10)) - alpha*TL_Mut(7) - delta_TL*TL_Mut(7);
dTL_Mut(8)  = p*((1-mu0)^5*k_T*T1_Mut(8)     + k_T*T1_Mut(3)*mu2*(1-mu0)^4     + k_T*T1_Mut(5)*mu3*(1-mu0)^4 + k_T*mu2*(1-mu0)^4*T1_Mut(10) + k_T*mu2*(1-mu0)^4*T1_Mut(13)) - alpha*TL_Mut(8) - delta_TL*TL_Mut(8);
dTL_Mut(9)  = p*((1-mu0)^5*k_T*T1_Mut(9)     + k_T*T1_Mut(3)*mu2*(1-mu0)^4     + k_T*T1_Mut(6)*mu3*(1-mu0)^4 + k_T*mu1*(1-mu0)^4*T1_Mut(10) + k_T*mu2*(1-mu0)^4*T1_Mut(12)) - alpha*TL_Mut(9) - delta_TL*TL_Mut(9);
dTL_Mut(10) = p*((1-mu0)^5*k_T*T1_Mut(10)   + k_T*T1_Mut(7)*mu3*(1-mu0)^4     + k_T*T1_Mut(8)*mu2*(1-mu0)^4 + k_T*mu2*(1-mu0)^4*T1_Mut(9) + k_T*mu2*(1-mu0)^4*T1_Mut(14)) - alpha*TL_Mut(10) - delta_TL*TL_Mut(10);
dTL_Mut(11) = p*((1-mu0)^5*k_T*T1_Mut(11)   + k_T*T1_Mut(2)*mu2*(1-mu0)^4     + k_T*T1_Mut(3)*mu1*(1-mu0)^4 + k_T*mu2*(1-mu0)^4*T1_Mut(12) + k_T*mu1*(1-mu0)^4*T1_Mut(13)) - alpha*TL_Mut(11) - delta_TL*TL_Mut(11);
dTL_Mut(12) = p*((1-mu0)^5*k_T*T1_Mut(12)   + k_T*mu1*(1-mu0)^4*T1_Mut(9)     + k_T*T1_Mut(11)*mu2*(1-mu0)^4 + k_T*mu1*(1-mu0)^4*T1_Mut(14)) - alpha*TL_Mut(12) - delta_TL*TL_Mut(12);
dTL_Mut(13) = p*((1-mu0)^5*k_T*T1_Mut(13)   + k_T*T1_Mut(8)*mu1*(1-mu0)^4     + k_T*T1_Mut(11)*mu2*(1-mu0)^4 + k_T*mu2*(1-mu0)^4*T1_Mut(14)) - alpha*TL_Mut(13) - delta_TL*TL_Mut(13);
dTL_Mut(14) = p*((1-mu0)^5*k_T*T1_Mut(14)   + k_T*T1_Mut(10)*mu1*(1-mu0)^4 + k_T*T1_Mut(12)*mu2*(1-mu0)^4    + k_T*mu2*(1-mu0)^4*T1_Mut(13)) - alpha*TL_Mut(14) - delta_TL*TL_Mut(14);

dM2_WT      = (1-mu0)^5*k_M*M1_WT      + k_M*mu1*(1-mu0)^4*M1_Mut(1)     + k_M*mu2*(1-mu0)^4*M1_Mut(4)                                   - delta_M2*M2_WT;
dM2_Mut(1)  = (1-mu0)^5*k_M*M1_Mut(1)     + k_M*M1_WT*mu3*(1-mu0)^4         + k_M*mu2*(1-mu0)^4*M1_Mut(2) + k_M*mu2*(1-mu0)^4*M1_Mut(3)   - delta_M2*M2_Mut(1);
dM2_Mut(2)  = (1-mu0)^5*k_M*M1_Mut(2)     + k_M*M1_Mut(1)*mu1*(1-mu0)^4     + k_M*mu2*(1-mu0)^4*M1_Mut(11)                              - delta_M2*M2_Mut(2);
dM2_Mut(3)  = (1-mu0)^5*k_M*M1_Mut(3)     + k_M*M1_Mut(1)*mu2*(1-mu0)^4     + k_M*mu3*(1-mu0)^4*M1_Mut(4) + k_M*mu1*(1-mu0)^4*M1_Mut(8) + k_M*mu2*(1-mu0)^4*M1_Mut(9) + k_M*mu2*(1-mu0)^4*M1_Mut(11) - delta_M2*M2_Mut(3);
dM2_Mut(4)  = (1-mu0)^5*k_M*M1_Mut(4)     + k_M*M1_WT*mu2*(1-mu0)^4         + k_M*M1_Mut(3)*mu1*(1-mu0)^4 + k_M*mu1*(1-mu0)^4*M1_Mut(5) + k_M*mu2*(1-mu0)^4*M1_Mut(6)  - delta_M2*M2_Mut(4);
dM2_Mut(5)  = (1-mu0)^5*k_M*M1_Mut(5)     + k_M*mu2*(1-mu0)^4*M1_Mut(4)     + k_M*mu2*(1-mu0)^4*M1_Mut(7) + k_M*mu1*(1-mu0)^4*M1_Mut(8)  - delta_M2*M2_Mut(5);
dM2_Mut(6)  = (1-mu0)^5*k_M*M1_Mut(6)     + k_M*M1_Mut(4)*mu2*(1-mu0)^4     + k_M*mu1*(1-mu0)^4*M1_Mut(7) + k_M*mu1*(1-mu0)^4*M1_Mut(9)  - delta_M2*M2_Mut(6);
dM2_Mut(7)  = (1-mu0)^5*k_M*M1_Mut(7)     + k_M*M1_Mut(5)*mu2*(1-mu0)^4     + k_M*M1_Mut(6)*mu2*(1-mu0)^4 + k_M*mu1*(1-mu0)^4*M1_Mut(10)  - delta_M2*M2_Mut(7);
dM2_Mut(8)  = (1-mu0)^5*k_M*M1_Mut(8)     + k_M*M1_Mut(3)*mu2*(1-mu0)^4     + k_M*M1_Mut(5)*mu3*(1-mu0)^4 + k_M*mu2*(1-mu0)^4*M1_Mut(10) + k_M*mu2*(1-mu0)^4*M1_Mut(13)  - delta_M2*M2_Mut(8);
dM2_Mut(9)  = (1-mu0)^5*k_M*M1_Mut(9)     + k_M*M1_Mut(3)*mu2*(1-mu0)^4     + k_M*M1_Mut(6)*mu3*(1-mu0)^4 + k_M*mu1*(1-mu0)^4*M1_Mut(10) + k_M*mu2*(1-mu0)^4*M1_Mut(12)  - delta_M2*M2_Mut(9);
dM2_Mut(10) = (1-mu0)^5*k_M*M1_Mut(10)   + k_M*M1_Mut(7)*mu3*(1-mu0)^4     + k_M*M1_Mut(8)*mu2*(1-mu0)^4 + k_M*mu2*(1-mu0)^4*M1_Mut(9) + k_M*mu2*(1-mu0)^4*M1_Mut(14)  - delta_M2*M2_Mut(10);
dM2_Mut(11) = (1-mu0)^5*k_M*M1_Mut(11)   + k_M*M1_Mut(2)*mu2*(1-mu0)^4     + k_M*M1_Mut(3)*mu1*(1-mu0)^4 + k_M*mu2*(1-mu0)^4*M1_Mut(12) + k_M*mu1*(1-mu0)^4*M1_Mut(13)  - delta_M2*M2_Mut(11);
dM2_Mut(12) = (1-mu0)^5*k_M*M1_Mut(12)   + k_M*mu1*(1-mu0)^4*M1_Mut(9)     + k_M*M1_Mut(11)*mu2*(1-mu0)^4 + k_M*mu1*(1-mu0)^4*M1_Mut(14)  - delta_M2*M2_Mut(12);
dM2_Mut(13) = (1-mu0)^5*k_M*M1_Mut(13)   + k_M*M1_Mut(8)*mu1*(1-mu0)^4     + k_M*M1_Mut(11)*mu2*(1-mu0)^4 + k_M*mu2*(1-mu0)^4*M1_Mut(14)  - delta_M2*M2_Mut(13);
dM2_Mut(14) = (1-mu0)^5*k_M*M1_Mut(14)   + k_M*M1_Mut(10)*mu1*(1-mu0)^4 + k_M*M1_Mut(12)*mu2*(1-mu0)^4    + k_M*mu2*(1-mu0)^4*M1_Mut(13)  - delta_M2*M2_Mut(14);


dX(1) = dTU; dX(2) = dMU; dX(3) = dT1_WT; dX(4) = dT2_WT; dX(5) = dTL_WT; dX(6) = dM1_WT; dX(7) = dM2_WT; dX(8) = dVI_WT; dX(9) = dVNI_WT;

j = 1;
for i = 1:length(Res)-1
    dX(j+9)  = dT1_Mut(i);
    dX(j+10) = dT2_Mut(i);
    dX(j+11) = dTL_Mut(i);
    dX(j+12) = dM1_Mut(i);
    dX(j+13) = dM2_Mut(i);
    dX(j+14) = dVI_Mut(i);
    j = j+6;
end

j = 94;
for i = 1:length(Res)-1
    dX(j) = dVNI_Mut(i);
    j = j+1;
end

end