%%% 
%%% This function scales anatomical and physiological data of a reference
%%% individual to some target values.
%%% 
%%% Version: February 10th, 2014. 
%%% For references and citation, please see MAIN script.
%%% 
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the 
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported 
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function individual = GenericPBPKmodel_scaleIndividual(individual,method)


switch method.BSD
    
    case 'LBW';
        
        individual = GenericPBPKmodel_LBW_based_scaling(individual,method);
        
    case 'BW';
        
        GenericPBPKmodel_ReportErrorMessage('Not yet implemented!');
        
    otherwise
        GenericPBPKmodel_ReportErrorMessage('Unknown method to scale an individual!');
end;

individual = GenericPBPKmodel_update_dosing(individual);

end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_LBW_based_scaling(reference,method)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

T = reference.T; 

%%% adopt refence values and adapt subsequently those parameters that
%%% are assumed to depend on the covariates BW and BH
individual = reference;

BH        = method.BH;
BW        = method.BW;
BMI       = BW/BH^2;

%%% body weight, heigt and body mass index
individual.BH        = BH;
individual.BW        = BW;
individual.BMI       = BW/BH^2;

% LBW and adipose tissue
LBW  = FFM(BW,BH,individual.sex);  % Note: LBW approximated by FFM!

individual.OW(T.adi) = BW-LBW;

% scaling of skin
individual.BSA       = BSA(BW,BH);
SV.ski               = individual.BSA / reference.BSA;
individual.OW(T.ski) = SV.ski * reference.OW(T.ski);

% scaling of brain
SV.bra               = 1;
individual.OW(T.bra) = SV.bra * reference.OW(T.bra);

% scaling of all remaining tissues
m  = setdiff(T.tissueDB,[T.adi T.ski T.bra]);
individual_OWres     = individual.BW - sum(individual.OW([T.adi T.ski T.bra]));
reference_OWres      = reference.BW  - sum(reference.OW([T.adi T.ski T.bra]));
SV.tis               = individual_OWres/reference_OWres;
individual.OW(m)     = SV.tis * reference.OW(m);

m = T.tissueDB;
individual.V.tis(m)     = individual.OW(m)./reference.dens(m);

% scaling of blood flows
SQ                   = SV.tis;
individual.co        = SQ * reference.co;
individual.Q.blo     = SQ * reference.Q.blo;

% scaling the clearances. 
if strcmp(individual.drug.class,'sMD')
    ref_CLint = reference.drug.CLint.tis; ref_V_tis = reference.V.tis;
    individual.drug.CLint.tis = ref_CLint./ref_V_tis.*individual.V.tis;
end;

% dosing
study = reference.study;
for d = {'po','bolus','infusion'}
    
    route = char(d);
    if ~strcmp(study.dosing.(route).per,'fixed')
        BSD  = study.dosing.(route).per;
        study.(route).dose = study.dosing.(route).dose*individual.(BSD);
    end;
    
end;
individual.study = study;

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function out = FFM(BW,BH,sex)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

% determine fat free mass (FFM) according to
% Green B, Duffull SB. Clin Pharmacol Ther Vol 72, 2002

BMI = BW/BH^2;
switch sex
    case 'male'
        out = 9.27e3*BW/(6.68e3+216*BMI);
    case 'female'
        out = 9.27e3*BW/(8.78e3+244*BMI);
    otherwise
        fprintf('\n\n Unknow sex to compute LBW! \n\n');
        error('in determining LBW');
end;

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function out = BSA(BW,BH)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

% Determine body surface area (BSA) in [m^2]
% Source: Mosteller, New Engl J Med, Vol 317, 1987

out = sqrt(BH*BW/36);

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_update_dosing(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% update dosing information, since the individual dose might have changed 
%%% due to scaling

study = individual.study;
for d = 1:length(study.route)
    
    route  = study.route{d};
    dosing = study.dosing.(route);
    if  strcmp(dosing.per,'fixed')
        study.(route).dose = dosing.dose;
    else
        BSD  = dosing.per;
        if ~isfield(individual,BSD)
            message = sprintf(' Unknown choice of dose per ''%s'' for %s dosing',BSD,route);
            GenericPBPKmodel_ReportErrorMessage(message);
        end;
        study.(route).dose = dosing.dose*individual.(BSD);
    end;
    if strcmp(route,'infusion')
        study.infusion.tend = dosing.tend;
    end;
    
end;
individual.study = study;

end

%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
