%%% 
%%% This function allows to specify random perturbations of the parameters
%%% of some individual
%%% 
%%% Version: February 10th, 2014. 
%%% For references and citation, please see MAIN script.
%%% 
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the 
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported 
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function individual = GenericPBPKmodel_randomEffects(individual,randomEffects)


%%% Check, whether there are random effects to be considered
%%%
check = GenericPBPKmodel_checkForRandomEffectsAndReport(randomEffects);

if strcmp(check,'no random effects considered')
    return;
end;


%%% Go individual by individual ...
%%%
for id=1:length(individual)

    switch individual(id).drug.class
        
        case 'sMD';
            facX = randomEffects.sMD.facX;
            
            individual(id) = randomEffect_on_CLint_hep(individual(id),facX.CLint_hep);
            individual(id) = randomEffect_on_K_tis_up(individual(id),facX.K_tis_up);
            
        case 'mAb';
            
            message = 'Random effect(s) for mAbs not yet implemented!';
            GenericPBPKmodel_ReportErrorMessage(message);

            
    end;
    
end;


end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function check = GenericPBPKmodel_checkForRandomEffectsAndReport(randomEffects)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% Check, whether there are random effects to be considered
%%%
facX.CLint_hep = randomEffects.sMD.facX.CLint_hep;
facX.K_tis_up  = randomEffects.sMD.facX.K_tis_up;

if (facX.CLint_hep>0) || (facX.K_tis_up>0)
    fprintf('\n'); fprintf(2,'       >>> Random effect(s) on');
    if facX.CLint_hep>0
        fprintf(2,' CLint_hep,');
    end;
    if facX.K_tis_up>0
        fprintf(2,' K_tis,');
    end;
    fprintf(2,'\b considered! <<<');
    check = 'random effects considered';
elseif (facX.CLint_hep<0) || (facX.K_tis_up<0)
    message = 'randomEffects.facX negative for some parameter!';
    GenericPBPKmodel_ReportErrorMessage(message);
else
    check = 'no random effects considered';
end;

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = randomEffect_on_CLint_hep(individual,facX)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

if facX==0
    return; % no random effect on CLint_hep to be considered
end;

T = individual.T;
CLint_ref = individual.drug.CLint.tis(T.liv);
     
rand1 = rand;
CLint_liv = [rand1<0.5]*CLint_ref*(1+facX*rand) + [rand1>=0.5]*CLint_ref/(1+facX*rand); 

individual.drug.CLint.tis(T.liv) = CLint_liv;
end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = randomEffect_on_K_tis_up(individual,facX)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

if facX==0
    return; % no random effect on K_tis_up to be considered
end;

T = individual.T;
K_ref = individual.drug.K.tis_up;
CLintK_hep = individual.drug.CLint.tis(T.liv)*K_ref(T.liv);
      
%%% How to perturb K.tis_up for the different tissues:
%%% faxX_allTisIdentical = 1
%%% --> Use one random number to perturb all K.tis
%%%  or
%%% faxX_allTisIdentical = 0
%%% --> Use one random number per K.tis for perturbation
%%%
facX_allTisIdentical = 0;

if facX_allTisIdentical
    rand1 = rand(1)*ones(size(K_ref));
    rand2 = rand(1)*ones(size(K_ref));
else
    rand1 = rand(size(K_ref));
    rand2 = rand(size(K_ref));
    
end;
    
K_tis_up = [rand1<0.5].*K_ref.*(1+facX*rand2) + [rand1>=0.5].*K_ref./(1+facX*rand2);

individual.drug.K.tis_up = K_tis_up;
individual.drug.CLint.tis(T.liv) = CLintK_hep/K_tis_up(T.liv);
end


%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

