%%% 
%%% This function specifies a classical n-compartment model with linear
%%% clearance, stated in terms of macro constants
%%% 
%%% Version: February 10th, 2014. 
%%% For references and citation, please see MAIN script.
%%% 
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the 
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported 
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function individual = GenericPBPKmodel_CMTmodel_xCMT_PLASMA_macroConstants_linearCL(individual)


%%% The model assumes n=length(model.V) compartments and n-1=length(Q)
%%% intercompartmental flows, in addition to central and peripheral 
%%% clearances. The first compartment is by default the central compartment
%%% It is assumed that C_cen = C_pla
%%%

if (length(individual.model.V)-1) ~= length(individual.model.Q)
    message = ('Condition: number of flows = number of volumes - 1 violated!');
    GenericPBPKmodel_ReportErrorMessage(message);
end;

%%% define model specific parameters
%%%
individual = GenericPBPKmodel_defineModelParameters(individual);

%%% simulate classical CMT model 
%%%
model = individual.model; S = model.S; study = individual.study;
X0 = model.X0; sim.t = []; sim.X = []; 

for d=1:study.numberOfDosing
    
    %%% account for dosing for each new dosing interval
    X0(S.bolus)   = X0(S.bolus) + study.bolus.dose*model.SF.mg_to_nmol/model.V.iv_bolus;
    X0(S.IVbag)   = study.infusion.dose*model.SF.mg_to_nmol; 
    X0(S.GItract) = X0(S.GItract) + study.po.dose*model.SF.mg_to_nmol;
    %%% solve system of ODEs 'GenericPBPKmodel_RHS' with initial conditions 'X0'
    [t,X] = ode15s(@GenericPBPKmodel_RHS,study.observationTime,X0',[],individual);
    
    %%% transform relative to absolute time and modify last time point
    %%% to allow for trough value measurement
    Tend  = max(study.observationTime);
    t = t + (d-1)*Tend; 
    if d~=study.numberOfDosing
        t(end)  = t(end) - 1e-10; 
    end;
    
    %%% store current output in sim structure
    sim.t = [sim.t; t];
    sim.X = [sim.X; X];

    X0 = X(end,:)'; 

end;

%%% determine standart simulation output
%%%
individual = GenericPBPKmodel_determineStandartOutput(individual,sim);

%%% make specific plots
%%%
GenericPBPKmodel_specificGraphicalOutput(individual);

end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_defineModelParameters(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model = individual.model;

%%% -----------------------------------------------------------------------
%%% Define indexing
%%%

drug = individual.drug;

%%% tissue indexing
%%%
S = individual.T;

S.maxIndex.cmt = length(model.V);
S.allCmt       = [1:S.maxIndex.cmt];
S.cen = 1;
if S.maxIndex.cmt>1
    S.per = [2:S.maxIndex.cmt];
end;
S.bolus = S.cen;

S.initialize.cmt.NaN   = NaN*ones(1,max(S.allCmt));

%%% dosing and metabolism indexing

S.GItract = 1 + S.maxIndex.cmt; % gastro-intestinal tract (for po dosing)
S.IVbag   = 2 + S.maxIndex.cmt; % IVbag (for infusion dosing)
S.metab   = 3 + S.maxIndex.cmt; % metabolized drug

S.maxIndex.dosing = 3 + S.maxIndex.cmt;

%%% additional indexing
%%%

S.maxIndex.all = 0 + S.maxIndex.dosing;

%%% ODE indexing (for relative indexing)
%%%
S.C_cmt = 1:S.maxIndex.cmt; % all tissues


%%% -----------------------------------------------------------------------
%%% Define CMT model parameters 
%%%

%%% volume for i.v. bolus dose
V.cmt      = model.V; % pre-specified volumes are assumed to be the cmt volumes
V.iv_bolus = V.cmt(S.cen);

%%% determine intercompartmental flow for central compartment as sum over
%%% all peripheral intercompartmental flows
Q.cmt = [sum(model.Q) model.Q];

%%% first order oral absorption rate constant in [1/min]

if individual.study.po.dose>0 && ...
   (isnan(individual(2).model.F.bio) || isnan(individual(2).model.lambda_po))
    GenericPBPKmodel_ReportErrorMessage('F.bio or lambda_po = NaN');
else
    model.F.bio = 0; model.lambda_po = 0;

end

%%% initial condition of ODE
%%%
X0 = zeros(1,S.maxIndex.all);


%%% -----------------------------------------------------------------------
%%% Assign model parameters 
%%%

model.S  = S;
model.V  = V;
model.Q  = Q;
model.SF = drug.SF;
model.X0 = X0;

individual.model = model;
end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function dX = GenericPBPKmodel_RHS(t,X,individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% initialize output vector
dX = zeros(size(X));

%%% model and indexing
model = individual.model; S = model.S;

%%% variables (always use row vector notation)
C_cmt     = X(S.C_cmt)';
A_GItract = X(S.GItract)';

%%% tissue volumes, blood flows, extraction ratios, clearance etc.
V     = model.V.cmt;
Q     = model.Q.cmt; 
CL    = model.CL;
F     = model.F;
infusion = individual.study.infusion;

infusion_rate = 0;
if (infusion.tend > 0) && (t<=infusion.tend)
    infusion_rate = model.SF.mg_to_nmol * infusion.dose/infusion.tend;
end;
lambda_po = model.lambda_po; 

%%% -----------------------------------------------------------------------
%%% START OF ODEs
%%%
VdC_cmt = zeros(size(C_cmt)); 

if length(S.allCmt) > 1
    
    Cin_cen    = sum(Q(S.per).*C_cmt(S.per)) / Q(S.cen);    
    VdC_cmt(S.per) = Q(S.per).*( C_cmt(S.cen) - C_cmt(S.per) ) - CL(S.per)*C_cmt(S.per);
    VdC_cmt(S.cen) = Q(S.cen) *( Cin_cen  - C_cmt(S.cen) ) ...
                    -CL(S.cen)*C_cmt(S.cen) + infusion_rate + ...
                    lambda_po*A_GItract; 
else
    VdC_cmt(S.cen) = -CL(S.cen)*C_cmt(S.cen) + infusion_rate ...
                     +lambda_po*A_GItract; 
end;

%%% drug amount in GItract for absorption
dA_GItract = -lambda_po*A_GItract;

%%% drug amount in IVbag for infusion
dA_IVbag = -infusion_rate;

%%% drug amount metabolized or excreted
dA_metab   = +CL(S.cen)*C_cmt(S.cen) + sum(CL(S.per)*C_cmt(S.per)) + (1-F.bio)*lambda_po*A_GItract;

%%%
%%% END OF ODEs 
%%% -----------------------------------------------------------------------

%%% converting amounts to concentrations
dC_cmt = zeros(size(C_cmt)); 
dC_cmt(S.allCmt) = VdC_cmt(S.allCmt)./V(S.allCmt);


%%% output vector (always in column vector notation)
dX(S.C_cmt)   = dC_cmt';
dX(S.GItract) = dA_GItract';
dX(S.IVbag)   = dA_IVbag';
dX(S.metab)   = dA_metab';

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_determineStandartOutput(individual,sim)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model = individual.model; info = individual.info; S = model.S; 
pred  = individual.pred; 

%%% assign predicted profiles
%%%
info.pred.t         = 'simulation time in [min]';
pred.t              = sim.t;

info.pred.C.cmt     = 'concentration-time profile in different compartments [nmol/L]';
pred.C.cmt          = sim.X(:,S.C_cmt);

info.pred.A.tot     = 'amount in different compartments [nmol]';
pred.A.tot          = pred.C.cmt(:,S.allCmt)*diag(model.V.cmt(S.allCmt));

info.pred.A.body    = 'total amount in the body in [nmol]';
pred.A.body         = sum(pred.A.tot,2);

info.pred.A.GItract = 'remaining amount in the GI tract in [nmol]';
pred.A.GItract      = sim.X(:,S.GItract);

info.pred.A.IVbag   = 'remaining amount in the IVbag in [nmol]';
pred.A.IVbag        = sim.X(:,S.IVbag);

info.pred.A.metab   = 'amount metabolized in [nmol]';
pred.A.metab        = sim.X(:,S.metab);


%%% determine standard output in [mg/L] or [mg]
%%%
stdout = individual.stdout; T = stdout.T;
initialize.NaN = ones(size(pred.t))*T.initialize.tissueDB.NaN; SF = model.SF; 

info.stdout.t          = 'simulation time in [min]';
stdout.t               = pred.t; 

info.stdout.C.tis      = 'concentration-time profile in tissue space [mg/L]';
stdout.C.tis           = initialize.NaN;
stdout.C.tis(:,T.pla)  = SF.nmol_to_mg*pred.C.cmt(:,S.cen);

info.stdout.nC.tis     = 'normalized concentration-time profile (nC.tis = C.tis/eK.tis) in tissue space [mg/L] ';
stdout.nC.tis          = initialize.NaN;
stdout.nC.tis(:,T.pla) = stdout.C.tis(:,T.pla); % model assumes that central = plasma concentration

info.stdout.A.body     = 'total amount in the body in [mg]';
stdout.A.body          = SF.nmol_to_mg*pred.A.body;

info.stdout.A.GItract  = 'remaining amount in the GI tract in [mg]';
stdout.A.GItract       = SF.nmol_to_mg*pred.A.GItract;

info.stdout.A.IVbag    = 'remaining amount in the IVbag in [mg]';
stdout.A.IVbag         = SF.nmol_to_mg*pred.A.IVbag;

info.stdout.A.metab    = 'amount metabolized in [mg]';
stdout.A.metab         = SF.nmol_to_mg*pred.A.metab;


individual.info   = info;
individual.pred   = pred;
individual.stdout = stdout;

end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_specificGraphicalOutput(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% define here specific output for this model

end

%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

