%%% 
%%% This function specifies a simplfied PBPK model for mAb with
%%% extravasation-limited tissue distribution and some target
%%% 
%%% Version: February 10th, 2014. 
%%% For references and citation, please see MAIN script.
%%% 
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the 
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported 
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function individual = GenericPBPKmodel_mAb_PBPK_11CMT_extravasationLimited_Target(individual)

%%% check whether drug class is really mAb
%%%
if ~strcmp(individual.drug.class,'mAb')
    message = sprintf('Model >>mAb_PBPK_11CMT_extravasationLimited<< not defined for drug type %s!',individual.drug.class);
    GenericPBPKmodel_ReportErrorMessage(message);
end;

%%% define model specific parameters
%%%
individual = GenericPBPKmodel_defineModelParameters(individual);

%%% simulate PBPK model 
%%%
model = individual.model; S = model.S; study = individual.study;
X0 = model.X0; sim.t = []; sim.X = []; 

for d=1:study.numberOfDosing
    
    %%% account for dosing for each new dosing interval
    X0(S.bolus)   = X0(S.bolus) + study.bolus.dose*model.SF.mg_to_nmol/model.V.iv_bolus;
    X0(S.IVbag)   = study.infusion.dose*model.SF.mg_to_nmol; 

    %%% solve system of ODEs 'GenericPBPKmodel_RHS' with initial conditions 'X0'
    [t,X] = ode15s(@GenericPBPKmodel_RHS,study.observationTime,X0',[],individual);
    
    %%% transform relative to absolute time and modify last time point
    %%% to allow for trough value measurement
    Tend  = max(study.observationTime);
    t = t + (d-1)*Tend; 
    if d~=study.numberOfDosing
        t(end)  = t(end) - 1e-10; 
    end;
    
    %%% store current output in sim structure
    sim.t = [sim.t; t];
    sim.X = [sim.X; X];

    X0 = X(end,:)'; 

end;

%%% determine standart simulation output
%%%
individual = GenericPBPKmodel_determineStandartOutput(individual,sim);

%%% graphical output specific to this model
%%%
GenericPBPKmodel_specificGraphicalOutput(individual);

end 
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_defineModelParameters(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model   = individual.model;

%%% -----------------------------------------------------------------------
%%% Define indexing
%%%

%%% tissue indexing
%%%
S = individual.T;

S.allTis      = [S.lun S.adi S.hea S.kid S.mus S.bon S.ski S.gut S.spl S.liv S.pla];
S.allTisExPla = [S.lun S.adi S.hea S.kid S.mus S.bon S.ski S.gut S.spl S.liv      ];
S.bolus       = S.pla;    % i.v. bolus administration 
S.tissueDB    = S.allTis; % specify which tissues are part of the general tissue DB

S.visTis      = [S.gut S.spl S.liv];
S.nonvisTis   = [S.lun S.adi S.hea S.kid S.mus S.bon S.ski];

S.maxIndex.tis = max(S.allTis); 
S.initialize.tis.NaN   = NaN*ones(1,S.maxIndex.tis);

%%% dosing and metabolism indexing

S.GItract = 1 + S.maxIndex.tis; % gastro-intestinal tract (for po dosing)
S.IVbag   = 2 + S.maxIndex.tis; % IVbag (for infusion dosing)
S.metab   = 3 + S.maxIndex.tis; % metabolized drug

S.maxIndex.dosing = 3 + S.maxIndex.tis;

%%% additional indexing
%%%

S.maxIndex.all = 0 + S.maxIndex.dosing;

%%% ODE indexing (for relative indexing)
%%%
S.C_tis   =   1:S.maxIndex.tis; % all tissues


%%% -----------------------------------------------------------------------
%%% Define PBPK parameters 
%%%

a   = S.allTis;
mAb = individual.drug;

%%% tissue volumes 
%%%
V.tis      = S.initialize.tis.NaN;
V.tis(a)   = individual.V.tis(a);

%%% volume for i.v. bolus dosing
%%%
V.iv_bolus = V.tis(S.pla);  

%%% blood flows (ensure closed circulatory system!)
%%%
Q.blo    = S.initialize.tis.NaN;
Q.blo(S.allTisExPla) = individual.Q.blo(S.allTisExPla);

Q.pla    = (1-individual.hct)*Q.blo; % peripheral plasma flow

%%% lymph flow as fraction of peripheral plasma and blood flow
%%%
fLymph = S.initialize.tis.NaN;
fLymph(S.visTis)    = 0.02;
fLymph(S.nonvisTis) = 0.04;

Q.lymph = fLymph.*Q.pla;
Q.lymph(S.pla) = sum(Q.lymph(S.allTisExPla));

%%% define vascular reflection coefficints
%%%
sigma.vas       = S.initialize.tis.NaN;
sigma.vas([S.liv S.spl S.gut])       = 0.90;
sigma.vas([S.hea S.kid S.lun])       = 0.95;
sigma.vas([S.mus S.ski S.adi S.bon]) = 0.98;
sigma.vas(S.pla) = 1 - sum( Q.lymph(S.allTisExPla).*(1-sigma.vas(S.allTisExPla)) ) /Q.lymph(S.pla); 

%%% define elimination-corrected tissue partition coefficients
eK.tis = S.initialize.tis.NaN;

useABCs = 0;
if isfield(model,'useABCs') && strcmp(model.useABCs,'yes')
    useABCs = 1;
    fprintf(2,'   >> ABCs from Shah and Betts 2013 used <<'); fprintf('\n');
end;

if useABCs ==1
    method.name = 'ShahBetts'; drug = individual.drug;
    ABC_ShahBetts = GenericPBPKmodel_TissuePartitionCoefficients(individual,drug,method);
    eK.tis(S.allTis) = ABC_ShahBetts(S.allTis)./(1-sigma.vas(S.allTis));
else
    eK.tis(S.hea) = 2.322;
    eK.tis(S.kid) = 2.576;
    eK.tis(S.liv) = 1.324;
    eK.tis(S.lun) = 2.152;
    eK.tis(S.gut) = 0.623;
    eK.tis(S.spl) = 0.303;
    eK.tis(S.ski) = 6.270;
    eK.tis([S.mus S.adi S.bon])  = 1.695;
end;
eK.tis(S.pla) = 1/(1-sigma.vas(S.pla));

%%% tissue-specific extraction ratio
E.tis = S.initialize.tis.NaN;
E.tis(S.allTis) = 0;

%%% blood-to-plasma ratio assuming that mAb do not distribute or bind to
%%% erythrocytes
%%%
BP = (1-individual.hct);

%%% tissue plasma clearance
CLpla.tis = S.initialize.tis.NaN;
CLpla.tis(S.allTis) = 0;

scenario = 7;
if isfield(model,'scenario')
    scenario = model.scenario;
    fprintf(2,'   >> scenario No.%d used <<',scenario); fprintf('\n');
end;

switch scenario
    case 2;
        E.tis([S.adi S.liv S.mus S.spl]) = 0.029;
    case 4;
        E.tis(S.ski) = 0.095;
    case 5;
        E.tis(S.mus) = 0.14;
    case 6;
        E.tis(S.liv) = 0.042;
    case 7;
        CLpla.tis(S.pla) = 9.0278e-8; % 9.0278e-08 L/min = 9.e-8 * 1e3 * 60*24 mL/day = 0.13 mL/day (as in the article)
    case 8;
        E.tis(S.spl) = 0.5;
        CLpla.tis(S.pla) = 9.0278e-8 - E.tis(S.spl)*Q.lymph(S.spl)*(1-sigma.vas(S.spl));
    otherwise
        GenericPBPKmodel_ReportErrorMessage('Unknown scenario!');
end; 
CLpla.tis(S.allTisExPla) = E.tis(S.allTisExPla).*Q.lymph(S.allTisExPla).*(1-sigma.vas(S.allTisExPla));


%%% elimination corrected partition coefficients
%%%
K.tis = eK.tis./(1-E.tis); 

%%% define intrinsic tissue clearance
%%%
CLint.tis = S.initialize.tis.NaN;
CLint.tis = Q.lymph./K.tis.*E.tis./(1-E.tis);

%%% account for FcRn status 
if strcmp(model.FcRn_status,'knock-out')
    
    %%% define knock-out extraction ratio and clearance
    %%% Note: elimination corrected partition coefficients
    %%% (and thus also the ABCs) are assumed to be the
    %%% same as for the wild-type (supported by Shah & Betts, 2013)
    facE = 70;
    E.tis = facE*E.tis./(1-E.tis + facE*E.tis);  
    K.tis = eK.tis./(1-E.tis);
    CLint.tis = Q.lymph./K.tis.*E.tis./(1-E.tis);

    if scenario~=1
        GenericPBPKmodel_ReportErrorMessage('Knock-out case has been implemented only for scenario 1!')
    end;
    
    CLpla.tis = 23*CLpla.tis;
end;

%%% double check the p.o. dose = 0
if individual.study.po.dose>0
    GenericPBPKmodel_ReportErrorMessage('po administration of mAb not supported!');
end;


%%% Target related parameters
%%%
if length(model.V_max)~=length(S.allTis)
    message = 'Number of specified V_max value differs from number of tissues';
    GenericPBPKmodel_ReportErrorMessage(message);
end;
Vmax = S.initialize.tis.NaN;
Vmax(S.allTis) = model.V_max;

if length(model.B_max)~=length(S.allTis)
    message = 'Number of specified B_max value differs from number of tissues';
    GenericPBPKmodel_ReportErrorMessage(message);
end;
Bmax = S.initialize.tis.NaN;
Bmax(S.allTis) = model.B_max;

kdeg = S.initialize.tis.NaN;
kdeg(S.allTis) = 0;
tis = S.allTis([Bmax(S.allTis)~=0]); % tissues with non-zero Bmax
kdeg(tis) = Vmax(tis)./Bmax(tis);

%%% correct Km value with Kint defined as
%%% Kint = C_int/C_tis = (A_int/V_int) / (A_tis/V_tis) = V_tis/V_int = 1/fVtis.int
%%% assuming A_int = A_tis
K.int2tis = S.initialize.tis.NaN;
K.int2tis(S.allTis) = 1./individual.fVtis.int(S.allTis); 

Km = S.initialize.tis.NaN;
Km(S.allTis) = 1;
Km(S.allTis) = model.K_m;

Km(S.allTisExPla) = Km(S.allTisExPla)./K.int2tis(S.allTisExPla);


%%% compute ABCs and effective flows
ABC.tis = (1-sigma.vas).*eK.tis;
Leff.lymph = (1-sigma.vas).*Q.lymph;

%%% double check the p.o. dose = 0
if individual.study.po.dose>0
    GenericPBPKmodel_ReportErrorMessage('po administration of mAb not supported!');
end;

%%% initial condition of ODEs
%%%
X0 = zeros(1,S.maxIndex.all);


%%% -----------------------------------------------------------------------
%%% Assign model parameters 
%%%

model.S      = S;
model.V      = V;
model.Q      = Q;
model.K      = K;
model.E      = E;
model.eK     = eK;
model.BP     = BP;
model.fLymph = fLymph;
model.sigma  = sigma;
model.CLint  = CLint;
model.CLpla  = CLpla;
model.Bmax   = Bmax;
model.Vmax   = Vmax;
model.kdeg   = kdeg;
model.Km     = Km;
model.ABC    = ABC;
model.L      = Leff;
model.SF     = mAb.SF;
model.X0     = X0;

individual.model = model;

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function dX = GenericPBPKmodel_RHS(t,X,individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% initialize output vector
dX = zeros(size(X));

%%% model and indexing
model = individual.model; S = model.S;

%%% variables (always use row vector notation)
C_tis    = X(S.C_tis)';

%%% tissue volumes, blood flows, endosomal clearance etc.
V        = model.V.tis;
Q        = model.Q.lymph; 
K        = model.K.tis; 
sigma    = model.sigma.vas;
CLint    = model.CLint.tis; 
CLpla    = model.CLpla.tis; 
Bmax     = model.Bmax;
kdeg     = model.kdeg;
Km       = model.Km;
infusion = individual.study.infusion;

infusion_rate = 0;
if (infusion.tend > 0) && (t<=infusion.tend)
    infusion_rate = model.SF.mg_to_nmol * infusion.dose/infusion.tend;
end;

%%% -----------------------------------------------------------------------
%%% START OF ODEs
%%%
VdC_tis = zeros(size(C_tis));

%%% compute concentration available for target binding
all = S.allTis;
C_eff(all) = C_tis(all)-Bmax(all)-Km(all);
C_ex(all)  = 1/2*( C_eff(all) + sqrt( C_eff(all).^2 + 4*Km(all).*C_tis(all) ) );

%%% all tissues except blood 
tis = S.allTisExPla;
VdC_tis(tis)   = Q(tis).*( (1-sigma(tis))*C_ex(S.pla) - C_ex(tis)./K(tis) ) ...
                 - CLint(tis).*C_ex(tis)...
                 - kdeg(tis).*Bmax(tis).*C_ex(tis)./(Km(tis)+C_ex(tis));

%%% blood
VdC_tis(S.pla) = sum( Q(tis).*C_ex(tis)./K(tis) ) ...
                - sum( Q(tis).*(1-sigma(tis))*C_ex(S.pla) ) - CLpla(S.pla)*C_ex(S.pla) ...
                - kdeg(S.pla).*Bmax(S.pla).*C_ex(S.pla)./(Km(S.pla)+C_ex(S.pla)) ...
                + infusion_rate; 

%%% drug amount in IVbag for infusion
dA_IVbag = - infusion_rate;

%%% drug amount metabolized or excreted
dA_metab   = +CLpla(S.pla)*C_ex(S.pla) + sum(CLint(tis).*C_ex(tis)) ...
             +sum( kdeg(all).*Bmax(all).*C_ex(all)./(Km(all)+C_ex(all)) );

%%%
%%% END OF ODEs 
%%% -----------------------------------------------------------------------


%%% converting amounts to concentrations
dC_tis = zeros(size(C_tis)); 
dC_tis(S.allTis) = VdC_tis(S.allTis)./V(S.allTis);


%%% output vector (always in column vector notation)
dX(S.C_tis) = dC_tis(S.C_tis)';
dX(S.IVbag) = dA_IVbag';
dX(S.metab) = dA_metab';

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_determineStandartOutput(individual,sim)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model = individual.model; info = individual.info; S = model.S; 
pred  = individual.pred; 

%%% assign predicted profiles
%%%
info.pred.t         = 'simulation time in [min]';
pred.t              = sim.t;

%%% initialize all compartments with NaN and assign only to those
%%% compartments values that are part of the model topology
initialize.NaN = ones(size(pred.t))*S.initialize.tis.NaN; tis = S.allTis;

info.pred.C.tis     = 'concentration-time profile in tissue space [nmol/L]';
pred.C.tis          = initialize.NaN;
pred.C.tis(:,tis)   = sim.X(:,tis);

%%% determine interstitial concentration

info.pred.C.int     = 'concentration-time profile in interstitital space [nmol/L]';
pred.C.int          = initialize.NaN;
pred.C.int(:,tis)   = pred.C.tis(:,tis)*diag(model.K.int2tis(tis));

%%% determine extra-cellular concentration in interstitial space

C_eff               = initialize.NaN;
C_eff(:,tis)        = pred.C.tis(:,tis) - ones(size(pred.t))*( model.Bmax(tis) + model.Km(tis) );
C_ex                = initialize.NaN;

C_ex(:,tis)         = 1/2*( C_eff(:,tis) + sqrt( C_eff(:,tis).^2 + 4*pred.C.tis(:,tis)*diag(model.Km(tis)) ) );

info.pred.sat       = 'fraction of saturation of target system';
pred.sat            = initialize.NaN;
pred.sat(:,tis)     = C_ex(:,tis) ./ ( ones(size(pred.t))*model.Km(tis) + C_ex(:,tis) );


%%% determine the saturation level

info.pred.A.tis     = 'amount in tissue space [nmol]';
pred.A.tis          = initialize.NaN;
pred.A.tis(:,tis)   = pred.C.tis(:,tis)*diag(model.V.tis(tis));

info.pred.A.body    = 'total amount in the body in [nmol]';
pred.A.body         = sum(pred.A.tis(:,tis),2);

info.pred.A.GItract = 'remaining amount in the GI tract in [nmol]';
pred.A.GItract      = sim.X(:,S.GItract);

info.pred.A.IVbag   = 'remaining amount in the IVbag in [nmol]';
pred.A.IVbag        = sim.X(:,S.IVbag);

info.pred.A.metab   = 'amount metabolized in [nmol]';
pred.A.metab        = sim.X(:,S.metab);


%%% determine standard output in [mg/L] or [mg]
%%%
stdout = individual.stdout; T = stdout.T;
initialize.NaN = ones(size(pred.t))*T.initialize.tissueDB.NaN; SF = model.SF; 

info.stdout.t         = 'simulation time in [min]';
stdout.t              = pred.t;

info.stdout.C.tis     = 'concentration-time profile in tissue space [mg/L]';
stdout.C.tis          = initialize.NaN;
stdout.C.tis(:,tis)   = SF.nmol_to_mg*pred.C.tis(:,tis);

stdout.C.tis(:,S.pla) = SF.nmol_to_mg*pred.C.tis(:,S.pla);
stdout.C.tis(:,S.blo) = stdout.C.tis(:,S.pla)*model.BP;
stdout.C.tis(:,[S.ven S.art]) = stdout.C.tis(:,S.blo)*[1 1];

info.stdout.nC.tis    = 'normalized concentration-time profile (nC.tis = C.tis/(eK.tis*(1-sigma)) in tissue space [mg/L] ';
stdout.nC.tis         = initialize.NaN;
for k = intersect(T.tissueDB,S.tissueDB)
    stdout.nC.tis(:,k)  = stdout.C.tis(:,k) / ( model.eK.tis(k).*(1-model.sigma.vas(k)) ) ;
end;
stdout.nC.tis(:,S.pla)= stdout.C.tis(:,S.pla);
stdout.nC.tis(:,[S.ven S.art]) = stdout.C.tis(:,S.blo)*([1 1]/model.BP);

if model.correct_for_residual_blood 
    resBlo = individual.fVtis.res; resPla = (1-individual.hct)*resBlo;
    for k = intersect(T.tissueDB,setdiff(S.tissueDB,[S.pla,S.blo,S.ven,S.art]))
        stdout.C.tis(:,k) = (1-resBlo(k))*stdout.C.tis(:,k) + resPla(k)*stdout.C.tis(:,model.S.pla);
    end;
end;

info.stdout.A.body    = 'total amount in the body in [mg]';
stdout.A.body         = SF.nmol_to_mg*pred.A.body;

info.stdout.A.GItract = 'remaining amount in the GI tract in [mg]';
stdout.A.GItract      = SF.nmol_to_mg*pred.A.GItract;

info.stdout.A.IVbag   = 'remaining amount in the IVbag in [mg]';
stdout.A.IVbag        = SF.nmol_to_mg*pred.A.IVbag;

info.stdout.A.metab   = 'amount metabolized in [mg]';
stdout.A.metab        = SF.nmol_to_mg*pred.A.metab;


individual.info   = info;
individual.pred   = pred;
individual.stdout = stdout;

end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_specificGraphicalOutput(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

return;

k = individual.T.pla; stdout = individual.stdout;
t = stdout.t/(60*24); C_tis = stdout.C.tis(:,k);

id = individual.id;
Km = individual.model.Km(k); Bmax = individual.model.Bmax(k);
C_eff = C_tis-Bmax-Km;
C_ex  = 1/2*( C_eff + sqrt( C_eff.^2 + 4*Km.*C_tis ) );
C_rs  = C_tis - C_ex;

figure(50);  if id==1 clf; end; 
hold on;
semilogy(t,C_tis,'k--',t,C_ex,'b-');
%ylim([1e-5 1e4]); xlim([min(t), max(t)]);
xlabel('t in [day]'); ylabel('Cpla in [mg/L]');
title(sprintf('id=%d, dose = %2.2f mg/kg',id,individual.study.dosing.bolus.dose));
hold off; 
set(gca,'YScale','log');

end

%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

