%%%
%%% Version: February 10th, 2014. 
%%% 
%%% This script specifies all inputs for the simulation of the 
%%% pharmacokinetics of different drugs in a virtual population 
%%% to be specified.
%%% 
%%% For reference, please cite:
%%%
%%% --> For small molecule drugs:
%%%
%%%     W. Huisinga, A. Solms, L. Fronton, S. Pilari,
%%%     Modeling Interindividual Variability in Physiologically Based 
%%%     Pharmacokinetics and Its Link to Mechanistic Covariate Modeling,
%%%     CPT: Pharmacometrics & Systems Pharmacology (2012) 1, e5; 
%%%     doi:10.1038/psp.2012.3
%%% 
%%% --> For monoclonal antibodies:
%%%
%%%     L. Fronton, S. Pilari, W. Huisinga
%%%     Monoclonal antibody disposition: A simplified PBPK model and its 
%%%     implications for the derivation & interpretation of classical 
%%%     compartment models,
%%%     J. Pharmacokinet Pharmacodyn (2014) 
%%%     doi:10.1007/s10928-014-9349-1
%%% 
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the 
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported 
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%

clear all;


%%% =================================================================
%%% The structure 'mammals' contains all the reference anatomical and
%%% physiological values that are used to generate individuals of the 
%%% species 
%%%
%%% * human, rat, mouse. 
%%% 
%%% The structure T contains the tissue indexing
%%%

mammals = GenericPBPKmodel_speciesDatabase;
T = mammals.T;


%%% =================================================================
%%% The structure 'drug' contains all physico-chemical drug properties
%%% as well as species related PK parameters (e.g., fuP, BP, CLblood).
%%% Choose a drug from the list below, classified into 
%%% 
%%% * small molecule drugs (sMD), monoclonal antibodies (mAb)
%%%
%%% Small molecule drugs: 
%%% further subdivided into moderate-to-strong bases, 
%%% weak bases, neutrals, acids and zwitterions. This property determines 
%%% the method to a-priori predict the tissue partition coefficients. 
%%% 
%%% Moderate-to-strong bases: Amitriptyline, Caffeine, 
%%% Diltiazem, Diphenhydramine, Imipramine, Lidocaine, Metoprolol, 
%%% Pindolol, Sildefanil, Theophylline
%%%
%%% Weak bases: Alprazolam, Diazepam, Flunitrazepam, Midazolam, Triazolam
%%%
%%% Acids: Amobarbital, Diclofenac, Hexobarbital, Ibuprofen, Methohexital, 
%%% Thiopental, Tolbutamide, Valproate, Warfarin
%%%
%%% test drugs: drugA (similar to Warfarin), drugB (similar to Lidocaine)
%%%
%%% Monoclonal antibodies:
%%% Zalutumumab, mAb7E3
%%%

drug.name = 'mAb7E3'; 
drug = GenericPBPKmodel_drugDatabase(drug.name);


%%% =================================================================
%%% The structure 'study' contains information about the dosing, dose and
%%% the route of administration (iv bolus, iv infusion, po), the
%%% simulation time in [min], the output time points, etc.
%%% Dosing can be fixed in unit [mg] or per body size descriptor (BSD)  
%%% in unit [mg/per unit of the BSD], e.g., [mg/kg BW]
%%% The observation time is the time between two dosings.
%%% 
%%% Units: dose in [mg] or [mg/BSD]
%%%        time in [min]
%%%

dosing.bolus.dose       = 8;   
dosing.bolus.per        = 'BW'; 

dosing.infusion.dose    = 0;  
dosing.infusion.per     = 'fixed'; 
dosing.infusion.tend    = 60; 

dosing.po.dose          = 0; 
dosing.po.per           = 'fixed';  

study.dosing            = dosing;
study.observationTime   = [0 60*24*60]; % min*hour*day
study.numberOfDosing    = 1;           


%%% =================================================================
%%% Determine all parameters for the reference individuals:
%%% 
%%% * human: adults 35y (20-50y, m/f), and children 15y (m/f), 10y (u), 5y (u)
%%%          with the last two being unisex (u), 
%%% * rat  : 250g , 475g 
%%% * mouse: 20g, 40g, Bax, FaH (NB: Bax and FaH refer to parameters as in Baxter 1994 and in Fronton 2014, respectively)
%%%

reference = GenericPBPKmodel_referenceIndividual(mammals,drug,study);


%%% =================================================================
%%% Generate virtual population based on antropometric data.
%%% Approaches for generating the population:
%%%
%%% human_sMD_PBPK_male      = adult male reference individual
%%% human_sMD_PBPK_adults    = adult m/f reference individuals
%%% human_sMD_PBPK_all       = age5, age10, age15m/f, age35m/f reference
%%%                            individuals
%%% human_sMD_PBPK_random    = 'size' many individuals of 'age35m' with mean BW & BH
%%%                            identical to reference.agesex and CV of BH & BMI 
%%%                            as specified in de la Grandmaison et al (2001)
%%%                            if combined with randomEffects
%%% human_sMD_PBPK_obese     = four individuals with BH identical to 'subtype' and
%%%                            BMI = [0.7 1 1.3 2]*BMI of reference 'subtype'
%%%
%%% rat_sMD_PBPK             = 250g and 475g rats
%%% rat250_male_sMD_PBPK     = 250g rat and human male adult
%%% rat_human_sMD_PBPK       = all subtypes of rat and human species
%%%
%%% sMD_PBPK_lumping         = compare detailed to lumped PBPK model, based on well-stirred tissue model
%%%                            for subtype specified below
%%% sMD_PBPK_vs_CMT          = PBPK model vs. classical 2 cmt model
%%% sMD_PBPK_identical       = 'size' many individuals of 'subtype', only reasonalbe 
%%%   
%%%
%%% mouse_mAb_PBPK           = mouse 11cmt PBPK model for mAbs
%%%                             by default 3 individuals;
%%%                             1: residual blood correction
%%%                             2: NO residual blood correction
%%%                             3: experimental data
%%% mouse_mAb_PBPK_vs_HaB    = 2CMT corrected Hansen and Balthasar model (mAb and endogenous IgG)
%%%                            mouse 11cmt PBPK model for mAbs
%%%                             by default 3 individuals;
%%%                             1: PBPK model with residual blood correction
%%%                             2: 2CMT corrected Hansen and Balthasar model (mAb and endogenous IgG)
%%%                             3: experimental data
%%% mouse_mAb_PBPK_lumping   = compare detailed to lumped PBPK model, only for mouse subtype
%%%                             by default 3 individuals;
%%%                             1: lumped PBPK model with residual blood correction
%%%                             2: PBPK model with residual blood correction
%%%                             3: experimental data
%%% mouse_mAb_PBPK_target    = compare detailed PBPK without and including a target
%%%                             1: PBPK model without a target and without residual blood correction
%%%                             2: PBPK model with a target and without residual blood correction
%%%                             3: experimental data

population.how2generate = 'mouse_mAb_PBPK';     % see above for choices
population.subtype      = 'human35m';           % only valid for SMD and when "population.how2generate" contains the option '_random','_identical','_lumping'
population.size         = 30;                   % only valid for SMD and when "population.how2generate" contains the option '_random','_identical'

individual = GenericPBPKmodel_virtualPopulation(reference,population);

%%% =================================================================
%%% Include random effects on  
%%% (a) hepatic (intrinsic) clearance CLint_hep = CLint.tis(liv):
%%%     Given the reference value CLint_ref, choose new value
%%%     CLint_hep randomly in [CLint_ref/(1+facX), CLint_ref*(1+facX)]
%%%     
%%% (b) Tissue-to-unbound plasma partition coefficients K.tis_up:
%%%     Given the reference value K_ref, choose new value
%%%     K.tis_up randomly in [K_ref/(1+facX), K_ref*(1+facX)]
%%%     Only tissue, but not K.ery_up is considered
%%%

randomEffects.sMD.facX.CLint_hep = 0;
randomEffects.sMD.facX.K_tis_up  = 0;

individual = GenericPBPKmodel_randomEffects(individual,randomEffects);


%%% =================================================================
%%% Simulate PBPK model for all individuals
%%%

individual = GenericPBPKmodel_simulatePBPKmodels(individual);


%%% =================================================================
%%% Data analysis and descriptive statistics of individuals and 
%%% the population
%%% 

stats = GenericPBPKmodel_dataAnalysis(individual);


%%% =================================================================
%%% Graphical output. Specify below, which plots to be shown
%%% 

ShowGraphics.allindividual.plasma  = 'yes';     % 'yes' or 'no'
ShowGraphics.allindividual.YScale  = 'log';     % 'log' or 'lin'
ShowGraphics.allindividual.figNr   = 10;

ShowGraphics.allTissues.individual = [1 2 3];   % specify id of individuals, e.g. [1 2 7]; [] = 'No plot'
                                                % refer to specifications for virtual poulation for referencing of default individuals 
ShowGraphics.allTissues.YScale     = 'log';     % 'log' or 'lin'
ShowGraphics.allTissues.figNr      = 11;
ShowGraphics.allTissues.oneFigOnly = 'yes';     % use same figure for all individuals 'yes'/'no' 

ShowGraphics.normConc.individual   = [];        % specify id of individual, e.g. [1]; [] = 'No plot'
                                                % refer to specifications for virtual poulation for for referencing of default individuals 
ShowGraphics.normConc.figNr        = 21;

ShowGraphics.percentiles.plasma    = 'yes';     % applies only to a virtual population of size 10 or larger 
ShowGraphics.percentiles.YScale    = 'log';
ShowGraphics.percentiles.figNr     = 25;

ShowGraphics.lowerLimitY           = 1e-2;      % sets the lower value of the Y axis

ShowGraphics.TestForLossOfMass.id    = [];      % for testing purposes.
                                                % specify id of individual, e.g. [1]; [] = 'No plot'
                                                % refer to specifications for virtual poulation for for referencing of default individuals
ShowGraphics.TestForLossOfMass.figNr = 30;


GenericPBPKmodel_graphicalOutput(individual,ShowGraphics); 

%%%
%%% =================================================================
