%%% 
%%% This function load experimental data specified in some file name.
%%% 
%%% Version: February 10th, 2014. 
%%% For references and citation, please see MAIN script.
%%% 
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the 
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported 
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function individual = GenericPBPKmodel_ExperimentalData(individual)


%%% load experimental data
%%%
individual = GenericPBPKmodel_loadExperimentalData(individual);

%%% determine standart output
%%%
individual = GenericPBPKmodel_determineStandartOutput(individual);

%%% make specific plots
%%%
GenericPBPKmodel_specificGraphicalOutput(individual);

end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

 


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_loadExperimentalData(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% Note: It is assumed that the names (abreviations) of the tissues
%%% specified below are part of the generic list defined in the species
%%% data base

model = individual.model;

%%% Source : Garg, PhD thesis (2007), published also in Garg and Balthasar, 
%%%          J Pharmacokinet Pharmacodyn (2007) 34:687?709
%%% Species: mouse, 25g (see p. 696, Section 'Model parameters')
%%% Units  : t in [days], C in [nM]


expdata = eval(model.filename);

unit = expdata.unit;
SF   = expdata.SF;

%%% -----------------------------------------------------------------------
%%% Define indexing
%%%

%%% tissue indexing
%%%
S = individual.T;

tissues = fieldnames(expdata.C);
S.allTis = [];
timepoints = [];
for k = 1:length(tissues)
    tis = tissues{k};
    S.allTis = union(S.allTis,S.(tis));
    timepoints = union(timepoints,expdata.t.(tis));
end
S.tissueDB = S.allTis;

S.maxIndex.tis = max(S.tissueDB);
S.initialize.time.NaN   = NaN*ones(length(timepoints),1);
S.initialize.conc.NaN   = NaN*ones(length(timepoints),S.maxIndex.tis);
S.initialize.amount.NaN = NaN*ones(length(timepoints),1);

%%% -----------------------------------------------------------------------
%%% Assign experimental data, i.e., data predicted by the experiment 
%%%

t = timepoints;
C.exp = S.initialize.conc.NaN;
limits.t = ones(S.maxIndex.tis,1)*[0 max(t)];
for k = S.allTis
    tis  = S.name{k};
    [~,ind,~] = intersect(timepoints,expdata.t.(tis));
    C.exp(ind,S.(tis)) = expdata.C.(tis);
    limits.t(S.(tis),:) = expdata.limits.t.(tis);
end;


%%% -----------------------------------------------------------------------
%%% Assign model parameters 
%%%

model.S  = S;
model.SF = SF;
model.expdata = expdata;
model.color   = expdata.color;
model.limits  = limits;
model.unit    = unit;

pred.t = t;
pred.C = C;

individual.model = model;
individual.pred  = pred;

end



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_determineStandartOutput(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model = individual.model; info = individual.info; S = model.S; 
pred  = individual.pred; SF = model.SF;

%%% determine standard output
%%%
stdout = individual.stdout; 

%%% determine those tissues that the experimental data and the standard
%%% model have in common
T = stdout.T; initialize.NaN = ones(size(S.initialize.time.NaN))*T.initialize.tissueDB.NaN;
tis = intersect(T.tissueDB,S.tissueDB);

info.stdout.t         = 'simulation time in [min]';
stdout.t              = SF.original_to_min*pred.t;

info.stdout.limits    = 'interesting range of experimental data';
stdout.limits.t       = SF.original_to_min*model.limits.t;

info.stdout.C.tis     = 'concentration-time profile in tissue space [mg/L]';
stdout.C.tis          = initialize.NaN;
stdout.C.tis(:,tis)   = SF.original_to_mg_per_L*pred.C.exp(:,tis);

stdout.C.tis(:,T.pla) = SF.original_to_mg_per_L*pred.C.exp(:,S.pla);


info.stdout.nC.tis    = 'normalized concentration-time profile (nC.tis = C.tis/eK.tis) in tissue space [mg/L] ';
stdout.nC.tis         = initialize.NaN;

info.stdout.A.body    = 'total amount in the body in [mg]';
stdout.A.body         = S.initialize.amount.NaN;

info.stdout.A.GItract = 'remaining amount in the GI tract in [mg]';
stdout.A.GItract      = S.initialize.amount.NaN;

info.stdout.A.IVbag   = 'remaining amount in the IVbag in [mg]';
stdout.A.IVbag        = S.initialize.amount.NaN;

info.stdout.A.metab   = 'amount metabolized in [mg]';
stdout.A.metab        = S.initialize.amount.NaN;


individual.info   = info;
individual.pred   = pred;
individual.stdout = stdout;

end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_specificGraphicalOutput(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% define here specific output for the experimental data

return;

pred = individual.pred; model = individual.model; S = model.S; color = model.color;

%%% plot experimental plasma and tissue data
%%%
figNr = 100;
figure(figNr); clf;
hold on;
for n = S.allTis
    tis = S.name{n};
    ind = find(isfinite(pred.C.exp(:,S.(tis))));
    semilogy(pred.t(ind),pred.C.exp(ind,S.(tis)),color.(tis),'LineWidth',4,'LineStyle','-');
    semilogy(pred.t(ind),pred.C.exp(ind,S.(tis)),'d','MarkerEdgeColor','k','MarkerFaceColor','g','MarkerSize',5);
end;
hold off;
xlabel(sprintf('time in [%s]',model.unit.t)); ylabel(sprintf('C_{tis} in [%s]',model.unit.C));
xlim([0 10]);
title('Experimental data')
set(gca,'YScale','log'); axis square;
fett(figNr); % print -depsc -f100 experimentalData

%%% plot normalized experimental plasma and tissue data
%%%
figNr = figNr+1;
figure(figNr); clf;
hold on;
for n = S.allTis
    tis = S.name{n};
    ind = find(isfinite(pred.C.exp(:,S.(tis))));
    Z = pred.C.exp(ind(end),S.(tis));
    if n == S.pla
        Z = pred.C.exp(ind(end)-3,S.(tis))*1.13;
    end;
    semilogy(pred.t(ind),pred.C.exp(ind,S.(tis))/Z,color.(tis),'LineWidth',4,'LineStyle','-');
    semilogy(pred.t(ind),pred.C.exp(ind,S.(tis))/Z,'d','MarkerEdgeColor','k','MarkerFaceColor','g','MarkerSize',5);
end;
hold off;
xlabel(sprintf('time in [%s]',model.unit.t)); ylabel('C_{tis}/C_{tis}(Tend)');
xlim([0 10]);
title('Normalized experimental data')
set(gca,'YScale','log'); axis square;
fett(figNr); % print -depsc -f101 experimentalDataNormalized


end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = fett(figNr)
    
figure(figNr);
fs = 20;
set(gca,'FontSize',fs);
set(get(gca,'title'),'Fontsize',fs);
set(get(gca,'xlabel'),'Fontsize',fs);
set(get(gca,'ylabel'),'Fontsize',fs);

end

%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

