%%% Version: Saskia May 29th, 2015
%%%
%%% This function scales anatomical and physiological data of a reference
%%% Individual to some target values.
%%%
%%% For references and citation, please see MAIN script.
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function Individual = GenericPBPKmodel_scaleIndividual(Individual)


switch Individual.species.scalingMethod
    
    case 'LBW';
        
        Individual = GenericPBPKmodel_LBW_based_scaling(Individual);
        
    case 'BW';
        
        Individual = GenericPBPKmodel_BW_based_scaling(Individual);
        
        
    otherwise
        message = sprintf('Unknown method >>%s<< to scale an Individual!',Individual.species.scalingMethod);
        GenericPBPKmodel_ReportErrorMessage(message);
end;



end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function target = GenericPBPKmodel_BW_based_scaling(reference)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

T = reference.species.T;

%%% adopt reference values and adapt subsequently those parameters that
%%% are assumed to depend on the covariates BW and BH
target = reference;

species = target.species;

reference.species.BW = species.BWref;

%%% scaling of brain
SV.bra            = 1;
species.OW.tis(T.bra) = SV.bra * reference.species.OW.tis(T.bra);

%%% determine scaling factor for all remaining tissues
species_BWrest    = species.BW - species.OW.tis(T.bra);
reference_BWrest  = reference.species.BW  - reference.species.OW.tis(T.bra);
SV.tis            = species_BWrest/reference_BWrest;

m  = setdiff(T.tissueDB,[T.bra]);

%%% scaling of all remaining tissues
species.OW.tis(m) = SV.tis * reference.species.OW.tis(m);

%%% scaling of total blood volume
species.V.blood  = SV.tis * reference.species.V.blood;
species.V.vas    = species.fVblood.reg * species.V.blood;

%%% update fractional organ weights
species.fBW.OWexp = species.OW.tis/species.BW;
species.fBW.blood = species.V.blood/species.BW;

m = T.tissueDB;
species.V.tis(m) = species.OW.tis(m)./species.dens(m);

species.V.tot    = species.V.vas + species.V.tis;


%%% scaling of blood flows
SQ               = SV.tis;
species.co       = SQ * reference.species.co;
species.Q.blo    = species.fQco * species.co;


%%% assign scaled values
species.SQ=SQ;

target.species   = species;
target.species.SQ= species.SQ;


end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function target = GenericPBPKmodel_LBW_based_scaling(reference)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

T = reference.species.T;

%%% adopt reference values and adapt subsequently those parameters that
%%% are assumed to depend on the covariates BW and BH
target = reference;

species = reference.species;
BH  = species.BH;
BW  = species.BW;

reference.species.BH = species.BHref;
reference.species.BW = species.BWref;

%%% body  mass index
species.BMI = BW/BH^2;

%%% LBW and adipose tissue
species.LBW  = FFM(BW,BH,species.sex);  % Note: LBW approximated by FFM!

species.OW.tis(T.adi) = BW-species.LBW;

%%% scaling of skin
species.BSA = BSA(BW,BH);
SV.ski      = species.BSA / reference.species.BSA;
species.OW.tis(T.ski) = SV.ski * reference.species.OW.tis(T.ski);

%%%% scaling of brain
SV.bra = 1;
species.OW.tis(T.bra) = SV.bra * reference.species.OW.tis(T.bra);

%%% determine scaling factor for all remaining tissues
m  = setdiff(T.tissueDB,[T.adi T.ski T.bra]);
species_BWrest    = species.BW - sum(species.OW.tis([T.adi T.ski T.bra]));
reference_BWrest  = reference.species.BW  - sum(reference.species.OW.tis([T.adi T.ski T.bra]));
SV.tis            = species_BWrest/reference_BWrest;

%%% determine scaling of all remaining tissues
species.OW.tis(m) = SV.tis * reference.species.OW.tis(m);

%%% scaling of total blood volume
species.V.blood  = SV.tis * reference.species.V.blood;
species.V.vas    = species.fVblood.reg * species.V.blood;

%%% update fractional organ weights
species.fBW.OWexp = species.OW.tis/species.BW;
species.fBW.blood = species.V.blood/species.BW;

m = T.tissueDB;
species.V.tis(m) = species.OW.tis(m)./species.dens(m);
species.V.tot    = species.V.vas + species.V.tis;

%%% scaling of blood flows
SQ               = SV.tis;
species.co       = SQ * reference.species.co;
species.Q.blo    = species.fQco * species.co;

%%% assign scaled values
target.species   = species;

end



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function out = FFM(BW,BH,sex)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

% determine fat free mass (FFM) according to
% Green B, Duffull SB. Clin Pharmacol Ther Vol 72, 2002

BMI = BW/BH^2;
switch sex
    case 'male'
        out = 9.27e3*BW/(6.68e3+216*BMI);
    case 'female'
        out = 9.27e3*BW/(8.78e3+244*BMI);
    otherwise
        fprintf('\n\n Unknow sex to compute LBW! \n\n');
        error('in determining LBW');
end;

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function out = BSA(BW,BH)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

% Determine body surface area (BSA) in [m^2]
% Source: Mosteller, New Engl J Med, Vol 317, 1987

out = sqrt(BH*BW/36);

end


%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
