%%% Version: October 31st, 2014
%%% 
%%% This function specifies a large virtual population towards automated
%%% quality checking.
%%% 
%%% For references and citation, please see MAIN script.
%%% 
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the 
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported 
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function individual = GenericPBPKmodel_qualityCheck(reference)


individual_struct = struct('id',[],'details',[],'color',[],'species',[],'dosing',[],'drug',[],'model',[],'pred',[],'stdout',[],'info',[]);;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% sMD - sMD - sMD - sMD - sMD - sMD - sMD - sMD - sMD - sMD - sMD - sMD -  
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

single_dose.drug      = [];
single_dose.route     = [];            % 'iv_bolus', 'iv_infusion', 'po'
single_dose.dose      = [];            % in [mg] or [mg]/per BSD (body size descriptor)'
single_dose.doseUnit  = 'mg';          % 'mg' or 'mg/kg BW', 'mg/kg LBW', 'mg/m2 BSA' 
single_dose.time      = 0;             % time of first administration 
single_dose.timeUnit  = 'h';
single_dose.repeats   = 1;             % number of administation events
single_dose.Tinf      = NaN;           % duration of iv infusion 
single_dose.interval  = NaN;           % multiple dosing interval 


dosing       = single_dose; % dose in [mg], adminstration at t=0, time in [h]
dosing.drug  = 'drugB';
dosing.dose  = 100;
dosing.doseUnit  = 'mg';
dosing.route = 'iv_bolus';


% ======================================================================= %
fprintf('\n   --> case rat_human_sMD_PBPK (id 1-8)');  
% ======================================================================= %

%%% allocating memory for efficiency
individual(8) = individual_struct;

id = 1;
individual(id).species    = reference.human.male_35y_70kg_176cm;
individual(id).dosing     = dosing;
individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';
individual(id).color      = 'r-';

id = 2;
individual(id).species    = reference.human.female_35y_60kg_163cm;
individual(id).dosing     = dosing;
individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';
individual(id).color      = 'b-';

id = 3;
individual(id).species    = reference.human.male_15y_54kg_167cm;
individual(id).dosing     = dosing;
individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';
individual(id).color      = 'r--';

id = 4;
individual(id).species    = reference.human.female_15y_53kg_161cm;
individual(id).dosing     = dosing;
individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';
individual(id).color      = 'b--';

id = 5;
individual(id).species    = reference.human.female_10y_32kg_138cm;
individual(id).dosing     = dosing;
individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';
individual(id).color      = 'k-';

id = 6;
individual(id).species    = reference.human.male_5y_19kg_109cm;
individual(id).dosing     = dosing;
individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';
individual(id).color      = 'k--';

id = 7;
individual(id).species    = reference.rat.male_475g;
individual(id).dosing     = dosing;
individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';
individual(id).color      = 'm--';

id = 8;
individual(id).species    = reference.rat.male_250g;
individual(id).dosing     = dosing;
individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';
individual(id).color      = 'm-';

N_tot = 8;
        
% ======================================================================= %
fprintf('\n   --> case human_sMD_PBPK_random (id %d-%d)',N_tot+1,N_tot+5); 
% ======================================================================= %

%%% based on de la Grandmaison characteristics
%%% the original puplication by de la Grandmaison et al (2001)
%%% contained 355 male/329 female adults

%%% specify size (N) of population and template species
N = 5;
templateSpecies = reference.human.male_35y_70kg_176cm;

%%% allocating memory for efficiency
individual(N+N_tot) = individual_struct;

%%% generating randomly pairs of BH and BW
[BH, BW] = delaGrandmaison(N,templateSpecies.sex);

for id=(1:N)+N_tot
    
    individual(id).species    = templateSpecies;
    individual(id).species.BH = BH(id-N_tot);
    individual(id).species.BW = BW(id-N_tot);
    
    %%% time span needs to be identical for all individuals to
    %%% allow for percentile plots
    individual(id).pred.timeSpan = [0:0.1:12];      % simulation time span
    individual(id).pred.timeUnit = 'h';             % unit of timeSpan
    
    individual(id).dosing     = dosing;
    individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';
    individual(id).color      = 'k-';
    
    
end;

N_tot = N_tot + N;
        
% ======================================================================= %
fprintf('\n   --> case human_sMD_PBPK_obese (id %d-%d)',N_tot+1,N_tot+4); 
% ======================================================================= %
       
        
%%% generate five individuals with varying BW resp. BMI
templateSpecies = reference.human.male_35y_70kg_176cm;

BWref = templateSpecies.BW;
BW    = [0.7*BWref BWref 1.3*BWref 2*BWref];
color = {'m-', 'r-', 'b-', 'b--'};

%%% allocating memory for efficiency
N = length(BW);
individual(N+N_tot) = individual_struct;

for id=(1:N)+N_tot
    
    individual(id).species    = templateSpecies;
    individual(id).species.BW = BW(id-N_tot);
    individual(id).dosing     = dosing;
    individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';
    individual(id).color      = color{id-N_tot};
    
end;

N_tot = N_tot + N;


% ======================================================================= %
fprintf('\n   --> case sMD_PBPK_lumping (id %d-%d)',N_tot+1,N_tot+2);
% ======================================================================= %
        
%%% compare full to lumped model
templateSpecies = reference.human.male_35y_70kg_176cm;

%%% allocating memory for efficiency
individual(2+N_tot) = individual_struct;

id = N_tot + 1;
individual(id).species    = templateSpecies;
individual(id).dosing     = dosing;
individual(id).model.type = 'sMD_lumpedPBPK_xCMT_wellstirred';
individual(id).color      = 'b-';

lumping = { {'ven','lun','art','bra','hea','kid','gut','spl'},{'liv'}, {'ski'},{'adi','mus'},{'bon'} };
individual(id).model.lumping = lumping;

id = N_tot + 2;
individual(id).species    = templateSpecies;
individual(id).dosing     = dosing;
individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';
individual(id).pred.plot_normalized_conc = 'yes';
individual(id).color      = 'r--';
                
N_tot = N_tot + 2;

% ======================================================================= %
fprintf('\n   --> case sMD_PBPK_vs_CMT (id %d-%d)',N_tot+1,N_tot+2); 
% ======================================================================= %

%%% compare PBPK to classical compartment model
%%% with C_central = C_plasma (not C_blood)!

%%% compare full to lumped model
templateSpecies = reference.human.male_35y_70kg_176cm;

%%% allocating memory for efficiency
individual(2+N_tot) = individual_struct;

id = N_tot+1;
individual(id).species         = templateSpecies;
individual(id).dosing          = dosing;
individual(id).model.type      = 'CMTmodel_xCMT_PLASMA_macroConstants_linearCL';
individual(id).model.V         = [12.8*2.4*0.85 54.3*2.4*0.85];  % in [L]
individual(id).model.Q         = [1.76*0.85];    % in [L/min]
individual(id).model.CL        = [1.095*0.85 0]; % in [L/min]
individual(id).model.F.bio     = 1.0;       % set to NaN if not needed
individual(id).model.lambda_po = 0.018;     % set to NaN if not needed
individual(id).color           = 'b-';

id = N_tot+2;
individual(id).species         = templateSpecies;
individual(id).dosing          = dosing;
individual(id).model.type      = 'sMD_PBPK_13CMT_wellstirred';
individual(id).color           = 'r--';

N_tot = N_tot + 2;

        
% ======================================================================= %
fprintf('\n   --> case sMD_PBPK_iv_bolus_infusion_po (id %d-%d)',N_tot+1,N_tot+3); 
% ======================================================================= %
        
%%% compare full to lumped model
templateSpecies = reference.human.male_35y_70kg_176cm;

%%% different routes
dosing_iv_bolus = dosing;
dosing_iv_bolus.route = 'iv_bolus';

dosing_iv_infusion = dosing;
dosing_iv_infusion.route = 'iv_infusion';
dosing_iv_infusion.Tinf = 1;

dosing_po = dosing;
dosing_po.route = 'po';

%%% allocating memory for efficiency
individual(N_tot+3) = individual_struct;

id = N_tot+1;
individual(id).species    = templateSpecies;
individual(id).dosing     = dosing_iv_bolus;
individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';
individual(id).color      = 'r-';

id = N_tot+2;
individual(id).species    = templateSpecies;
individual(id).dosing     = dosing_iv_infusion;
individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';
individual(id).color      = 'b--';

id = N_tot+3;
individual(id).species    = templateSpecies;
individual(id).dosing     = dosing_po;
individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';
individual(id).color      = 'k--';

N_tot = N_tot+3;


% ======================================================================= %
fprintf('\n   --> case sMD_PBPK_multiple_dosing (id %d-%d)',N_tot+1,N_tot+3);
% ======================================================================= %

%%% compare different multiple dosing schedule:
%%% constant infusion with 1h infusions every 24h, 12h or 8h.
%%%
templateSpecies = reference.human.male_35y_70kg_176cm;

%%% define different multiple dosing schedules,
%%% take drug and dose as defined in main script
%%%
dosing.route = 'iv_infusion';
dosing.Tinf  = 1;

dosing_24h = dosing;
dosing_24h.Tinf = 1;
dosing_24h.interval = 24;
dosing_24h.repeats  = 5;

dosing_12h = dosing;
dosing_12h.dose = dosing.dose/2;
dosing_12h.Tinf = 1;
dosing_12h.interval = 12;
dosing_12h.repeats  = 2*dosing_24h.repeats;

dosing_8h = dosing;
dosing_8h.dose = dosing.dose/3;
dosing_8h.Tinf = 1;
dosing_8h.interval = 8;
dosing_8h.repeats  = 3*dosing_24h.repeats;

dosing_const_infusion = dosing;
dosing_const_infusion.dose = dosing.dose*dosing_24h.repeats;
dosing_const_infusion.Tinf = 24*dosing_24h.repeats;
dosing_const_infusion.repeats = 1;
dosing_const_infusion.interval = NaN;

%%% allocating memory for efficiency
individual(3+N_tot) = individual_struct;

id = N_tot+1;
individual(id).species    = templateSpecies;
individual(id).dosing     = dosing_const_infusion;
individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';
individual(id).color      = 'r-';

id = N_tot+2;
individual(id).species    = templateSpecies;
individual(id).dosing     = dosing_24h;
individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';
individual(id).color      = 'b-';

id = N_tot+3;
individual(id).species    = templateSpecies;
individual(id).dosing     = dosing_12h;
individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';
individual(id).color      = 'k-';

N_tot = N_tot+3;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% mAb - mAb - mAb - mAb - mAb - mAb - mAb - mAb - mAb - mAb - mAb - mAb -  
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

single_dose.drug      = [];
single_dose.route     = [];            % 'iv_bolus', 'iv_infusion', 'po'
single_dose.dose      = [];            % in [mg] or [mg]/per BSD (body size descriptor)'
single_dose.doseUnit  = 'mg/kg BW';          % 'mg' or 'mg/kg BW', 'mg/kg LBW', 'mg/m2 BSA' 
single_dose.time      = 0;             % time of first administration 
single_dose.timeUnit  = 'h';
single_dose.repeats   = 1;             % number of administation events
single_dose.Tinf      = NaN;           % duration of iv infusion 
single_dose.interval  = NaN;           % multiple dosing interval 


dosing       = single_dose; % dose in [mg/kg BW], adminstration at t=0, time in [h]
dosing.drug  = 'mAb7E3';
dosing.dose  = 8;
dosing.route = 'iv_bolus';




% ======================================================================= %
fprintf('\n   --> case mouse_mAb_PBPK (id %d-%d)',N_tot+1,N_tot+6);
% ======================================================================= %

mouseSpecies = reference.mouse.male_25g_Fro;

%%% allocating memory for efficiency
individual(N_tot+4) = individual_struct;

id = N_tot+1;
individual(id).species    = mouseSpecies;
individual(id).dosing     = dosing;
individual(id).model.type = 'mAb_PBPK_11CMT_extravasLim_int';
individual(id).model.FcRn_status = 'wild-type';
individual(id).color      = 'r--';

id = N_tot+2;
individual(id).species    = mouseSpecies;
individual(id).dosing     = dosing;
individual(id).model.type = 'ExperimentalData';
individual(id).model.filename = 'ExperimentalData_mouse_wt_mAb7E3_Garg2007';
individual(id).color      = 'k-';

id = N_tot+3;
individual(id).species    = mouseSpecies;
individual(id).dosing     = dosing;
individual(id).model.type = 'mAb_PBPK_11CMT_extravasLim_int';
individual(id).model.FcRn_status = 'knock-out';
individual(id).color      = 'r--';

id = N_tot+4;
individual(id).species    = mouseSpecies;
individual(id).dosing     = dosing;
individual(id).model.type = 'ExperimentalData';
individual(id).model.filename = 'ExperimentalData_mouse_ko_mAb7E3_Garg2007';
individual(id).color      = 'k-';

N_tot = N_tot+4;

% ======================================================================= %
fprintf('\n   --> case mouse_mAb_PBPK_vs_HaB (id %d-%d)',N_tot+1,N_tot+3); 
% ======================================================================= %

mouseSpecies = reference.mouse.male_25g_Fro;

%%% allocating memory for efficiency
individual(N_tot+3) = individual_struct;

id = N_tot+1;
individual(id).species    = mouseSpecies;
individual(id).dosing     = dosing;
individual(id).model.type = 'mAb_PBPK_11CMT_extravasLim_int';
individual(id).model.FcRn_status = 'wild-type';
individual(id).color      = 'r-';

id = N_tot+2;
individual(id).species    = mouseSpecies;
individual(id).dosing     = dosing;
individual(id).model.type = 'mAb7E3_mouse_2CMT_corrHansenBalthasar';
individual(id).color      = 'b--';

id = N_tot+3;
individual(id).species    = mouseSpecies;
individual(id).dosing     = dosing;
individual(id).model.type = 'ExperimentalData';
individual(id).model.filename = 'ExperimentalData_mouse_wt_mAb7E3_Garg2007';
individual(id).color      = 'k-';

N_tot = N_tot+3;

        
% ======================================================================= %
fprintf('\n   --> case mouse_mAb_PBPK_lumping (id %d-%d)',N_tot+1,N_tot+3);
% ======================================================================= %

mouseSpecies = reference.mouse.male_25g_Fro;

%%% allocating memory for efficiency
individual(N_tot+3) = individual_struct;

id = N_tot+1;
individual(id).species    = mouseSpecies;
individual(id).dosing     = dosing;
individual(id).model.type = 'mAb_lumpedPBPK_xCMT_extravasLim_int';
individual(id).model.correct_for_residual_blood = 0;
individual(id).model.useABCs = 1;
individual(id).model.FcRn_status = 'wild-type';
individual(id).color      = 'b-';
%%% lumping
lumping = { {'pla'},{'lun','hea','kid','gut','spl','liv'},{'ski','adi','mus','bon'} };
individual(id).model.lumping = lumping;

id = N_tot+2;
individual(id).species    = mouseSpecies;
individual(id).dosing     = dosing;
individual(id).model.type = 'mAb_PBPK_11CMT_extravasLim_int';
individual(id).model.correct_for_residual_blood = 0;
individual(id).model.useABCs = 1;
individual(id).model.FcRn_status = 'wild-type';
individual(id).pred.plot_normalized_conc = 'yes';
individual(id).color      = 'r--';

id = N_tot+3;
individual(id).species    = mouseSpecies;
individual(id).dosing     = dosing;
individual(id).model.type = 'ExperimentalData';
individual(id).model.filename = 'ExperimentalData_mouse_wt_mAb7E3_Garg2007';
individual(id).color      = 'k-';

N_tot = N_tot+3;

% ======================================================================= %
fprintf('\n   --> case mouse_mAb_PBPK_target (id %d-%d)',N_tot+1,N_tot+3);
% ======================================================================= %

mouseSpecies = reference.mouse.male_25g_Fro;
T = GenericPBPKmodel_tissueIndexing;

%%% allocating memory for efficiency
individual(N_tot+3) = individual_struct;

id = N_tot+1;
individual(id).species    = mouseSpecies;
individual(id).dosing     = dosing;
individual(id).model.type = 'mAb_PBPK_11CMT_extravasLim_int_target';
individual(id).model.FcRn_status = 'wild-type';
individual(id).color      = 'b-';
%%% specify target expressing tissues
%%%
target_expr_tissues = [T.mus];
all_equal           = ones(size(target_expr_tissues));
individual(id).model.B_max = 6e-2   * all_equal; % in [nmol]
individual(id).model.K_m   = 1.2e-1 * all_equal; % in [nmol/L]
individual(id).model.k_deg = 5e-5   * all_equal; % in [1/h]
individual(id).model.target_expr_tissues = target_expr_tissues;

id = N_tot+2;
individual(id).species    = mouseSpecies;
individual(id).dosing     = dosing;
individual(id).model.type = 'mAb_PBPK_11CMT_extravasLim_int';
individual(id).model.correct_for_residual_blood = 0;
individual(id).model.useABCs = 1;
individual(id).model.FcRn_status = 'wild-type';
individual(id).color      = 'r--';

id = N_tot+3;
individual(id).species    = mouseSpecies;
individual(id).dosing     = dosing;
individual(id).model.type = 'ExperimentalData';
individual(id).model.filename = 'ExperimentalData_mouse_wt_mAb7E3_Garg2007';
individual(id).color      = 'k-';

N_tot = N_tot+3;


end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [BH,BW] = delaGrandmaison(N,sex)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% Assign parameter values for underlying distributions
%%% To generate pairs of (BH,BW) values, first (BH,BMI) pairs are
%%% generated, assuming that
%%% BH  ~ Normal(muBH,sdBH^2)
%%% BMI ~ logNormal(log_muBMI,log_sdBMI^2)
%%% These are then used to generate BW from the relation BMI = BW/BH^2
%%%
switch sex
    
    case 'male';
        muBH  = 1.72; sdBH  = 0.075;
        muBMI = 22.8; sdBMI = 3.3;
        
    case 'female';
        muBH  = 1.61; sdBH  = 0.075;
        muBMI = 22.5; sdBMI = 4.5;
        
end;
%%% transforming mu and sd to the scale of the underlying normal distribution
log_muBMI = log(muBMI^2/sqrt(muBMI^2+sdBMI^2));
log_sdBMI = sqrt(log(sdBMI^2/muBMI^2+1));


BH  = muBH + sdBH*randn(1,N);
BMI = exp( log_muBMI + log_sdBMI*randn(1,N) );

BW  = BMI.*(BH.^2);

end



%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

