%%% Version: October 31st, 2016
%%%
%%% This script specifies all inputs for the simulation of the
%%% pharmacokinetics of different mAbs in a virtual population
%%% to be specified.
%%%

%%% For reference, please cite:
%%%
%%% --> For small molecule drugs:
%%%
%%%     W. Huisinga, A. Solms, L. Fronton, S. Pilari,
%%%     Modeling Interindividual Variability in Physiologically Based
%%%     Pharmacokinetics and Its Link to Mechanistic Covariate Modeling,
%%%     CPT: Pharmacometrics & Systems Pharmacology (2012) 1, e5;
%%%     doi:10.1038/psp.2012.3
%%%
%%% --> For monoclonal antibodies:
%%%
%%%     L. Fronton, S. Pilari, W. Huisinga
%%%     Monoclonal antibody disposition: A simplified PBPK model and its
%%%     implications for the derivation & interpretation of classical
%%%     compartment models,
%%%     J. Pharmacokinet Pharmacodyn (2014)
%%%     doi:10.1007/s10928-014-9349-1
%%%
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%

clear all; close all;

fprintf('\n\n Start simulation ...');


%%% =================================================================
%%% Specify dosing, including the following information
%%%
%%% 'drug'      sMD or mAb (see below)
%%% 'route'     of adminstration: iv bolus, iv infusion or per oral
%%% 'dose'      of the drug, specified in units specified by ...
%%% 'doseUnit'  unit of the dose in 'mg' or 'mg/kg BW', 'mg/kg LBW', 'mg/m2 BSA'
%%% 'time'      of drug administration in units specified by ...
%%% 'timeUnit'  unit of time in 'min', 'h', 'd' (day), 'w' (week)
%%% 'repeats'   single dose (=1) or multiple dosing (>1)
%%% 'Tinf'      duration of infusion (if applicable)
%%% 'interval'  dosing interval (multiple dosing case, if applicable)
%%%
%%% SMALL MOLECULE DRUGS:
%%% further subdivided into moderate-to-strong bases,
%%% weak bases, neutrals, acids and zwitterions. This property determines
%%% the method to a-priori predict the tissue partition coefficients.
%%%
%%% Moderate-to-strong bases: Amitriptyline, Caffeine,
%%% Diltiazem, Diphenhydramine, Imipramine, Lidocaine, Metoprolol,
%%% Pindolol, Sildefanil, Theophylline
%%%
%%% Weak bases: Alprazolam, Diazepam, Flunitrazepam, Midazolam, Triazolam
%%%
%%% Acids: Amobarbital, Diclofenac, Hexobarbital, Ibuprofen, Methohexital,
%%% Thiopental, Tolbutamide, Valproate, Warfarin
%%%
%%% test drugs: drugA (similar to Warfarin), drugB (similar to Lidocaine)
%%%
%%% MONOCLONAL ANTIBODIES:


single_dose.drug      = [];
single_dose.route     = [];                         % 'iv_bolus', 'iv_infusion', 'po'
single_dose.dose      = [];                         % in [mg] or [mg]/per BSD (body size descriptor)'
single_dose.doseUnit  ='mg';                        % 'mg' or 'mg/kg BW', 'mg/kg LBW', 'mg/m2 BSA'
single_dose.time      = 0;                          % time of first administration
single_dose.timeUnit  = 'min';
single_dose.repeats   = 1;                          % number of administration events
single_dose.Tinf      = NaN;
single_dose.interval  = NaN;

dosing       = single_dose;
dosing.drug  = 'mAbMOPC21';
dosing.dose  = 0.0038;
dosing.route = 'iv_bolus';



%%% =================================================================
%%% Load characteristics of reference individuals
%%%
%%% The following reference individuals are available
%%%
%%%     reference.human.male_35y_70kg_176cm
%%%     reference.human.female_35y_60kg_163cm
%%%     reference.human.male_15y_54kg_167cm
%%%     reference.human.female_15y_53kg_161cm
%%%     reference.human.unisex_10y_32kg_138cm
%%%     reference.human.unisex_5y_19kg_109cm
%%%
%%%     reference.rat.male_250g
%%%     reference.rat.male_475g
%%%
%%%     reference.mouse.male_25g
%%%     reference.mouse.male_40g
%%%     reference.mouse.male_22g_Bax
%%%     reference.mouse.male_22g_Fro

reference = GenericPBPKmodel_referenceIndividual();


%%% =================================================================
%%% Generate virtual population based on antropometric data.


population.how2generate = 'identical_mAb_PBPK_nude_FcRnplasma';



individual = GenericPBPKmodel_virtualPopulation(population,dosing,reference);




%%% =================================================================
%%% Assign anatomical & physiolocial species data and assign
%%% drug-specific data relevant for the chosen dosing
%%%

individual = GenericPBPKmodel_assignSpeciesData(individual);
T = GenericPBPKmodel_tissueIndexing;

individual = GenericPBPKmodel_assignDrugData(individual);


%%% =================================================================
%%% Random effects
%%% State factor of perturbation, i.e., if facX = 2, then the perturbed
%%% value of a quantity X will ly between 1/(1+facX) * X and (1+facX) * X
%%%


k=1;
randomEffects.mAb.facX.V_blood  =12/10*k;
randomEffects.mAb.facX.hct      =11/10*k;
randomEffects.mAb.facX.Dose     =0*k;
individual = GenericPBPKmodel_randomEffects(individual,randomEffects);



%%% =================================================================
%%% Simulate PBPK model for all individuals
%%%
%%% Note: If timeSpan consists only of initial and final time, i.e.,
%%%     timeSpan = [T0 Tend],
%%% then Tend will be automatically adapted, if final dosing event is
%%% at a time later than Tend

pred.timeSpan = [0 6];          % simulation time span
pred.timeUnit = 'd';             % unit of timeSpan

individual    = GenericPBPKmodel_simulatePBPKmodels(individual,pred);


%%% =================================================================
%%% Data analysis and descriptive statistics of individuals and
%%% the population
%%%

stats = GenericPBPKmodel_dataAnalysis(individual);


%%% =================================================================
%%% Graphical output. Specify below, which plots to be shown
%%%

ShowGraphics.plasma.allIndividuals = 'yes';     % 'yes' or 'no'
ShowGraphics.plasma.YScale         = 'log';     % 'log' or 'lin'
ShowGraphics.plasma.figNr          = 10;

allIds = 1:length(individual);
ShowGraphics.allTissues.individual = [];   % specify id of individuals, e.g. [1 2 7]; [] = 'No plot'

ShowGraphics.allTissues.YScale     = 'log';     % 'log' or 'lin'
ShowGraphics.allTissues.figNr      = 11;
ShowGraphics.allTissues.oneFigOnly = 'yes';     % use same figure for all individuals 'yes'/'no'

ShowGraphics.percentiles.plasma    = 'no';     % applies only to a virtual population of size 10 or larger
ShowGraphics.percentiles.YScale    = 'log';
ShowGraphics.percentiles.figNr     = 25;

ShowGraphics.lowerLimitY           = 1e-1;      % sets the lower value of the Y axis

ShowGraphics.TestForLossOfMass.id    = [];      % for testing purposes.



% GenericPBPKmodel_graphicalOutput(individual,ShowGraphics);


fprintf('Simulation done\n')
fprintf('Compute Percentiles\n')

%%%compute percentiles
showpercentilesplasma =1;
showpercentilestissues=1;

if showpercentilesplasma
    
    figure25=figure(25); clf
    
    
    %%% Plot percentiles for venous blood concentrations for entire population
    %%% Only applicable for population of size>=10
    
    T = individual(1).stdout.T;
    compound = individual(1).drug.listOfDrugs{1};
    
    xmin=-0.25;
    
    switch individual(1).drug.(compound).class
        
        
        case 'mAb'
            %%% scale time from [min] to [day]
            ShowGraphics.SF.time = 1/(60*24);
            ShowGraphics.unit.time = 'day';
            
            %%% leave concentration in [mg]/L
            ShowGraphics.SF.conc = 1;
            ShowGraphics.unit.conc = 'mg/L';
            
            %%% uncomment below in case you want [nmol/L]
            ShowGraphics.SF.conc = individual(1).drug.(compound).SF.mg_to_nmol;
            ShowGraphics.unit.conc = 'nmol/L';
            
        case 'pAb' %e.g. IVIG
            %%% scale time from [min] to [day]
            ShowGraphics.SF.time = 1/(60*24);
            ShowGraphics.unit.time = 'day';
            
            %%% leave concentration in [mg]/L
            ShowGraphics.SF.conc = 1;
            ShowGraphics.unit.conc = 'mg/L';
            
            %%% uncomment below in case you want [nmol/L]
            ShowGraphics.SF.conc = individual(1).drug.(compound).SF.mg_to_nmol;
            ShowGraphics.unit.conc = 'nmol/L';
            
        otherwise
            
            %%% leave time in [min]
            ShowGraphics.SF.time = 1;
            ShowGraphics.unit.time = 'min';
            
            %%% leave concentration in [mg/L]
            ShowGraphics.SF.conc = 1;
            ShowGraphics.unit.conc = 'mg/L';
            
    end;
    
    
    %%% combine all plasma concentration-time profiles in a single matrix
    %%%
    
    tspan = individual(1).stdout.t* ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    i=0;j=1;
    dosegroup= individual(1).pred.dosegroup;
    color(1)={individual(1).color};
    for id=1: length(individual)
        if ~strcmp(individual(id).model(1).type,'ExperimentalData')
            if dosegroup == individual(id).pred.dosegroup;
                
                C_pla(id-i,:,dosegroup) = individual(id).stdout.C.pla(ind,:).*ShowGraphics.SF.conc;
                
            else
                dosegroup = individual(id).pred.dosegroup;
                C_pla(1,:,dosegroup) = individual(id).stdout.C.pla(ind,:).*ShowGraphics.SF.conc ;
                color(j+1)={individual(id).color};
                j=j+1;
                i=id;
            end
        end
        
    end
    for i=1:length(C_pla(1,1,:))
        percentile = prctile(C_pla(:,:,i),[ 5 50 95])';
        
        hold on
        plot(tspan(ind),percentile(:,3), color{i},'LineStyle','--','LineWidth',1.3);
        hold on
        plot(tspan(ind),percentile(:,1),color{i},'LineStyle','--','LineWidth',1.3);
        hold on
        plot(tspan(ind),percentile(:,2),color{i},'LineWidth',1.3);
       
   
    end
    hold on;
    
    
   for id=1:length(individual)
        if strcmp(individual(id).model(1).type,'ExperimentalData')
        
        t     = individual(id).stdout.t* ShowGraphics.SF.time;
        C.pla = individual(id).stdout.C.pla.*ShowGraphics.SF.conc;
        color1= char(individual(id).color);
        ind   = find(isfinite(C.pla)); %plot experimental data
        
        plot(t(ind),C.pla(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
        
        %%pred.errorbar in [nM]
        errorbar_up   =  individual(id).pred.errorbar.up.C.pla;
        errorbar_down =  individual(id).pred.errorbar.down.C.pla;
        errorbar(t(ind),C.pla(ind),errorbar_down(ind),errorbar_up(ind),'ko','LineWidth',3);
        end
   end
    
    set(gca,'Layer','top');
    
    title('plasma');
    set(gca,'box','on')
    lw = 2;   %line width
    fs = 18;  %fontsize
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    
    xlim([xmin, tspan(end)]);
    ymin = ShowGraphics.lowerLimitY;
    ylim([ymin 3*10^1]);
    
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    set(gca,'xticklabel',{[]})
    grid off;
    hold on;
    
     set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('plasma','FontSize', 28);
    
    
    % % % Saving eps with matlab
    filename='Fig7a';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
    
end



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% Tissues
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

if showpercentilestissues
    
    ymin=5*10^-2;
    ymax=3*10^0;
    xmin=-0.25;
    
    %%% Plot percentiles for venous blood concentrations for entire population
    %%% Only applicable for population of size>=10
    
    T = individual(1).stdout.T;  %figNr  = ShowGraphics.percentiles.figNr ;
    
    %%% combine all plasma concentration-time profiles in a single matrix
    %%%
    
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    
    
    C.hea=individual(id).stdout.C.tis(:,T.hea).*ShowGraphics.SF.conc;
    C.mus=individual(id).stdout.C.tis(:,T.mus).*ShowGraphics.SF.conc;
    C.kid=individual(id).stdout.C.tis(:,T.kid).*ShowGraphics.SF.conc;
    C.lun=individual(id).stdout.C.tis(:,T.lun).*ShowGraphics.SF.conc;
    C.adi=individual(id).stdout.C.tis(:,T.adi).*ShowGraphics.SF.conc;
    C.bon=individual(id).stdout.C.tis(:,T.bon).*ShowGraphics.SF.conc;
    C.ski=individual(id).stdout.C.tis(:,T.ski).*ShowGraphics.SF.conc;
    C.spl=individual(id).stdout.C.tis(:,T.spl).*ShowGraphics.SF.conc;
    C.liv=individual(id).stdout.C.tis(:,T.liv).*ShowGraphics.SF.conc;
    C.gut=individual(id).stdout.C.tis(:,T.gut).*ShowGraphics.SF.conc;
    C.tum=individual(id).stdout.C.tis(:,T.tum).*ShowGraphics.SF.conc;
    
    
    
    
    i=0;j=1;
    dosegroup= individual(1).pred.dosegroup;
    color(1)={individual(1).color};
    for id=1: length(individual)
        if ~strcmp(individual(id).model(1).type,'ExperimentalData')
            if dosegroup == individual(id).pred.dosegroup;
                
                C.hea=individual(id).stdout.C.tis(:,T.hea).*ShowGraphics.SF.conc;
                C.mus=individual(id).stdout.C.tis(:,T.mus).*ShowGraphics.SF.conc;
                C.kid=individual(id).stdout.C.tis(:,T.kid).*ShowGraphics.SF.conc;
                C.lun=individual(id).stdout.C.tis(:,T.lun).*ShowGraphics.SF.conc;
                C.adi=individual(id).stdout.C.tis(:,T.adi).*ShowGraphics.SF.conc;
                C.bon=individual(id).stdout.C.tis(:,T.bon).*ShowGraphics.SF.conc;
                C.ski=individual(id).stdout.C.tis(:,T.ski).*ShowGraphics.SF.conc;
                C.spl=individual(id).stdout.C.tis(:,T.spl).*ShowGraphics.SF.conc;
                C.liv=individual(id).stdout.C.tis(:,T.liv).*ShowGraphics.SF.conc;
                C.gut=individual(id).stdout.C.tis(:,T.gut).*ShowGraphics.SF.conc;
                C.tum=individual(id).stdout.C.tis(:,T.tum).*ShowGraphics.SF.conc;
                
                C_hea(id-i,:,dosegroup) = C.hea(ind,:);
                C_mus(id-i,:,dosegroup) = C.mus(ind,:);
                C_kid(id-i,:,dosegroup) = C.kid(ind,:);
                C_lun(id-i,:,dosegroup) = C.lun(ind,:);
                C_adi(id-i,:,dosegroup) = C.adi(ind,:);
                C_bon(id-i,:,dosegroup) = C.bon(ind,:);
                C_ski(id-i,:,dosegroup) = C.ski(ind,:);
                C_spl(id-i,:,dosegroup) = C.spl(ind,:);
                C_liv(id-i,:,dosegroup) = C.liv(ind,:);
                C_gut(id-i,:,dosegroup) = C.gut(ind,:);
                C_tum(id-i,:,dosegroup) = C.tum(ind,:);
                
            else
                dosegroup = individual(id).pred.dosegroup;
                
                
                C_hea(1,:,dosegroup) = C.hea(ind,:) ;
                C_mus(1,:,dosegroup) = C.mus(ind,:) ;
                C_kid(1,:,dosegroup) = C.kid(ind,:) ;
                C_lun(1,:,dosegroup) = C.lun(ind,:) ;
                C_adi(1,:,dosegroup) = C.adi(ind,:) ;
                C_bon(1,:,dosegroup) = C.bon(ind,:) ;
                C_ski(1,:,dosegroup) = C.ski(ind,:) ;
                C_gut(1,:,dosegroup) = C.gut(ind,:) ;
                C_spl(1,:,dosegroup) = C.spl(ind,:) ;
                C_liv(1,:,dosegroup) = C.liv(ind,:) ;
                C_tum(1,:,dosegroup) = C.tum(ind,:) ;
                
                color(j+1)= {individual(id).color};
                j=j+1;
                
                i=id;
            end
        end
        
    end
    
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%Heart
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    figure26=figure(26); clf
    
    for i=1:length(C_hea(1,1,:))
        percentile = prctile(C_hea(:,:,i),[5 50 95])';
        
        
        plot(tspan(ind),percentile(:,3), color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,1),color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,2),color{i},'LineWidth',1.3)
        hold on;
        
    end
    
    hold on
    
    for id=1:length(individual)
        if strcmp(individual(id).model(1).type,'ExperimentalData')
        
        t      = individual(id).stdout.t*ShowGraphics.SF.time;
        C.hea  = individual(id).stdout.C.tis(:,T.hea).*ShowGraphics.SF.conc;
        color1 = char(individual(id).color);
        ind    = find(isfinite(C.hea)); %plot experimental data
        plot(t(ind),C.hea(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
        %%%% errorbars in tissues:  errorbars in [nM]
        pred=individual(id).pred;
        errorbar_up   =  pred.errorbar.up.C.tis(:,T.hea);
        errorbar_down =  pred.errorbar.down.C.tis(:,T.hea);
        errorbar(t(ind),C.hea(ind),errorbar_down(ind),errorbar_up(ind),'ko');
        end
    end
    
    set(gca,'Layer','top');
    title('heart');
    
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    
    grid off; xlim([xmin, tspan(end)]); ylim([ymin ymax]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    set(gca,'xticklabel',{[]})
    hold('all')
    
     set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('heart','FontSize', 28);
    
    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig7c';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
    
    hold on
    
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% Muscle
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    
    figure27=figure(27); clf
    
    
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    for i=1:length(C_mus(1,1,:))
        percentile = prctile(C_mus(:,:,i),[5 50 95])';
       
        hold on;
        
        plot(tspan(ind),percentile(:,3), color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,1),color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,2),color{i},'LineWidth',1.3)
        hold on;
        
    end
    
    
    
    for id=1:length(individual)
        if strcmp(individual(id).model(1).type,'ExperimentalData')
        
        t      = individual(id).stdout.t*ShowGraphics.SF.time;
        C.mus  = individual(id).stdout.C.tis(:,T.mus).*ShowGraphics.SF.conc;
        color1 = char(individual(id).color);
        ind    = find(isfinite(C.mus)); %plot experimental data
        plot(t(ind),C.mus(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
        %%%% errorbars in tissues: errorbars in [nM]
        pred=individual(id).pred;
        errorbar_up   =  pred.errorbar.up.C.tis(:,T.mus);
        errorbar_down =  pred.errorbar.down.C.tis(:,T.mus);
        errorbar(t(ind),C.mus(ind),errorbar_down(ind),errorbar_up(ind),'ko');
        end
    end
    
    hold on
    
    set(gca,'Layer','top');
    title('muscle');
    xlim([xmin, tspan(end)]);
    grid off; ylim([ymin ymax]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    set(gca,'xticklabel',{[]})
    box on
    
     set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('muscle','FontSize', 28);
    
    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig7e';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% Kidney
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    figure28=figure(28); clf
    
    
    for i=1:length(C_kid(1,1,:))
        percentile = prctile(C_kid(:,:,i),[5 50 95])';
        hold on;
       
 
        plot(tspan(ind),percentile(:,3), color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,1),color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,2),color{i},'LineWidth',1.3)
        hold on;
        
    end
    
    
    
    for id=1:length(individual)
        if strcmp(individual(id).model(1).type,'ExperimentalData')
        
        t      = individual(id).stdout.t*ShowGraphics.SF.time;
        C.kid  = individual(id).stdout.C.tis(:,T.kid).*ShowGraphics.SF.conc;
        color1 = char(individual(id).color);
        ind = find(isfinite(C.kid)); 
        plot(t(ind),C.kid(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
        %%%% errorbars in tissues: errorbars in [nM]
        pred=individual(id).pred;
        errorbar_up   =  pred.errorbar.up.C.tis(:,T.kid);
        errorbar_down =  pred.errorbar.down.C.tis(:,T.kid);
        errorbar(t(ind),C.kid(ind),errorbar_down(ind),errorbar_up(ind),'ko');
        end
    end
    
    hold on
    
    set(gca,'Layer','top');
    title('kidney');
    xlim([xmin, tspan(end)]);
    grid off; ylim([ymin ymax]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    box on
    set(gca,'xticklabel',{[]},'yticklabel',{[]});
    
     set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('kidney','FontSize', 28);
    
    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig7d';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% Lung
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    figure29=figure(29); clf
    
    
    for i=1:length(C_lun(1,1,:))
        percentile = prctile(C_lun(:,:,i),[5 50 95])';
        hold on;
        
        
        plot(tspan(ind),percentile(:,3), color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,1),color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,2),color{i},'LineWidth',1.3)
        hold on;
        
    end
    
    
    
     for id=1:length(individual)
        if strcmp(individual(id).model(1).type,'ExperimentalData')
        
        t     = individual(id).stdout.t*ShowGraphics.SF.time;
        C.lun = individual(id).stdout.C.tis(:,T.lun).*ShowGraphics.SF.conc;
        color1= char(individual(id).color);
        ind   = find(isfinite(C.lun)); 
        plot(t(ind),C.lun(ind),color1, 'LineStyle', '.', 'MarkerSize',10);
        plot(t(ind),C.lun(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
        %%%% errorbars in tissues:  errorbars in [nM]
        pred=individual(id).pred;
        errorbar_up   =  pred.errorbar.up.C.tis(:,T.lun);
        errorbar_down =  pred.errorbar.down.C.tis(:,T.lun);
        errorbar(t(ind),C.lun(ind),errorbar_down(ind),errorbar_up(ind),'ko');
        end
     end
    
    hold on
    
    set(gca,'Layer','top');
    title('lung');
    xlim([xmin, tspan(end)]);
    ylim([10^-1 3*10^1]);
    grid off;
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    set(gca,'xticklabel',{[]},'yticklabel',{[]});
    box on
    
     set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('lung','FontSize', 28);
    
    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig7b';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
    hold on
    
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% Bone
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    figure31=figure(31); clf
    
    
    for i=1:length(C_bon(1,1,:))
        percentile = prctile(C_bon(:,:,i),[5 50 95])';
        
        
        hold on;
        plot(tspan(ind),percentile(:,3), color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,1),color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,2),color{i},'LineWidth',1.3)
        
        hold on;
        
    end
    
    
    
    for id=1:length(individual)
        if strcmp(individual(id).model(1).type,'ExperimentalData')
        
        t      = individual(id).stdout.t*ShowGraphics.SF.time;
        C.bon  = individual(id).stdout.C.tis(:,T.bon).*ShowGraphics.SF.conc;
        color1 = char(individual(id).color);
        ind    = find(isfinite(C.bon)); 
        plot(t(ind),C.bon(ind),color1, 'LineStyle', '.', 'MarkerSize',10);
        plot(t(ind),C.bon(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
        %%%% errorbars in tissues: errorbars in [nM]
        pred=individual(id).pred;
        errorbar_up   =  pred.errorbar.up.C.tis(:,T.bon);
        errorbar_down =  pred.errorbar.down.C.tis(:,T.bon);
        errorbar(t(ind),C.bon(ind),errorbar_down(ind),errorbar_up(ind),'ko');
        end
    end
    
    hold on
    
    set(gca,'Layer','top');
    title('bone');
    xlim([xmin, tspan(end)]);
    ylim([ymin ymax]);
    grid off;
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    set(gca,'yticklabel',{[]});
    box on
    
     set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('bone','FontSize', 28);
    
    
    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig7j';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
    
    
    hold on
    
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% Skin
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    figure32=figure(32); clf
    
    
    for i=1:length(C_ski(1,1,:))
        percentile = prctile(C_ski(:,:,i),[5 50 95])';
        
        
        hold on;
        
        plot(tspan(ind),percentile(:,3), color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,1),color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,2),color{i},'LineWidth',1.3)
        
        hold on;
        
    end
    
    
    
     for id=1:length(individual)
        if strcmp(individual(id).model(1).type,'ExperimentalData')
        
        t     = individual(id).stdout.t*ShowGraphics.SF.time;
        C.ski = individual(id).stdout.C.tis(:,T.ski).*ShowGraphics.SF.conc;
        color1= char(individual(id).color);
        ind   = find(isfinite(C.ski)); 
        plot(t(ind),C.ski(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
        %%%% errorbars in tissues: errorbars in [nM]
        pred=individual(id).pred;
        errorbar_up   =  pred.errorbar.up.C.tis(:,T.ski);
        errorbar_down =  pred.errorbar.down.C.tis(:,T.ski);
        errorbar(t(ind),C.ski(ind),errorbar_down(ind),errorbar_up(ind),'ko');
        end
     end
    hold on
    
    set(gca,'Layer','top');
    title('skin');
    xlim([xmin, tspan(end)]);
    grid off; ylim([ymin ymax]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    set(gca,'xticklabel',{[]},'yticklabel',{[]});
    box on
    
     set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('skin','FontSize', 28);
    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig7f';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
    
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% Gut
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    figure33=figure(33); clf
    
    
    for i=1:length(C_gut(1,1,:))
        percentile = prctile(C_gut(:,:,i),[5 50 95])';
        
        
        hold on;
        plot(tspan(ind),percentile(:,3), color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,1),color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,2),color{i},'LineWidth',1.3)
        
        
        hold on;
        
    end
    
    
    
     for id=1:length(individual)
        if strcmp(individual(id).model(1).type,'ExperimentalData')
        
        t      = individual(id).stdout.t*ShowGraphics.SF.time;
        C.gut  = individual(id).stdout.C.tis(:,T.gut).*ShowGraphics.SF.conc;
        color1 = char(individual(id).color);
        ind    = find(isfinite(C.gut)); 
        plot(t(ind),C.gut(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
        %%%% errorbars in tissues: errorbars in [nM]
        pred=individual(id).pred;
        errorbar_up   =  pred.errorbar.up.C.tis(:,T.gut);
        errorbar_down =  pred.errorbar.down.C.tis(:,T.gut);
        errorbar(t(ind),C.gut(ind),errorbar_down(ind),errorbar_up(ind),'ko');
        end
     end
    
    hold on
    set(gca,'Layer','top');
    title('gut');
    xlim([xmin, tspan(end)]);
    grid off; ylim([ymin ymax]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    set(gca,'xticklabel',{[]},'yticklabel',{[]});
    box on
    
     set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('gut','FontSize', 28);
    
    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig7h';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
    
    hold on
    
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% Spleen
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    figure34=figure(34); clf
    
    
    for i=1:length(C_spl(1,1,:))
        percentile = prctile(C_spl(:,:,i),[5 50 95])';
        
        
        hold on;
        plot(tspan(ind),percentile(:,3), color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,1),color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,2),color{i},'LineWidth',1.3)
        
        
        hold on;
        
    end
    
    
    
     for id=1:length(individual)
        if strcmp(individual(id).model(1).type,'ExperimentalData')
        
        t      = individual(id).stdout.t*ShowGraphics.SF.time;
        C.spl  = individual(id).stdout.C.tis(:,T.spl).*ShowGraphics.SF.conc;
        color1 = char(individual(id).color);
        ind    = find(isfinite(C.spl)); 
        plot(t(ind),C.spl(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
        %%%% errorbars in tissues: errorbars in [nM]
        pred=individual(id).pred;
        errorbar_up   =  pred.errorbar.up.C.tis(:,T.spl);
        errorbar_down =  pred.errorbar.down.C.tis(:,T.spl);
        errorbar(t(ind),C.spl(ind),errorbar_down(ind),errorbar_up(ind),'ko');
        end
     end
    
    hold on
    
    set(gca,'Layer','top');
    title('spleen');
    xlim([xmin, tspan(end)]);
    grid off; ylim([ymin ymax]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    box on
    
    
     set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('spleen','FontSize', 28);
    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig7i';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
    hold on
    
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% Liver
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    figure35=figure(35); clf
    
    for i=1:length(C_liv(1,1,:))
        percentile = prctile(C_liv(:,:,i),[5 50 95])';
        
        
        hold on;
        plot(tspan(ind),percentile(:,3), color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,1),color{i},'LineStyle','--','LineWidth',1.3)
        hold on
        plot(tspan(ind),percentile(:,2),color{i},'LineWidth',1.3)
        
        hold on;
        
    end
    
    
     
     for id=1:length(individual)
        if strcmp(individual(id).model(1).type,'ExperimentalData')
        t     = individual(id).stdout.t*ShowGraphics.SF.time;
        C.liv = individual(id).stdout.C.tis(:,T.liv).*ShowGraphics.SF.conc;
        color1= char(individual(id).color);
        ind   = find(isfinite(C.liv)); %plot experimental data
        plot(t(ind),C.liv(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
        
        %%%% errorbars in tissues:  errorbars in [nM]
        pred          =individual(id).pred;
        errorbar_up   =  pred.errorbar.up.C.tis(:,T.liv);
        errorbar_down =  pred.errorbar.down.C.tis(:,T.liv);
        errorbar(t(ind),C.liv(ind),errorbar_down(ind),errorbar_up(ind),'ko');
        end
     end
    
    hold on
    
    set(gca,'Layer','top');
    title('liver');
    xlim([xmin, tspan(end)]);
    grid off; ylim([ymin ymax]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    lw = 2;   %line width
    fs = 18;  %fontsize
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    set(gca,'xticklabel',{[]});
    
    box on
    
     set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('liver','FontSize', 28);
    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig7g';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
    
    
end

