%%% Version: March 16th, 2015
%%%
%%% This function generate random numbers suitable to include
%%% random perturbations in some model file.
%%%
%%% For references and citation, please see MAIN script.
%%%
%%% Copyright (C) 2015, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function individual = GenericPBPKmodel_randomEffects(individual,randomEffects)


%%% Check, whether there are random effects to be considered
%%%
check = GenericPBPKmodel_checkForRandomEffectsAndReport(randomEffects);

if strcmp(check,'no random effects considered')
    return;
end;


%%% Go individual by individual ...
%%%
for id=1:length(individual)
    
    %listOfCompounds = individual(id).drug.listOfCompounds;
    listOfDrugs  = individual(id).drug.listOfDrugs;
    for m = 1:length(listOfDrugs)
        
        compound = listOfDrugs{m};
        drug     = individual(id).drug.(compound);
        
        
        
        species  = individual(id).species;
        S       = individual(id).species.T;
        dosing  = individual(id).dosing;
        
        
        S.organs   = [S.lun S.adi S.hea S.kid S.mus S.bon S.ski S.gut S.spl S.liv      ];
        
        
        switch drug.class
            
            case 'sMD';
                facX = randomEffects.sMD.facX;
                
                CLint_hep = randomEffect_on_CLint_hep(drug,facX.CLint_hep);
                K_tis_up  = randomEffect_on_K_tis_up(drug,facX.K_tis_up);
                
                individual(id).drug.(compound).CLint.hep.tis = CLint_hep;
                individual(id).drug.(compound).K.tis_up      = K_tis_up;
                
            case 'mAb';
                
                
                facX = randomEffects.mAb.facX;
                
                hct     = randomEffect_on_hct(drug,facX.hct,species);
                V_blood = randomEffect_on_V_blood(drug,facX.V_blood,species);
                Dose    = randomEffect_on_Dose(drug,facX.Dose,species, S, dosing);
                
                
                
                individual(id).species.hct     = hct;
                individual(id).species.V.blood = V_blood;
                individual(id).dosing(1).dose  = Dose;
                
                
            case 'pAb';
                
                
                facX = randomEffects.mAb.facX;
                
                hct     = randomEffect_on_hct(drug,facX.hct,species);
                V_blood = randomEffect_on_V_blood(drug,facX.V_blood,species);
                Dose    = randomEffect_on_Dose(drug,facX.Dose,species, S, dosing);
                
                
                individual(id).species.hct     = hct;
                individual(id).species.V.blood = V_blood;
                individual(id).dosing(1).dose  = Dose;
                
                
                
        end;
    end;
    
end;


end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function check = GenericPBPKmodel_checkForRandomEffectsAndReport(randomEffects)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% Check, whether there are random effects to be considered
%%%
%facX.CLint_hep = randomEffects.sMD.facX.CLint_hep;
%facX.K_tis_up  = randomEffects.sMD.facX.K_tis_up;

facX.hct           = randomEffects.mAb.facX.hct;
facX.V_blood       = randomEffects.mAb.facX.V_blood;
facX.Dose          = randomEffects.mAb.facX.Dose;



if (facX.hct>0) || (facX.V_blood>0)   || (facX.Dose>0)
    
    fprintf('\n'); fprintf(2,'       >>> Random effect(s) on');
    if facX.hct>0
        fprintf(2,' hct,');
    end;
    if facX.V_blood>0
        fprintf(2,' V_blood,');
    end;
    if facX.Dose>0
        fprintf(2,' Dose,');
    end;
    
    
    fprintf(2,'\b considered! <<<');
    check = 'random effects considered';
    
elseif (facX.hct<0) || (facX.V_blood<0)    || (facX.Dose<0)
    
    message = 'randomEffects.facX negative for some parameter!';
    GenericPBPKmodel_ReportErrorMessage(message);
    
else
    
    check = 'no random effects considered';
    
end;

end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function randfac = randomfactor(howmany,maxfac)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

rand1 = rand(howmany);

rand2 = rand(howmany);

randfac = (rand1<0.5)./(1+(maxfac-1)*rand2) + (rand1>=0.5).*(1+(maxfac-1)*rand2);


end




%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function hct = randomEffect_on_hct(drug,fac,species)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

if fac==0
    hct = species.hct;
else
    hct = randomfactor(1,fac) * species.hct;
    
end;



end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function Dose = randomEffect_on_Dose(drug,fac,species,S, dosing)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

if fac==0
    
    Dose = dosing(1).dose ;
else
    
    
    Dose =  randomfactor(size(dosing(1).dose),fac).*  dosing(1).dose;
    
end;



end




%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function V_blood = randomEffect_on_V_blood(drug,fac,species)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


if fac==0
    V_blood = species.V.blood;
else
    V_blood = randomfactor(1,fac) *  species.V.blood;
end;

end




%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

