%%% Version: October 31st, 2014
%%%
%%% This function specifies all drug related data
%%%
%%% For references and citation, please see MAIN script.
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function drug = GenericPBPKmodel_drugDatabase(compound)


%%% ========================================================================================================
%%% == DATA FIELDS, UNITS, and PARAMETER DESCRIPTION
%%%

%%% physico-chemical drug properties (with default values)
%%%
data.drug.name                = '';       info.name         = 'name of the drug';
data.drug.info                = [];
data.drug.class               = '';       info.class        = 'small molecule drug (sMD), monoclonal antibody (mAb)';
data.drug.subclass            = '';       info.subclass     = 'mod./strong base, weak base, neutral, acid';
data.drug.MW                  = NaN;      info.MW           = 'molecular weight in [Da]=[g/mol]';

%%% data only related to small molecule drugs
%%%
data.drug.pKa                 = NaN;      info.pKa          = 'pKa value';
data.drug.logPow              = NaN;      info.logPow       = 'octanol-water partition coefficient (log value)';
data.drug.logPvow             = NaN;      info.logPvow      = 'vegtable oil-water partition coefficient (log value); estimated from logPov if not speciefied';

%%% species-related parameters (with default values)
data.drugspecies.fuP              = NaN;  info.fuP          = 'fraction unbound in plasma (species specific)';
data.drugspecies.BP               = NaN;  info.BP           = 'blood-to-plasma partition coefficient (species specific)';
data.drugspecies.K_ery_up         = NaN;  info.K_ery_up     = 'erythrocyte-to-unbound plasma partition coefficient (species specific)';
data.drugspecies.CLblood_kgBW.hep = 0;    info.CLblood_kgBW = 'tissue specific blood clearance in [L/min] per kg BW (species specific)';
data.drugspecies.CLintK_kgOW.liv  = 0;    info.CLintK_kgOW  = 'tissue specific intrinsic clearance * partition coefficient in [L/min] per kg OW (species specific)';
data.drugspecies.CLblood_kgBW.ren = 0;
data.drugspecies.E.feces          = 0;    info.E.feces      = 'fraction excreted in feces (species specific)';
data.drugspecies.E.gut            = 0;    info.E.gut        = 'fraction extracted in the gut (species specific)';
data.drugspecies.lambda_po        = 0;    info.lambda_po    = 'first order absorption rate constant in [1/min] (species specific)';

%%% data only related to monoclonal antibodies
%%%



% combine both structures for initialization
mammals.type = {'rat','mouse','human'};
for s = 1:length(mammals.type)
    species = mammals.type{s};
    data.drug.(species) = data.drugspecies;
end;

%%% Some drug data, like the blood-to-plasma ratio BP or the hepatic
%%% blood clearance CLblood_kgBW.hep, depend on the species' subtype
%%% used to measure them. For example, BP depends on hct, while
%%% CLblood_kgBW.hep depends on the liver blood flow. The species'
%%% reference data below are used to convert such drug data into data
%%% that are assumed to depend only on the species, but not on the
%%% particular species' subtype. This conversion is done at the very
%%% end of the file.

refSubtype.mouse.BW     = 0.025;
refSubtype.mouse.hct    = 0.45;
refSubtype.mouse.OW_liv = 1.3725e-3;
refSubtype.mouse.Q_liv  = 2.8009e-3;
refSubtype.rat.BW       = 0.250;
refSubtype.rat.hct      = 0.45;
refSubtype.rat.OW_liv   = 9.1500e-3;
refSubtype.rat.Q_liv    = 1.4457e-2;
refSubtype.human.BW     = 73;
refSubtype.human.hct    = 0.45;
refSubtype.human.OW_liv = 1.800;
refSubtype.human.Q_liv  = 1.6575;


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% DRUG TYPE: moderate to strong bases
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% =======================================================================
db.Amitriptyline            = data.drug;
db.Amitriptyline.name       = 'Amitriptyline';
db.Amitriptyline.class      = 'sMD';
db.Amitriptyline.subclass   = 'mod. to strong base';
db.Amitriptyline.MW         = 277.40;
db.Amitriptyline.pKa        = 9.40;
db.Amitriptyline.logPow     = 4.90;

db.Amitriptyline.rat.fuP    = 0.056;
db.Amitriptyline.rat.BP     = 0.86;

db.Amitriptyline.human.fuP  = 0.056;
db.Amitriptyline.human.BP   = 0.86;
db.Amitriptyline.human.CLblood_kgBW.hep = 12/1000; % Obach99
db.Amitriptyline.human.lambda_po    = (log(2)/1.36)/60; %Kuss1985


%%% =======================================================================
db.Caffeine                 = data.drug;
db.Caffeine.name            = 'Caffeine';
db.Caffeine.class           = 'sMD';
db.Caffeine.subclass        = 'mod. to strong base';
db.Caffeine.MW              = 194.19;
db.Caffeine.pKa             = 10.4;
db.Caffeine.logPow          = -0.091;

db.Caffeine.human.fuP       = 0.7;
db.Caffeine.human.BP        = 1.04;
db.Caffeine.human.CLblood_kgBW.hep = 1.4/1000;


%%% =======================================================================
db.Desiparmine              = data.drug;
db.Desipramine.name         = 'Desipramine';
db.Desipramine.class        = 'sMD';
db.Desipramine.subclass     = 'mod. to strong base';
db.Desipramine.MW           = 266.38;
db.Desipramine.pKa          = 10.32;
db.Desipramine.logPow       = 4.9;

db.Desipramine.human.fuP    = 0.19;
db.Desipramine.human.BP     = 0.96;
db.Desipramine.human.CLblood_kgBW.hep = 12/1000; % Obach99


%%% =======================================================================
db.Diltiazem                = data.drug;
db.Diltiazem.name           = 'Diltiazem';
db.Diltiazem.class          = 'sMD';
db.Diltiazem.subclass       = 'mod. to strong base';
db.Diltiazem.MW             = 414.52;
db.Diltiazem.pKa            = 7.7;
db.Diltiazem.logPow         = 2.67;
db.Diltiazem.human.fuP      = 0.2;
db.Diltiazem.human.BP       = 1.03;
db.Diltiazem.human.CLblood_kgBW.hep = 12/1000; % Obach99

%%% =======================================================================

db.Diphenhydramine          = data.drug;
db.Diphenhydramine.name     = 'Diphenhydramine';
db.Diphenhydramine.class    = 'sMD';
db.Diphenhydramine.subclass = 'mod. to strong base';
db.Diphenhydramine.MW       = 255.35;
db.Diphenhydramine.pKa      = 8.98;
db.Diphenhydramine.logPow   = 3.31;

db.Diphenhydramine.human.fuP = 0.089;
db.Diphenhydramine.human.BP = 0.8;
db.Diphenhydramine.human.CLblood_kgBW.hep = 9.5/1000; % Obach99


%%% =======================================================================
db.Imipramine               = data.drug;
db.Imipramine.name          = 'Imipramine';
db.Imipramine.class         = 'sMD';
db.Imipramine.subclass      = 'mod. to strong base';
db.Imipramine.MW            = 280.41;
db.Imipramine.pKa           = 9.5;
db.Imipramine.logPow        = 4.80;
db.Imipramine.logPvow       = 4.00;

db.Imipramine.rat.fuP       = 0.24;
db.Imipramine.rat.BP        = 1.67;

db.Imipramine.human.fuP     = 0.13; % RR07
db.Imipramine.human.BP      = 1.07; % RR07
db.Imipramine.human.CLblood_kgBW.hep = 12/1000; % Obach99, ml/min/kg -> l/min/kg


%%% =======================================================================
db.Lidocaine                = data.drug;
db.Lidocaine.name           = 'Lidocaine';
db.Lidocaine.class          = 'sMD';
db.Lidocaine.subclass       = 'mod. to strong base';
db.Lidocaine.MW             = 234.34;
db.Lidocaine.pKa            = 8.01;
db.Lidocaine.logPow         = 2.26;
db.Lidocaine.logPvow        = 1.27;

db.Lidocaine.rat.fuP        = 0.38;
db.Lidocaine.rat.BP         = 1.27;

db.Lidocaine.human.fuP      = 0.296;
db.Lidocaine.human.BP       = 0.84;
db.Lidocaine.human.CLblood_kgBW.hep = 15/1000; % ml/min/kg -> l/min/kg
db.Lidocaine.human.lambda_po        = 0.018; % 1/min Boyes etal (1970), J Pharmacol Exp Ther 174:1?8 (dogs)


%%% =======================================================================
db.Metoprolol               = data.drug;
db.Metoprolol.name          = 'Metoprolol';
db.Metoprolol.class         = 'sMD';
db.Metoprolol.subclass      = 'mod. to strong base';
db.Metoprolol.MW            = 267.36;
db.Metoprolol.pKa           = 9.7;
db.Metoprolol.logPow        = 2.15;

db.Metoprolol.human.fuP     = 0.9;
db.Metoprolol.human.BP      = 1.14;
db.Metoprolol.human.CLblood_kgBW.hep = 12.15/1000;


%%% =======================================================================
db.Pindolol                 = data.drug;
db.Pindolol.name            = 'Pindolol';
db.Pindolol.class           = 'sMD';
db.Pindolol.subclass        = 'mod. to strong base';
db.Pindolol.MW              = 248.32;
db.Pindolol.pKa             = 8.8;
db.Pindolol.logPow          = 1.75;

db.Pindolol.human.fuP       = 0.41;
db.Pindolol.human.BP        = 0.81;
db.Pindolol.human.CLblood_kgBW.hep = 4.2/1000;


%%% =======================================================================
db.Sildefanile              = data.drug;
db.Sildefanil.name          = 'Sildefanil';
db.Sildefanil.class         = 'sMD';
db.Sildefanil.subclass      = 'mod. to strong base';
db.Sildefanil.MW            = 474.58;
db.Sildefanil.pKa           = 7.6;
db.Sildefanil.logPow        = 2.75;

db.Sildefanil.human.fuP     = 0.04;
db.Sildefanil.human.BP      = 0.62;
db.Sildefanil.human.CLblood_kgBW.hep = 6/1000;


%%% =======================================================================
db.Theophylline             = data.drug;
db.Theophylline.name        = 'Theophylline';
db.Theophylline.class       = 'sMD';
db.Theophylline.subclass    = 'mod. to strong base';
db.Theophylline.MW          = 180.16;
db.Theophylline.pKa         = 8.71;
db.Theophylline.logPow      = 0.26;

db.Theophylline.human.fuP   = 0.6;
db.Theophylline.human.BP    = 0.83;
db.Theophylline.human.CLblood_kgBW.hep = 0.65/1000;


%%% =======================================================================
db.Verapamil                = data.drug;
db.Verapamil.name           = 'Verapamil';
db.Verapamil.class          = 'sMD';
db.Verapamil.subclass       = 'mod. to strong base';
db.Verapamil.MW             = 454.60;
db.Verapamil.pKa            = 8.5;
db.Verapamil.logPow         = 3.79;
db.Verapamil.logPvow        = 2.88;

db.Verapamil.rat.fuP        = 0.05;
db.Verapamil.rat.BP         = 0.85;

db.Verapamil.human.fuP      = 0.082; % RR07
db.Verapamil.human.BP       = 0.84; % RR07
db.Verapamil.human.CLblood_kgBW.hep = 19/1000; % Obach99
db.Verapamil.human.E.feces          = 0.22; % 12-33 %



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% DRUG TYPE: weak bases
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% =======================================================================
db.Alprazolam               = data.drug;
db.Alprazolam.name          = 'Alprazolam';
db.Alprazolam.class         = 'sMD';
db.Alprazolam.subclass      = 'weak base';
db.Alprazolam.MW            = 416.5172;
db.Alprazolam.pKa           = 2.4;
db.Alprazolam.logPow        = 2.3;

db.Alprazolam.rat.fuP       = 0.35;

db.Alprazolam.human.fuP     = 0.32; % Obach99
db.Alprazolam.human.BP      = 0.78; % Obach99
db.Alprazolam.human.CLblood_kgBW.hep = 0.76/1000; % Obach99
db.Alprazolam.human.E.feces = 0.85; % 80-90 %


%%% =======================================================================
db.Diazepam                 = data.drug;
db.Diazepam.name            = 'Diazepam';
db.Diazepam.class           = 'sMD';
db.Diazepam.subclass        = 'weak base';
db.Diazepam.MW              = 284.74;
db.Diazepam.pKa             = 3.4;
db.Diazepam.logPow          = 2.84;

db.Diazepam.rat.fuP         = 0.14;
db.Diazepam.rat.BP          = 1.04;

db.Diazepam.human.fuP       = 0.013; % Obach99
db.Diazepam.human.BP        = 0.71; % Obach99
db.Diazepam.human.CLblood_kgBW.hep = 0.6/1000; % Obach99
db.Diazepam.human.E.feces          = 0.97; % 94-100 %


%%% =======================================================================
db.Flunitrazepam            = data.drug;
db.Flunitrazepam.name       = 'Flunitrazepam';
db.Flunitrazepam.class      = 'sMD';
db.Flunitrazepam.subclass   = 'weak base';
db.Flunitrazepam.MW         = 313.2832;
db.Flunitrazepam.pKa        = 1.8;
db.Flunitrazepam.logPow     = 2.1;

db.Flunitrazepam.rat.fuP    = 0.25;

db.Flunitrazepam.human.fuP  = 0.25;
db.Flunitrazepam.human.BP   = 1.2;
db.Flunitrazepam.human.CLblood_kgBW.hep = 9.8/1000/73; % ml/min/(73kg BW) -> l/min/kg;
db.Flunitrazepam.human.E.feces         = 0.85;


%%% =======================================================================
db.Midazolam                = data.drug;
db.Midazolam.name           = 'Midazolam';
db.Midazolam.class          = 'sMD';
db.Midazolam.subclass       = 'weak base';
db.Midazolam.MW             = 325.767;
db.Midazolam.pKa            = 6.0;
db.Midazolam.logPow         = 3.15;

db.Midazolam.rat.fuP        = 0.059;

db.Midazolam.human.fuP      = 0.05; % Obach99
db.Midazolam.human.BP       = 0.567; % Obach99 (changed from 0.53 to 0.567)
db.Midazolam.human.CLblood_kgBW.hep = 8.7/1000; % Obach99
db.Midazolam.human.lambda_po        = 0.16; % 1/min
db.Midazolam.human.E.gut     = 0.444;
db.Midazolam.human.E.feces   = 0.4; % 28.2; 40-50


%%% =======================================================================
db.Triazolam                = data.drug;
db.Triazolam.name           = 'Triazolam';
db.Triazolam.class          = 'sMD';
db.Triazolam.subclass       = 'weak base';
db.Triazolam.MW             = 343.21;
db.Triazolam.pKa            = 2.0;
db.Triazolam.logPow         = 2.4;

db.Triazolam.rat.fuP        = 0.28;

db.Triazolam.human.fuP      = 0.1; % Obach99
db.Triazolam.human.BP       = 0.62; % Obach99
db.Triazolam.human.CLblood_kgBW.hep = 4.7/1000; % human value



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% DRUG TYPE: neutrals
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% =======================================================================
db.Digoxin                  = data.drug;
db.Digoxin.name             = 'Digoxin';
db.Digoxin.class            = 'sMD';
db.Digoxin.subclass         = 'neutral';
db.Digoxin.MW               = 780.9385;
db.Digoxin.logPow           = 1.23;

db.Digoxin.rat.fuP          = 0.61;

db.Digoxin.human.fuP        = 0.74; % RR07
db.Digoxin.human.BP         = 1;
db.Digoxin.human.CLblood_kgBW.hep = 12.2/60/73; % l/hr/(73 kg BW) -> l/min/kg


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% DRUG TYPE: acids
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% =======================================================================
db.Amobarbital              = data.drug;
db.Amobarbital.name         = 'Amobarbital';
db.Amobarbital.class        = 'sMD';
db.Amobarbital.subclass     = 'acid';
db.Amobarbital.MW           = 226.27;
db.Amobarbital.pKa          = 7.9;
db.Amobarbital.logPow       = 1.89;

db.Amobarbital.human.fuP    = 0.39;
db.Amobarbital.human.BP     = 1.5;
db.Amobarbital.human.CLblood_kgBW.hep = 0.35/1000;


%%% =======================================================================
db.Diclofenac               = data.drug;
db.Diclofenac.name          = 'Diclofenac';
db.Diclofenac.class         = 'sMD';
db.Diclofenac.subclass      = 'acid';
db.Diclofenac.MW            = 296.15;
db.Diclofenac.pKa           = 4.15;
db.Diclofenac.logPow        = 3.9;

db.Diclofenac.human.fuP     = 0.005;
db.Diclofenac.human.BP      = 0.567;
db.Diclofenac.human.CLblood_kgBW.hep = 7.6/1000;


%%% =======================================================================
db.Dideoxyinosine           = data.drug;
db.Dideoxyinosine.name      = 'Dideoxyinosine';
db.Dideoxyinosine.class     = 'sMD';
db.Dideoxyinosine.subclass  = 'acid';
db.Dideoxyinosine.MW        = 236.2273;
db.Dideoxyinosine.pKa       = 9.1;
db.Dideoxyinosine.logPow    = -1.2;

db.Dideoxyinosine.rat.fuP   = 0.97;

db.Dideoxyinosine.human.fuP = 0.96; % RR07


%%% =======================================================================
db.Hexobarbital             = data.drug;
db.Hexobarbital.name        = 'Hexobarbital';
db.Hexobarbital.class       = 'sMD';
db.Hexobarbital.subclass    = 'acid';
db.Hexobarbital.MW          = 236.27;
db.Hexobarbital.pKa         = 8.29;
db.Hexobarbital.logPow      = 1.74;

db.Hexobarbital.human.fuP   = 0.53;
db.Hexobarbital.human.BP    = 1;
db.Hexobarbital.human.CLblood_kgBW.hep = 3.6/1000;


%%% =======================================================================
db.Ibuprofen                = data.drug;
db.Ibuprofen.name           = 'Ibuprofen';
db.Ibuprofen.class          = 'sMD';
db.Ibuprofen.subclass       = 'acid';
db.Ibuprofen.MW             = 206.28;
db.Ibuprofen.pKa            = 4.7;
db.Ibuprofen.logPow         = 4.06;

db.Ibuprofen.human.fuP      = 0.01;
db.Ibuprofen.human.BP       = 0.567;
db.Ibuprofen.human.CLblood_kgBW.hep = 1.5/1000;


%%% =======================================================================
db.Methohexital             = data.drug;
db.Methohexital.name        = 'Methohexital';
db.Methohexital.class       = 'sMD';
db.Methohexital.subclass    = 'acid';
db.Methohexital.MW          = 262.3;
db.Methohexital.pKa         = 8.51;
db.Methohexital.logPow      = 1.72;

db.Methohexital.human.fuP   = 0.27;
db.Methohexital.human.BP    = 0.7;
db.Methohexital.human.CLblood_kgBW.hep = 16/1000;


%%% =======================================================================
db.Penicillin               = data.drug;
db.Penicillin.name          = 'Penicillin';
db.Penicillin.class         = 'sMD';
db.Penicillin.subclass      = 'acid';
db.Penicillin.MW            = 350.39;
db.Penicillin.pKa           = 2.7;
db.Penicillin.logPow        = 1.6;

db.Penicillin.rat.fuP       = 0.15;

db.Penicillin.human.fuP     = 0.2;


%%% =======================================================================
db.SalicyclicAcid           = data.drug;
db.SalicyclicAcid.name      = 'SalicyclicAcid';
db.SalicyclicAcid.class     = 'sMD';
db.SalicyclicAcid.subclass  = 'acid';
db.SalicyclicAcid.MW        = 138.1207;
db.SalicyclicAcid.pKa       = 3.0;
db.SalicyclicAcid.logPow    = 2.0;

db.SalicyclicAcid.rat.fuP   = 0.137;


%%% =======================================================================
db.Thiopental               = data.drug;
db.Thiopental.name          = 'Thiopental';
db.Thiopental.class         = 'sMD';
db.Thiopental.subclass      = 'acid';
db.Thiopental.MW            = 242.338;
db.Thiopental.pKa           = 7.5;
db.Thiopental.logPow        = 2.9;

db.Thiopental.rat.fuP       = 0.13;

db.Thiopental.human.fuP     = 0.18; % RR07
db.Thiopental.human.BP      = 0.88; % unknown source
db.Thiopental.human.CLblood_kgBW.hep = 2.02/1000; % 1.9-5.4 ml/min/kg -> l/min/kg


%%% =======================================================================
db.Tolbutamide              = data.drug;
db.Tolbutamide.name         = 'Tolbutamide';
db.Tolbutamide.class        = 'sMD';
db.Tolbutamide.subclass     = 'acid';
db.Tolbutamide.MW           = 270.35;
db.Tolbutamide.pKa          = 5.3;
db.Tolbutamide.logPow       = 2.39;

db.Tolbutamide.rat.fuP      = 0.27;

db.Tolbutamide.human.fuP    = 0.04; % Obach99
db.Tolbutamide.human.BP     = 0.567; % Obach99
db.Tolbutamide.human.CLblood_kgBW.hep = 0.36/1000; % Obach99
db.Tolbutamide.human.E.feces          = 0.95;


%%% =======================================================================
db.Valproate                = data.drug;
db.Valproate.name           = 'Valproate';
db.Valproate.class          = 'sMD';
db.Valproate.subclass       = 'acid';
db.Valproate.MW             = 144.2114;
db.Valproate.pKa            = 4.6;
db.Valproate.logPow         = 2.76;

db.Valproate.rat.fuP        = 0.37;

db.Valproate.human.fuP      = 0.099; % RR07
db.Valproate.human.BP       = 0.567; % E:P -> B:P
db.Valproate.human.CLblood_kgBW.hep = 0.11/1000; % 0.09-0.46 ml/min/kg -> l/min/kg


%%% =======================================================================
db.Warfarin                 = data.drug;
db.Warfarin.name            = 'Warfarin';
db.Warfarin.class           = 'sMD';
db.Warfarin.subclass        = 'acid';
db.Warfarin.MW              = 308.33;
db.Warfarin.pKa             = 5.08;
db.Warfarin.logPow          = 3;



db.Warfarin.human.fuP       = 0.01;
db.Warfarin.human.BP        = 0.567;
db.Warfarin.human.CLblood_kgBW.hep = 0.081/1000;


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% DRUG TYPE: zwitterions
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% =======================================================================
db.Ofloxacin                = data.drug;
db.Ofloxacin.name           = 'Ofloxacin';
db.Ofloxacin.class          = 'sMD';
db.Ofloxacin.subclass       = 'AB';
db.Ofloxacin.pKa            = [6.1 8.2];
db.Ofloxacin.logPow         = -0.4;

db.Ofloxacin.rat.fuP        = 0.77;
db.Ofloxacin.rat.BP         = 0.92;


%%% =======================================================================
db.Tetracycline             = data.drug;
db.Tetracycline.name        = 'Tetracycline';
db.Tetracycline.class       = 'sMD';
db.Tetracycline.subclass    = 'AB';
db.Tetracycline.pKa         = [6.1 8.2];
db.Tetracycline.logPow      = -0.4;

db.Tetracycline.rat.fuP     = 0.77;
db.Tetracycline.rat.BP      = 0.92;


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% DRUG TYPE: bi-bases
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% =======================================================================
db.Chlorpromazine           = data.drug;
db.Chlorpromazine.name      = 'Chlorpromazine';
db.Chlorpromazine.class     = 'sMD';
db.Chlorpromazine.subclass  = 'BB';
db.Chlorpromazine.MW        = 318.86;
db.Chlorpromazine.pKa       = [6.4 9.7];
db.Chlorpromazine.logPow    = 0.42;

db.Chlorpromazine.human.fuP = 0.036;
db.Chlorpromazine.human.BP  = 1.17;
db.Chlorpromazine.human.CLblood_kgBW.hep = NaN; % to be transformed: 11/1000; % Obach99


%%% =======================================================================
db.Sulpiride                = data.drug;
db.Sulpiride.name           = 'Sulpiride';
db.Sulpiride.class          = 'sMD';
db.Sulpiride.subclass       = 'BB';
db.Sulpiride.MW             = 341.43;
db.Sulpiride.pKa            = [9.01 10.19];
db.Sulpiride.logPow         = 0.57;

db.Sulpiride.human.fuP      = 1;
db.Sulpiride.human.BP       = 1;
db.Sulpiride.human.CLblood_kgBW.hep = NaN; % to be transformed: 1.5/1000;


%%% =======================================================================
db.Tiapamil                 = data.drug;
db.Tiapamil.name            = 'Tiapamil';
db.Tiapamil.class           = 'sMD';
db.Tiapamil.subclass        = 'BB';
db.Tiapamil.pKa             = [7 9];
db.Tiapamil.logPow          = 2.6;

db.Tiapamil.human.fuP       = 0.33;
db.Tiapamil.human.BP        = 1;


%%% =======================================================================
db.Timolol                  = data.drug;
db.Timolol.name             = 'Timolol';
db.Timolol.class            = 'sMD';
db.Timolol.subclass         = 'BB';
db.Timolol.MW               = 316.42;
db.Timolol.pKa              = [8.8 9.21];
db.Timolol.logPow           = 1.91;

db.Timolol.human.fuP        = 0.4;
db.Timolol.human.BP         = 0.81;
db.Timolol.human.CLblood_kgBW.hep = NaN; % to be transformed: 9.17/1000;


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% DRUG TYPE: virtuelle drugs for testing purposes
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% =======================================================================
db.drugA                    = data.drug;
db.drugA.name               = 'drugA'; % similar to warfarin
db.drugA.class              = 'sMD'; % similar to warfarin
db.drugA.subclass           = 'acid';
db.drugA.MW                 = 308.33;
db.drugA.pKa                = 5.08;
db.drugA.logPow             = 3;

db.drugA.rat.fuP            = 0.38;
db.drugA.rat.BP             = 1.27;
db.drugA.rat.CLblood_kgBW.hep = 0.1/1000;
db.drugA.rat.lambda_po      = 0.01;

db.drugA.mouse.fuP            = 0.21;
db.drugA.mouse.BP             = 1.00;
db.drugA.mouse.CLblood_kgBW.hep = 0.09/1000;
db.drugA.mouse.lambda_po      = 0.015;

db.drugA.human.fuP          = 0.01;
db.drugA.human.BP           = 0.567;
db.drugA.human.CLblood_kgBW.hep = 0.081/1000;
db.drugA.human.lambda_po    = 0.01;


%%% =======================================================================
db.drugB                    = data.drug;
db.drugB.name               = 'drugB'; % similar to Lidocaine
db.drugB.class              = 'sMD';
db.drugB.subclass           = 'mod. to strong base';
db.drugB.MW                 = 234.34;
db.drugB.pKa                = 8.01;
db.drugB.logPow             = 2.26;
db.drugB.logPvow            = 1.27;

db.drugB.rat.fuP            = 0.24;
db.drugB.rat.BP             = 1.27;
db.drugB.rat.CLblood_kgBW.hep = 55/1000; %
db.drugB.rat.lambda_po      = 0.2;

db.drugB.mouse.fuP            = 0.4;
db.drugB.mouse.BP             = 1.1;
db.drugB.mouse.CLblood_kgBW.hep = 37/1000; %
db.drugB.mouse.lambda_po      = 0.27;

db.drugB.human.fuP          = 0.296;
db.drugB.human.BP           = 0.84;
db.drugB.human.CLblood_kgBW.hep = 15/1000;
db.drugB.human.lambda_po    = 0.02;





%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% DRUG TYPE: antibodies
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% =======================================================================
db.IgGendo                 =  data.drug;
db.IgGendo.name            = 'IgGendo';
db.IgGendo.class           = 'pAb';
db.IgGendo.subclass        = 'IgGx';
db.IgGendo.MW              = 150*1e3;

%%% =======================================================================
db.mAb7E3               =  data.drug;
db.mAb7E3.name          = 'mAb7E3';
db.mAb7E3.class         = 'mAb';
db.mAb7E3.subclass      = 'IgG1';
db.mAb7E3.MW            = 150*1e3;


%%% =======================================================================
db.mAb8C2               =  data.drug;
db.mAb8C2.name          = 'mAb8C2';
db.mAb8C2.class         = 'mAb';
db.mAb8C2.subclass      = 'IgG1';
db.mAb8C2.MW            = 150*1e3;

%%% =======================================================================
db.IVIG                 =  data.drug;
db.IVIG.name            = 'IVIG';
db.IVIG.class           = 'pAb';
db.IVIG.subclass        = 'IgGx';
db.IVIG.MW              = 150*1e3;
%%% =======================================================================
db.mAbMOPC21               =  data.drug;
db.mAbMOPC21.name          = 'mAbMOPC21';
db.mAbMOPC21.class         = 'mAb';
db.mAbMOPC21.subclass      = 'IgG1';
db.mAbMOPC21.MW            = 150*1e3;

%%% =======================================================================
db.murineIgG1               =  data.drug;
db.murineIgG1.name          = 'murineIgG1'; %mouse monoclonal digoxin-specific IgG1  (Bazin-Redureau et al. 1997 J. Pharm Pharmacol.
db.murineIgG1.class         = 'mAb';
db.murineIgG1.subclass      = 'IgG1';
db.murineIgG1.MW            = 150*1e3;

%%% =======================================================================
db.ratIgG1               =  data.drug;
db.ratIgG1.name          = 'ratIgG1'; %rat monoclonal IgG1 directed against mouse IgG2b (Bazin-Redureau et al. 1997 J. Pharm Pharmacol.
db.ratIgG1.class         = 'mAb';
db.ratIgG1.subclass      = 'IgG1';
db.ratIgG1.MW            = 150*1e3;

%%% =======================================================================
db.humanIgG1               =  data.drug;
db.humanIgG1.name          = 'humanIgG1'; %rat monoclonal IgG1 directed against mouse IgG2b (Bazin-Redureau et al. 1997 J. Pharm Pharmacol.
db.humanIgG1.class         = 'mAb';
db.humanIgG1.subclass      = 'IgG1';
db.humanIgG1.MW            = 150*1e3;
%%% =======================================================================
%%% Assign values to drug structure
%%%


drug = db.(compound);
for s = 1:length(mammals.type)
    species = mammals.type{s};
    subtype = refSubtype.(species);
    
    switch drug.class
        case 'sMD';
            drug.(species).K_ery_up = determine_K_ery_up(drug.(species),subtype);
            
            CLblood_kgBW.hep = drug.(species).CLblood_kgBW.hep;
            drug.(species).CLintK_kgOW.liv = determine_CLintK_kgLiv(CLblood_kgBW.hep,subtype);
            
    end;
    
    %%% scaling factor between mg and nmol
    %%%
    drug.SF.mg_to_nmol = 1e6/drug.MW;
    drug.SF.nmol_to_mg = 1/drug.SF.mg_to_nmol;
    
end;

drug.info    = info;


end % of function drug = GenericPBPKmodel_drugDatabase(compound)


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function K_ery_up = determine_K_ery_up(drug_species,subtype)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

BP  = drug_species.BP;
fuP = drug_species.fuP;
hct = subtype.hct;

if BP<(1-hct)
    error('Blood-to-plasma ratio smaller than (1-hct) !');
end;

%%% erythrocyte-to-unbound plasma partition coefficient
K_ery_up = (BP-(1-hct)) / (hct*fuP);

end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function CLintK_kgLiv = determine_CLintK_kgLiv(CLblood_kgBW,subtype)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

BW     = subtype.BW;
OW_liv = subtype.OW_liv;
Q_liv  = subtype.Q_liv;

%%% hepatic blood clearance for reference species' subtype
CL_blood = CLblood_kgBW * BW;

if CL_blood>Q_liv
    error('CLblood.hep larger than Q_liv !');
end;

%%% intrinisic hepatic clearance * partition coefficient based on
%%% well-stirred liver model
CLintK = Q_liv * CL_blood / (Q_liv - CL_blood);

%%% intrinisic hepatic clearance * partition coefficient per kg liver OW
CLintK_kgLiv = CLintK/OW_liv;

end















