%%% Version: October 31st, 2014
%%%
%%%
%%% This function determines the dosing events and simulation time spans
%%% for a single and multiple dosing simulation with specified
%%% observation times
%%%
%%% For references and citation, please see MAIN script.
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function dosingEvent = GenericPBPKmodel_determineDosingEvents(Individual)

%%% In a FIRST STEP, all events of the form
%%% (a) observe the system at some specified time point (called 'obs')
%%% (b) administer an iv bolus or po dose, and
%%%     start or end an iv infusion (called 'dose')
%%% are extracted and storred in the structure 'event'.
%%%
%%% In a SECOND STEP, events are group in the structure 'simEvent'
%%% Each simulation event is defined by
%%% (a) its time 't0'
%%% (b) all administration events at time t0
%%% (c) the time 't1' of the next administration event (if any), or
%%%     the end of the simulation time span.
%%% The time t0, t1 define the time span 'tspan = [t0,t1]' that is used
%%% by the numerical integrator to solve the ODEs.
%%%

dosing = Individual.dosing;
pred   = Individual.pred;

%%% initialize first part of the event structure
M = length(pred.timeSpan);
event(M) = struct('t',[],'timeUnit',[],'type',[],'drug',[],'route',[],...
    'dose',[],'doseUnit',[],'dose_in_mg',[],'infusionRate',[],'rateUnit',[],'comment',[]);

%%% --- FIRST STEP --------------------------------------------------------
%%%
%%% (a) extract all observation time events. To this end, scale time to
%%% the internal unit [min].
%%%
SF2min = GenericPBPKmodel_ScalingFactor2internalUnits(pred.timeUnit);
tspan  = SF2min*pred.timeSpan;

no = 0;
for k=1:length(tspan)
    
    no = no+1;
    event(no).type = 'observation';
    event(no).t    = tspan(k);
    event(no).timeUnit = 'min';
    
end;

%%% (b) extract all dosing events. Do so for all dosing regimes. This this
%%% end, scale time to the internal unit [min] and dose to internal unit [nmol]
%%%
for k=1:length(dosing)
    
    
    %%% scaling factor to internal units [min] and [nmol]
    SF2min    = GenericPBPKmodel_ScalingFactor2internalUnits(dosing(k).timeUnit);
    compound  = dosing(k).drug;
    SFmg2nmol = Individual.drug.(compound).SF.mg_to_nmol;
    
    %%% time of dosing
    time   = SF2min * dosing(k).time;
    
    %%% multiple dosing interval (if applicable)
    tau    = SF2min * dosing(k).interval;
    if dosing(k).repeats>1 && isnan(tau)
        message = sprintf('If repeats>1 then dosing interval must not be equal to NaN.');
        GenericPBPKmodel_ReportErrorMessage(message);
    end;
    
    %%% account for dosing events
    for n=1:dosing(k).repeats
        
        no = no+1;
        event(no).t          = time;
        event(no).timeUnit   = 'min';
        event(no).type       = 'administration';
        event(no).drug       = dosing(k).drug;
        event(no).route      = dosing(k).route;
        event(no).dose_in_mg = GenericPBPKmodel_determineDose(dosing(k),Individual);
        event(no).dose       = SFmg2nmol * event(no).dose_in_mg;
        event(no).doseUnit   = 'nmol';
        
        %%% determine infusion rate and create aditional event for end of iv infusion
        if strcmp(dosing(k).route,'iv_infusion')
            
            %%% length of infusion
            Tinf    = SF2min * dosing(k).Tinf;
            infusionRate = event(no).dose/Tinf;
            
            event(no).infusionRate = infusionRate;
            event(no).rateUnit     = 'nmol/min';
            event(no).comment      = 'Start of infusion';
            
            %%% create new event for end of infusion
            no = no+1;
            event(no).t        = time+Tinf;
            event(no).timeUnit = 'min';
            event(no).type     = 'administration';
            event(no).drug     = dosing(k).drug;
            event(no).route    = dosing(k).route;
            event(no).dose     = 0;
            event(no).doseUnit = 'nmol';
            event(no).infusionRate = infusionRate;
            event(no).rateUnit     = 'nmol/min';
            event(no).comment      = 'End of infusion';
            
            if (dosing(k).repeats>1) && (Tinf>tau)
                message = sprintf('Length of iv infusion is larger than dosing interval!');
                GenericPBPKmodel_ReportErrorMessage(message);
            end;
            
        end;
        
        time = time + tau;
        
    end;
    
    
end;


%%% --- SECOND STEP -------------------------------------------------------
%%%
%%% Assign numerical values to administration and observation events to
%%% allow subsequently simple indexing.
administration = 1; observation = 2;

%%% determine the time and type of all events
N = length(event); type = zeros(1,N);
time = zeros(1,N);
for k=1:N
    time(k) = event(k).t;
    switch event(k).type
        case 'administration'
            type(k) = administration;
        case 'observation'
            type(k) = observation;
    end;
end;

%%% If tspan consists only of initial time T0 and final time Tend, i.e.
%%%     tspan = [T0, Tend],
%%% then adapt Tend, if end of last dosing event is after Tend
%%%
if (length(tspan) == 2) && (max(time) > tspan(2))
    time(2)    = max(time);
    event(2).t = max(time);
end;




%%% Determine the times for the ODE integration. These include
%%% (a) all adminstration times
%%% (b) start of simulation, if different from time of first
%%%     administration, e.g., in the case of pre-treatment simulation
%%% (c) end of simulation, in the case of simulation of the wash-out phase
%%%

%%% Ad (a): determine times of all administration events, sort in increasing order
[~, ind] = find(type==administration);
odeTime  = sort(unique(time(ind)));

%%% Ad (b): determine start of pre-treatment time (if applicable)
Tmin = min(time);
if Tmin<odeTime(1)
    odeTime = [Tmin odeTime];
end;

%%% Ad (c): determine end of simulation
Tmax = max(time);
if Tmax>odeTime(end)
    odeTime = [odeTime Tmax];
end;


%%% Determine final structure 'simEvent'
K = length(odeTime)-1;
dosingEvent(K) = struct('tspan',[],'admin',[]);
for k=1:K
    
    t0 = odeTime(k); t1 = odeTime(k+1);
    
    %%% determine all observation/administration times between t0 and t1
    [~, ind] = find( (t0<=time) & (time<=t1) );
    dosingEvent(k).tspan = sort(unique(time(ind)));
    
    %%% assign all administration events at time t0
    ind = find( (time==t0) & (type==administration) );
    if ~isempty(ind)
        dosingEvent(k).admin = event(ind);
    end;
    
end;

end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function dose = GenericPBPKmodel_determineDose(dosing,Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% scales the dose to the absolute amount

species = Individual.species;

message = 'no error';
switch dosing.doseUnit
    
    case 'mg';
        dose = dosing.dose;
        
    case 'mg/kg BW';
        dose = dosing.dose*species.BW;
        if isnan(dose)
            message = sprintf('Individual(%d).species.BW = NaN caused dose to be NaN !',Individual.id);
        end;
        
    case 'mg/kg LBW';
        dose = dosing.dose*species.LBW;
        if isnan(dose)
            message = sprintf('Individual(%d).species.LBW = NaN caused dose to be NaN  !',Individual.id);
        end;
        
    case 'mg/m2 BSA';
        dose = dosing.dose*species.BSA;
        
        if isnan(dose)
            message = sprintf('Individual(%d).species.BSA = NaN caused dose to be NaN  !',Individual.id);
        end;
    otherwise
        message = sprintf('Unknown unit of dose!');
        GenericPBPKmodel_ReportErrorMessage(message);
        
end;

if ~strcmp(message,'no error')
    GenericPBPKmodel_ReportErrorMessage(message);
end;

end

%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

