%%% Version: October 31st, 2014
%%%
%%% UPDATE:
%%% (1) listOfDrugs = [];
%%% 
%%% This function assigns the drug relevant data
%%% 
%%% For references and citation, please see MAIN script.
%%% 
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the 
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported 
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function Individual = GenericPBPKmodel_assignDrugData(Individual)

fprintf('\n   (b) Assigning drug-specific data');


%%% determine all drugs that are dosed in the study
%%%
for id = 1:length(Individual)

    dosing = Individual(id).dosing;
    
    listOfDrugs = [];
    for k = length(dosing):-1:1
        
        listOfDrugs{k} = dosing(k).drug;
        
    end;
    %%% sort out duplicates
    listOfDifferentDrugs = unique(listOfDrugs);
    
    %%% load drug data
    %%%
    for k = length(listOfDifferentDrugs):-1:1
        
        compound.name = listOfDifferentDrugs{k};
        compound = GenericPBPKmodel_drugDatabase(compound.name);
        
        %%% check, whether mixed drug-species data for the given species
        %%% are available 
        species = Individual(id).species.type;
        if ~isfield(compound,species)
            
            message = sprintf('No %s specific data for drug %s available!',species,compound.name);
            GenerikPBPKmodel_ReportErrorMessage(message);
            
        end;
         
        %%% assign mixed drug-species data
        mixed = fieldnames(compound.(species));
        
        for d =1:length(mixed)     
            data = mixed{d};
            compound.(data) = compound.(species).(data);
        end;
        
        switch compound.class
            
            case 'sMD';
            
                T   = Individual(id).species.T;
                hct = Individual(id).species.hct;
                fuP = compound.fuP;
                BP  = hct*compound.K_ery_up*fuP + (1-hct);
                compound.BP = BP;
                
                %%% tissue-to-unbound partition coefficients according to Rodgers&Rowland
                %%% default methods: RodgersRowland
                method.K_tis_up.approach = 'RodgersRowland';
                if isfield(Individual(id).model,'method') && isfield(Individual(id).model.method,'K_tis_up')
                    method.K_tis_up = Individual(id).model.method.K_tis_up;
                end;
                
                K.tis_up    = GenericPBPKmodel_TissuePartitionCoefficients(Individual(id),compound,method.K_tis_up);
                compound.K.tis_up      = K.tis_up;
                compound.info.K.tis_up = sprintf('tissue-to-unbound plasma partition coefficient (%s)',method.K_tis_up.approach);
                
                %%% intrinsic tissue clearance
                compound.CLint.tis      = T.initialize.tissueDB.NaN;
                compound.info.CLint.tis = 'tissue specific intrinsic clearance in [L/min]';
                
                K_liv_blo = K.tis_up(T.liv)*fuP/BP;
                compound.CLint.tis(T.liv) = compound.CLintK_kgOW.liv/K_liv_blo*Individual(id).species.OW.tis(T.liv);
            
            case {'mAb','pAb'};
                
                %%% use ABC values from Shah&Betts, assume that these already account for residual plasma
                %%% so that predcitions include residual blood
                method.ABC.approach = 'ShahBetts';
                if isfield(Individual(id).model,'method') && isfield(Individual(id).model.method,'ABC')
                    method.ABC = Individual(id).model.method.ABC;
                end;
                compound.ABC.tis = GenericPBPKmodel_TissuePartitionCoefficients(Individual(id),compound,method.ABC);
                               
            otherwise
                
        end;
        
        
        %%% delete all unnessesary data 
        all_species = {'human','rat','mouse'};
        for s = 1:length(all_species)
            species = all_species{s};
            if isfield(compound,species)
                compound = rmfield(compound,species);
            end;
        end;
        
        Individual(id).drug.(compound.name) = compound;
        
    end;
        
    Individual(id).drug.listOfDrugs = listOfDifferentDrugs;
    Individual(id).dosing = dosing;
    
end;

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES


%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

