%
%%% Version: October 31st, 2016
%%%
%%%
%%% This script specifies all inputs for the simulation of the
%%% pharmacokinetics of different mAbs in a virtual population
%%% to be specified.
%%%
%%% For reference, please cite:
%%%
%%% --> For small molecule drugs:
%%%
%%%     W. Huisinga, A. Solms, L. Fronton, S. Pilari,
%%%     Modeling Interindividual Variability in Physiologically Based
%%%     Pharmacokinetics and Its Link to Mechanistic Covariate Modeling,
%%%     CPT: Pharmacometrics & Systems Pharmacology (2012) 1, e5;
%%%     doi:10.1038/psp.2012.3
%%%
%%% --> For monoclonal antibodies:
%%%
%%%     L. Fronton, S. Pilari, W. Huisinga
%%%     Monoclonal antibody disposition: A simplified PBPK model and its
%%%     implications for the derivation & interpretation of classical
%%%     compartment models,
%%%     J. Pharmacokinet Pharmacodyn (2014)
%%%     doi:10.1007/s10928-014-9349-1
%%%
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%

clear all; close all;


%%% =================================================================
%%% Specify dosing, including the following information
%%%
%%% 'drug'      sMD or mAb (see below)
%%% 'route'     of adminstration: iv bolus, iv infusion or per oral
%%% 'dose'      of the drug, specified in units specified by ...
%%% 'doseUnit'  unit of the dose in 'mg' or 'mg/kg BW', 'mg/kg LBW', 'mg/m2 BSA'
%%% 'time'      of drug administration in units specified by ...
%%% 'timeUnit'  unit of time in 'min', 'h', 'd' (day), 'w' (week)
%%% 'repeats'   single dose (=1) or multiple dosing (>1)
%%% 'Tinf'      duration of infusion (if applicable)
%%% 'interval'  dosing interval (multiple dosing case, if applicable)
%%%
%%% SMALL MOLECULE DRUGS:
%%% further subdivided into moderate-to-strong bases,
%%% weak bases, neutrals, acids and zwitterions. This property determines
%%% the method to a-priori predict the tissue partition coefficients.
%%%
%%% Moderate-to-strong bases: Amitriptyline, Caffeine,
%%% Diltiazem, Diphenhydramine, Imipramine, Lidocaine, Metoprolol,
%%% Pindolol, Sildefanil, Theophylline
%%%
%%% Weak bases: Alprazolam, Diazepam, Flunitrazepam, Midazolam, Triazolam
%%%
%%% Acids: Amobarbital, Diclofenac, Hexobarbital, Ibuprofen, Methohexital,
%%% Thiopental, Tolbutamide, Valproate, Warfarin
%%%
%%% test drugs: drugA (similar to Warfarin), drugB (similar to Lidocaine)
%%%
%%% MONOCLONAL ANTIBODIES:
%%% Zalutumumab, mAb7E3
%%%

single_dose.drug      = [];
single_dose.route     = [];                         % 'iv_bolus', 'iv_infusion', 'po'
single_dose.dose      = [];                         % in [mg] or [mg]/per BSD (body size descriptor)'
single_dose.doseUnit  ='mg';                  % 'mg' or 'mg/kg BW', 'mg/kg LBW', 'mg/m2 BSA'
single_dose.time      = 0;                          % time of first administration
single_dose.timeUnit  = 'h';
single_dose.repeats   = 1;                          % number of administration events
single_dose.Tinf      = NaN;                        % duration of iv infusion
single_dose.interval  = NaN ;                       % multiple dosing interval

dosing       = single_dose;                         % dose in [mg/kg BW], administration at t=0, time in [h]
dosing.drug  = 'mAb7E3';                            %choose mAb7E3 mAb8C2 mAbMOPC21
dosing.dose  = 0.0038;
dosing.route = 'iv_bolus';                          %iv_infusion



%%% =================================================================
%%% Load characteristics of reference individuals
%%%
%%% The following reference individuals are available
%%%
%%%     reference.human.male_35y_70kg_176cm
%%%     reference.human.female_35y_60kg_163cm
%%%     reference.human.male_15y_54kg_167cm
%%%     reference.human.female_15y_53kg_161cm
%%%     reference.human.unisex_10y_32kg_138cm
%%%     reference.human.unisex_5y_19kg_109cm
%%%
%%%     reference.rat.male_250g
%%%     reference.rat.male_475g
%%%
%%%     reference.mouse.male_25g
%%%     reference.mouse.male_40g
%%%     reference.mouse.male_22g_Bax
%%%     reference.mouse.male_22g_Fro

reference = GenericPBPKmodel_referenceIndividual();


%%% =================================================================
%%% Generate virtual population based on antropometric data.
%%% Approaches for generating the population:

population.how2generate ='mAb_PBPK_NUDEmice_highFcRn_IVIG';

individual = GenericPBPKmodel_virtualPopulation(population,dosing,reference);




%%% =================================================================
%%% Assign anatomical & physiolocial species data and assign
%%% drug-specific data relevant for the chosen dosing
%%%

individual = GenericPBPKmodel_assignSpeciesData(individual);
T = GenericPBPKmodel_tissueIndexing;

individual = GenericPBPKmodel_assignDrugData(individual);


%%% =================================================================
%%% Random effects
%%% State factor of perturbation, i.e., if facX = 2, then the perturbed
%%% value of a quantity X will ly between 1/(1+facX) * X and (1+facX) * X
%%%


k=0;
randomEffects.mAb.facX.V_blood  =12/10*k;
randomEffects.mAb.facX.hct      =11/10*k;
randomEffects.mAb.facX.Dose     =0*k;
individual = GenericPBPKmodel_randomEffects(individual,randomEffects);



%%% =================================================================
%%% Simulate PBPK model for all individuals
%%%
%%% Note: If timeSpan consists only of initial and final time, i.e.,
%%%     timeSpan = [T0 Tend],
%%% then Tend will be automatically adapted, if final dosing event is
%%% at a time later than Tend

pred.timeSpan = [0 14];          % simulation time span
pred.timeUnit = 'd';             % unit of timeSpan

individual    = GenericPBPKmodel_simulatePBPKmodels(individual,pred);


%%% =================================================================
%%% Data analysis and descriptive statistics of individuals and
%%% the population
%%%

stats = GenericPBPKmodel_dataAnalysis(individual);


%%% =================================================================
%%% Graphical output. Specify below, which plots to be shown
%%%

ShowGraphics.plasma.allIndividuals = 'yes';     % 'yes' or 'no'
ShowGraphics.plasma.YScale         = 'log';     % 'log' or 'lin'
ShowGraphics.plasma.figNr          = 11;

allIds = 1:length(individual);
ShowGraphics.allTissues.individual = [];   % specify id of individuals, e.g. [1 2 7]; [] = 'No plot'
ShowGraphics.allTissues.YScale     = 'log';     % 'log' or 'lin'
ShowGraphics.allTissues.figNr      = 11;
ShowGraphics.allTissues.oneFigOnly = 'yes';     % use same figure for all individuals 'yes'/'no'

ShowGraphics.percentiles.plasma    = 'no';     % applies only to a virtual population of size 10 or larger
ShowGraphics.percentiles.YScale    = 'log';
ShowGraphics.percentiles.figNr     = 25;

ShowGraphics.lowerLimitY           = 1e-1;      % sets the lower value of the Y axis

ShowGraphics.TestForLossOfMass.id    = [];      % for testing purposes.



GenericPBPKmodel_graphicalOutput(individual,ShowGraphics);

GenerateFigure=1;




if GenerateFigure
    

    title('NUDE mice, mAb MOPC21+IVIG, high FcRn')

    xlabel([])
    ylabel([])
    
    lw = 2;   %line width
    fs = 20;  %fontsize
    
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',22);
    set(get(gca,'xlabel'),'Fontsize',22);
    set(get(gca,'ylabel'),'Fontsize',22);
    set(legend,'Fontsize',fs);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    
    % % % Saving eps with matlab
    filename='Fig9a';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
end




