%%% Version: October 31st, 2015
%%%
%%% This function load experimental data specified in some file name.
%%%
%%% For references and citation, please see MAIN script.
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function Individual = GenericPBPKmodel_ExperimentalData(Individual)


%%% load experimental data
%%%
[Individual, data] = GenericPBPKmodel_loadExperimentalData(Individual);

%%% assign dosing and pred
%%%
Individual = GenericPBPKmodel_assignDosingAndPred(Individual);

%%% determine standart output
%%%
Individual = GenericPBPKmodel_determineStandartOutput(Individual, data);

%%% make specific plots
%%%
GenericPBPKmodel_specificGraphicalOutput(Individual);


end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [Individual, data] = GenericPBPKmodel_loadExperimentalData(Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% Note: It is assumed that the names (abreviations) of the tissues
%%% specified below are part of the generic list defined in the species
%%% data base

model   = Individual.model;

%%% load experimental data
%%%
expdata = eval(model.filename);

unit = expdata.unit;
SF   = expdata.SF;

%%% -----------------------------------------------------------------------
%%% Define indexing
%%%

%%% tissue indexing
%%%
S = Individual.species.T;

tissues    = fieldnames(expdata.C);
S.allOrg   = [];
timepoints = [];

for k = 1:length(tissues)
    
    org = tissues{k};
    S.allOrg = union(S.allOrg,S.(org));
    timepoints = union(timepoints,expdata.t.(org));
    
end

S.tissueDB = S.allOrg;
S.allOrgExBlo = setdiff(S.allOrg,[S.ven S.art S.pla S.blo]);

S.maxIndex.org = max(S.tissueDB);
S.initialize.time.NaN   = NaN*ones(length(timepoints),1);
S.initialize.org.NaN    = NaN*ones(1,S.maxIndex.org);
S.initialize.amount.NaN = NaN*ones(length(timepoints),1);


%%% -----------------------------------------------------------------------
%%% Assign experimental data
%%%

t = SF.time_to_min * timepoints;

C.exp = ones(size(t))*S.initialize.org.NaN;

errorbar.up.C.exp   = ones(size(t))*S.initialize.org.NaN;
errorbar.down.C.exp = ones(size(t))*S.initialize.org.NaN;

errorbar.available  = expdata.errorbar.available;

limits.t = ones(S.maxIndex.org,1)*[0 max(t)];
for k = S.allOrg
    org  = S.name{k};
    [~,ind,~] = intersect(timepoints,expdata.t.(org));
    C.exp(ind,S.(org))  = SF.conc_to_nM  * expdata.C.(org);
    limits.t(S.(org),:) = SF.time_to_min * expdata.limits.t.(org);
    
    if ismember(org,errorbar.available)
        
        errorbar.up.C.exp(ind,S.(org))    = SF.conc_to_nM * expdata.errorbar.up.C.(org);
        errorbar.down.C.exp(ind,S.(org))  = SF.conc_to_nM * expdata.errorbar.down.C.(org);
        
        
        
    end;
end;

data.t = t;
data.C = C;
data.errorbar = errorbar;

%%% -----------------------------------------------------------------------
%%% Assign model parameters
%%%

model.S  = S;
model.SF = SF;
model.expdata = expdata;
model.color   = expdata.color;
model.limits  = limits;
model.unit    = unit;

Individual.model = model;

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function Individual = GenericPBPKmodel_assignDosingAndPred(Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model    = Individual.model;

Individual.dosing = model.expdata.dosing;
Individual.pred   = model.expdata.pred;

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function Individual = GenericPBPKmodel_determineStandartOutput(Individual,data)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model    = Individual.model;
pred     = Individual.pred;  info = Individual.info;
drugName = Individual.drug.listOfDrugs{1};
cmp      = Individual.drug.(drugName);

S = model.S; SF = cmp.SF;


%%% -----------------------------------------------------------------------
%%% assign predictions (in internal units)
%%%
info.pred.t         = 'simulation time in [min]';
pred.t              = data.t;

pred.errorbar.available = data.errorbar.available;
%%% initialize all compartments with NaN and assign only to those
%%% compartments values that have been measured
initialize.NaN = ones(size(pred.t))*S.initialize.org.NaN;
vas = [S.blo S.pla S.ven S.art]; allOrgExVas = setdiff(S.allOrg,vas);

%%% vascular concentrations (including venous plasma)
%%%
info.pred.C.vas     = 'concentration in vascular space [nmol/L]';
pred.C.vas          = initialize.NaN;
pred.C.vas(:,vas)   = data.C.exp(:,vas);

%%% tissue concentrations
%%%
info.pred.C.tis     = 'concentration in tissue space [nmol/L]';
pred.C.tis          = initialize.NaN;
pred.C.tis(:,allOrgExVas)   = data.C.exp(:,allOrgExVas);

%%% error bars for tissue concentrations
pred.errorbar.up.C.tis   = initialize.NaN;
pred.errorbar.down.C.tis = initialize.NaN;

pred.errorbar.up.C.tis(:,allOrgExVas)   = data.errorbar.up.C.exp(:,allOrgExVas);
pred.errorbar.down.C.tis(:,allOrgExVas) = data.errorbar.down.C.exp(:,allOrgExVas);


%%% plasma concentration
%%%
info.pred.C.pla      = 'concentration in venous plasma [nmol/L]';
pred.C.pla           = data.C.exp(:,S.pla);


%%% error bars for plasma concentrations
pred.errorbar.up.C.pla   = data.errorbar.up.C.exp(:,S.pla);
pred.errorbar.down.C.pla = data.errorbar.down.C.exp(:,S.pla);


%%% -----------------------------------------------------------------------
%%% determine standard output in [mg/L] or [mg]
%%%
stdout = Individual.stdout; stdout.T = Individual.species.T; T = stdout.T;
initialize.NaN = ones(size(S.initialize.time.NaN))*T.initialize.tissueDB.NaN;

info.stdout.t         = 'simulation time in [min]';
stdout.t              = pred.t;

info.stdout.limits    = 'interesting range of experimental data';
stdout.limits.t       = model.limits.t;


%%% tissue concentrations
%%%

tis = intersect(T.tissueDB,setdiff(S.tissueDB,vas));

info.stdout.C.tis     = 'concentration in tissue space [mg/L]';
stdout.C.tis          = initialize.NaN;
stdout.C.tis(:,tis)   = SF.nmol_to_mg * pred.C.tis(:,tis);

%%% plasma concentration
info.stdout.C.pla      = 'concentration in venous plasma [mg/L]';
stdout.C.pla  = SF.nmol_to_mg*pred.C.pla;

Individual.info   = info;
Individual.pred   = pred;
Individual.stdout = stdout;

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_specificGraphicalOutput(Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% define here specific output for the experimental data

return;

pred = Individual.pred; model = Individual.model; S = model.S; color = model.color;
SF_min2day = 1/(60*24);
t = SF_min2day*pred.t;

%%% plot experimental plasma and tissue data
%%%
figNr = 100;
figure(figNr); clf;
ind = find(isfinite(pred.C.vas(:,S.pla)));
semilogy(t(ind),pred.C.vas(ind,S.pla),color.pla,'LineWidth',4,'LineStyle','-');
hold on;
semilogy(t(ind),pred.C.vas(ind,S.pla),'d','MarkerEdgeColor','k','MarkerFaceColor','g','MarkerSize',5);
for n = S.allOrgExBlo
    tis = S.name{n};
    ind = find(isfinite(pred.C.tis(:,S.(tis))));
    semilogy(t(ind),pred.C.tis(ind,S.(tis)),color.(tis),'LineWidth',4,'LineStyle','-');
    semilogy(t(ind),pred.C.tis(ind,S.(tis)),'d','MarkerEdgeColor','k','MarkerFaceColor','g','MarkerSize',5);
end;
hold off;
xlabel('time in [days]'); ylabel(sprintf('C_{tis} in [%s]',model.unit.C));
xlim([0 10]);
title('Experimental data')
set(gca,'YScale','log'); axis square;
fett(figNr); % print -depsc -f100 experimentalData


%%% plot normalized experimental plasma and tissue data
%%%
figNr = figNr+1;
figure(figNr); clf;
ind = find(isfinite(pred.C.vas(:,S.pla)));
Z = pred.C.vas(ind(5),S.pla);
semilogy(t(ind),pred.C.vas(ind,S.pla)/Z,color.pla,'LineWidth',4,'LineStyle','-');
hold on;
semilogy(t(ind),pred.C.vas(ind,S.pla)/Z,'d','MarkerEdgeColor','k','MarkerFaceColor','g','MarkerSize',5);
for n = S.allOrgExBlo
    tis = S.name{n};
    ind = find(isfinite(pred.C.tis(:,S.(tis))));
    Z = pred.C.tis(ind(end),S.(tis));
    semilogy(t(ind),pred.C.tis(ind,S.(tis))/Z,color.(tis),'LineWidth',4,'LineStyle','-');
    semilogy(t(ind),pred.C.tis(ind,S.(tis))/Z,'d','MarkerEdgeColor','k','MarkerFaceColor','g','MarkerSize',5);
end;
hold off;
xlabel(sprintf('time in [%s]',model.unit.t)); ylabel('C_{tis}/C_{tis}(Tend)');
xlim([0 10]);
title('Normalized experimental data')
set(gca,'YScale','log'); axis square;
fett(figNr); % print -depsc -f101 experimentalDataNormalized


end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = fett(figNr)

figure(figNr);
fs = 20;
set(gca,'FontSize',fs);
set(get(gca,'title'),'Fontsize',fs);
set(get(gca,'xlabel'),'Fontsize',fs);
set(get(gca,'ylabel'),'Fontsize',fs);

end

%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

