%%% Version: November 6th, 2014
%%% 
%%% This function simulates the models of all individuals that comprise the
%%% virtual population
%%% 
%%% For references and citation, please see MAIN script.
%%% 
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the 
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported 
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


function individual = GenericPBPKmodel_simulatePBPKmodels(individual,pred)

%%% ======================================================================
%%% report about simulation progress
%%%
fprintf('\n   (c) Simulating PBPK model(s) ...\n');


for id=1:length(individual)
    
    %%% report about characteristics of the individual
    individual(id).id = id;
    GenericPBPKmodel_reportOnIndividual(individual(id));

    %%% determine simulation events (observation, administration), unless
    %%% individual is based on experimental data
    %%%
    if ~strcmp(individual(id).model.type,'ExprimentalData')
        
        individual(id) = GenericPBPKmodel_assignPred(individual(id),pred);
        
        dosingEvent    = GenericPBPKmodel_determineDosingEvents(individual(id));
        individual(id).pred.dosingEvent = dosingEvent;
        
    end;
    
    %%% simulate individual according to its model
    %%%
    individual(id) = eval(sprintf('GenericPBPKmodel_%s(individual(id))',individual(id).model.type));
    
   
end;

fprintf('\n\n')

end 


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_reportOnIndividual(Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

fprintf('\n   individual(%1.0f):   %s ---',Individual.id,Individual.details);
for k=1:length(Individual.drug.listOfDrugs)
    fprintf(' %s,',Individual.drug.listOfDrugs{k})
end;
fprintf('\b --- %s \n',Individual.model.type);



%%% --- report on tissues that are lumped together
if isfield(Individual.model,'lumping')
    
    lumping = Individual.model.lumping;
    noLumpedTis = length(lumping);
    fprintf('                    %d-CMTs: ',noLumpedTis);
    for k=1:noLumpedTis
        fprintf('{');
        for m=1:length(lumping{k})
            fprintf('%s,',lumping{k}{m});
        end;
        fprintf('\b},');
    end;
    fprintf('\b');
    
    fprintf('\n')
end;
    
    

end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_assignPred(individual,pred)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% check, whether pred already exists as substructure in individual
if isfield(individual,'pred')
    
    %%% specified pred will be added to existing pred, but preexisting
    %%% fields will not be overwritten!
    addpred = pred;
    pred    = individual.pred;
    
    %%% double check existing pred that timeUnit is given, if timeSpan
    %%% was given
    if isfield(pred,'timeSpan') && ~isfield(pred,'timeUnit')
        message('timeSpan specified without having specified timeUnit!');
        GenericPBPKmodel_reportErrorMessage(message);
    end
    
    addpredfields = fieldnames(addpred);
    for k = 1:length(addpredfields)
        field = addpredfields{k};
        if ~isfield(pred,field)
            pred.(field) = addpred.(field);
        end
    end;
    
end;

%%% transform into internal units
SF = GenericPBPKmodel_ScalingFactor2internalUnits(pred.timeUnit);
pred.timeSpan = SF * pred.timeSpan;
pred.timeUnit = 'min';

individual.pred = pred;
end

%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

