%%% Version: October 31st, 2016
%%%
%%%
%%% This function specifies a simplfied PBPK model for therapeutic mAb with
%%% extravasation-limited tissue distribution and no target, parameterized
%%% in terms of INTERSTITIAL volume rather than tissue volume, with linear clearance from plasma(implicitly consideration of FcRn)
%%%
%%%DETAILS:
%%%PBPK Model with tumor CMT and tumorgrowth without target
%%%detailed volume parameterization for tumor interstititial tumor volumes after Gullino et al. 1965
%%%constant Tumor plasma flow from Baxter et al. 1994
%%%For Mass balance: Tumor CMT is described by C_tum(t) with dilution term (- dV(t)/dt * Ctum(t)) and V_tum(t)

%%% For references and citation, please see MAIN script.
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function Individual = GenericPBPKmodel_mAb_PBPK_11CMT_extravasLim_int_Vgrowth_noTar_V(Individual)

%%% check whether drug class is really sMD
%%%
GenericPBPKmodel_checkForModelApplicability(Individual);

%%% define model specific parameters
%%%
Individual = GenericPBPKmodel_defineModelParameters(Individual);

%%% simulate PBPK model
%%%
model = Individual.model; dosingEvent = Individual.pred.dosingEvent;
X0 = model.X0; sim.t = []; sim.X = [];

for d=1:length(dosingEvent)
    
    %%% administration details and simulation time span
    admin = dosingEvent(d).admin;
    tspan = dosingEvent(d).tspan;
    
    %%% account for dosing
    X0 = GenericPBPKmodel_drugAdministration(X0,model,admin);
    
    %%% solve system of ODEs 'GenericPBPKmodel_RHS' with initial conditions 'X0'
    [t,X] = ode15s(@GenericPBPKmodel_RHS,tspan,X0',[],Individual);
    
    %%% store current output in sim structure
    sim.t = [sim.t; t]; sim.X = [sim.X; X];
    
    X0 = X(end,:)';
    
end;

%%% determine standard simulation output
%%%
Individual = GenericPBPKmodel_determineStandartOutput(Individual,sim);

%%% normalized concentration plot
%%%
GenericPBPKmodel_normalizedConcentrationPlot(Individual);

%%% graphical output specific to this model
%%%
GenericPBPKmodel_specificGraphicalOutput(Individual);

end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_checkForModelApplicability(Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

if length(Individual.drug.listOfDrugs)>1
    message = sprintf('Model >>%s<< not defined for multiple drugs',Individual.model.type);
    GenericPBPKmodel_ReportErrorMessage(message);
end;

compound = Individual.drug.listOfDrugs{1};

if ~strcmp(Individual.drug.(compound).class,'mAb')
    message = sprintf('Model >>%s<< not defined for drug type %s!',Individual.model.type,Individual.drug.(compound).class);
    GenericPBPKmodel_ReportErrorMessage(message);
end;

end



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function Individual = GenericPBPKmodel_defineModelParameters(Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model = Individual.model;

%%% -----------------------------------------------------------------------
%%% Define indexing
%%%

%%% tissue topology indexing
%%%

S = Individual.species.T;

S.allCmts      = [S.lun S.adi S.hea S.kid S.mus S.bon S.ski S.gut S.spl S.liv S.pla S.tum];   %tum eingefuegt
S.allCmtsExTum = [S.lun S.adi S.hea S.kid S.mus S.bon S.ski S.gut S.spl S.liv S.pla      ];    %extum eingefuegt
S.organs       = [S.lun S.adi S.hea S.kid S.mus S.bon S.ski S.gut S.spl S.liv       S.tum];    % tum eingefuegt
S.organsExTum  = [S.lun S.adi S.hea S.kid S.mus S.bon S.ski S.gut S.spl S.liv       ];  %extum eingefuegt
S.plasma       = [                                                            S.pla];
S.Tum          = [                                                                  S.tum];   % tum eingefuegt

S.tissueDB    = S.allCmts; % specify which tissues are part of the general tissue DB

S.visOrg      = [S.gut S.spl S.liv];
S.nonvisOrg   = [S.lun S.adi S.hea S.kid S.mus S.bon S.ski];

S.maxIndex.org = max(S.allCmts);
S.initialize.org.NaN   = NaN*ones(1,S.maxIndex.org);


%%% compartment indexing
%%%
S.C_pla = 1;                      % total plasma compartment
S.bolus = S.C_pla;                % i.v. bolus administration
S.C_int = (1:S.maxIndex.org) + 1; % interstitial compartments

S.V_tum = max(S.C_int)+1;         % tumor interstitial volume

S.maxIndex.cmt = max(S.V_tum);


%%% dosing and metabolism indexing
%%%
S.GItract = 1 + S.maxIndex.cmt; % gastro-intestinal tract (for po dosing)
S.IVbag   = 2 + S.maxIndex.cmt; % IVbag (for infusion dosing)
S.IVrate  = 3 + S.maxIndex.cmt; % IVbag (for infusion dosing)
S.metab   = 4 + S.maxIndex.cmt; % metabolized drug

S.maxIndex.dosing = 4 + S.maxIndex.cmt;



%%% additional indexing
%%%

S.maxIndex.all = 0 + S.maxIndex.dosing;


%%% -----------------------------------------------------------------------
%%% Define PBPK parameters
%%%

org     = S.organs; pla = S.pla;
orgexTum= S.organsExTum;
tum     = S.tum;

species  = Individual.species;

compound = Individual.drug.listOfDrugs{1};
mAb      = Individual.drug.(compound);


%%% total tumor volume
%%%
if isfield(model,'SCIDmice') && strcmp(model.SCIDmice,'yes')
    V.tot(tum)    = 2.5e-04; % [l] initial tumor volume for tumor growth (assumption: tumor growth data measured using caliper and thus, it is total volume with vascular volume)
end

if isfield(model,'NudeMice') && strcmp(model.NudeMice,'yes')
    V.tot(tum)    = 4.72e-04; % [l] initial tumor volume for tumor growth (assumption: tumor growth data measured using caliper and thus, it is total volume with vascular volume)
end
%%% vascular tumor volume
%%%
fVtot.vas(tum)= 6/100;%assuming 6% fraction of vascular tumor volume of total tumor volume

V.vas(tum)    = fVtot.vas(tum).*V.tot(tum);%initial vascular tumor volume

%%% tissue volumes
%%%
fVtot.tis(tum) = 77/100;%assuming 77% fraction of tissue tumor volume of total tumor volume

V.tis          = S.initialize.org.NaN;
V.tis(orgexTum)= species.V.tis(orgexTum);
V.tis(tum)     = fVtot.tis(tum)* V.tot(tum);%%%initial tumor tissue volume

fVtis.int           = S.initialize.org.NaN;
fVtis.int(orgexTum) = species.fVtis.int(orgexTum);
fVtis.int(tum)      = 42/100;   %%%%assuming 42% fraction of interstitial tumor volume of tissue tumor volume

V.int            = S.initialize.org.NaN;
V.int(orgexTum)  = fVtis.int(orgexTum).*V.tis(orgexTum);
V.int_intial(tum)= fVtis.int(S.tum)*V.tis(tum); %initial interstitial tumor tissue volume



%%% blood volume
%%%
V.blood_new = species.V.blood+V.vas(tum);%%%add vascular tumor volume to V.blood

%%%plasma volume
%%%
V.pla = (1-species.hct) * V.blood_new;

%%% volume for i.v. bolus dosing
%%%
V.iv_bolus = V.pla;

%%% blood flows (ensure closed circulatory system!)
%%%
Q.blo           = S.initialize.org.NaN;

if isfield(model,'variability_co') && strcmp(model.variability_co,'yes')
    
    mu             = species.co;
    std            = 20/100*mu;
    
    %%transform mean and sd to the scale of the underlying normal distribution
    %%%mean of normal distribution
    log_mu =  log(mu.^2./sqrt(mu.^2+std.^2));
    %%%standard deviation (sigma) of normal distribution
    log_sd = sqrt(log(std.^2./mu.^2+1));
    %%lognormally distributed values of delta with mu and stdev
    species.co=exp((log_mu +log_sd.* randn(1,1)));
    
    
end


Q.blo(orgexTum)    = species.fQco(orgexTum) * species.co;
Q.blo(S.lun)       = species.co;

%%% plasma flows (ensure closed circulatory system!)
%%%
Q.pla(orgexTum) = (1-species.hct)*Q.blo(orgexTum); % peripheral plasma flow
Q.pla(tum)      = 1e-4; %%%%peripheral plasma flow  of tumor in [l/min] based on Baxter et al. 1994

%%% lymph flow as fraction of peripheral plasma and blood flow
%%%
fLymph              = S.initialize.org.NaN;
fLymph(S.visOrg)    = 0.02;
fLymph(S.nonvisOrg) = 0.04;
fLymph(tum)         = 0.004;

%%%  assuming 30% variability on fLymph: to be uniformly distributed in log
%%%  space
if isfield(model,'variability_fLymph') && strcmp(Individual.model.variability_fLymph,'yes')
    a=7/10;
    b=13/10;
    
    LA = log(a); LB = log(b);
    fac1=exp(LA + (LB-LA) * rand(size(fLymph(S.visOrg))));
    fac2=exp(LA + (LB-LA) * rand(size(fLymph(S.nonvisOrg))));
    fac3=exp(LA + (LB-LA) * rand(size(fLymph(S.tum))));
    
    fLymph(S.visOrg)   = fac1.* fLymph(S.visOrg);
    fLymph(S.nonvisOrg)= fac2.* fLymph(S.nonvisOrg);
    fLymph(S.tum)      = fac3.* fLymph(S.tum);
end;


Q.lymph      = fLymph.*Q.pla;
Q.lymph(pla) = sum(Q.lymph(org));


%%% define vascular reflection coefficients
%%% Brown Mouse
sigma.vas = S.initialize.org.NaN;
%%%%sigmas Brown Mouse (set manually, fraction (1-sigma.vas) is reduced by
%%%%factor 0.5 from Fronton Mouse and skin put to another group


sigma.vas([S.liv S.spl S.gut])      = 0.95;
sigma.vas([S.hea S.ski S.kid S.lun])= 0.975;
sigma.vas([S.mus S.adi S.bon])      = 0.99;

if strcmp(Individual.estimation.modus,'off')
    if isfield(model,'SCIDmice') && strcmp(model.SCIDmice,'yes')
        sigma.vas(S.tum)    =0.7042;
    end
    
    if isfield(model,'NudeMice') && strcmp(model.NudeMice,'yes')
        sigma.vas(S.tum)    =0.7545;
    end
end



if isfield(model,'variability_sigma_lognormal') && strcmp(model.variability_sigma_lognormal,'yes')
    
    
    delta        = S.initialize.org.NaN;
    log_muDelta  = S.initialize.org.NaN;
    log_sdDelta  = S.initialize.org.NaN;
    
    muDelta      = S.initialize.org.NaN;
    sdDelta      = S.initialize.org.NaN;
    
    %mean delta
    muDelta([S.liv S.spl S.gut])      = (1-sigma.vas([S.liv S.spl S.gut])) ;
    muDelta([S.hea S.ski S.kid S.lun])= (1- sigma.vas([S.hea S.ski S.kid S.lun]));
    muDelta([S.mus  S.adi S.bon])     = (1-sigma.vas([S.mus  S.adi S.bon]));
    muDelta([S.tum])                  = (1-sigma.vas([S.tum]));
    
    
    
    %%%%std dev of the log normally distributed delta (1-sigma) with assumed
    %%%%CV=10% -> 10/100 * 0.025
    sdDelta([S.liv S.spl S.gut])       = 0.005;
    sdDelta([S.hea S.ski S.kid S.lun]) = 0.0025;
    sdDelta([S.mus  S.adi S.bon])      = 0.0005;
    sdDelta([S.tum])                   = (10/100)*muDelta([S.tum]);
    
    %%transform mean and sd to the scale of the underlying normal distribution
    %%%mean of normal distribution
    log_muDelta =  log(muDelta.^2./sqrt(muDelta.^2+sdDelta.^2));
    %%%standard deviation (sigma) of normal distribution
    log_sdDelta = sqrt(log(sdDelta.^2./muDelta.^2+1));
    
    
    %%lognormally distributed values of delta with mu and stdev
    delta=exp((log_muDelta +log_sdDelta.* randn(1,1)));
    
    
    %calculate reflection coefficinets sigma from delta
    sigma.vas([S.liv S.spl S.gut])          = 1-delta([S.liv S.spl S.gut]);
    sigma.vas([S.hea S.ski S.kid S.lun])    = 1- delta([S.hea S.ski S.kid S.lun]);
    sigma.vas([S.mus  S.adi S.bon])         =1-delta([S.mus  S.adi S.bon]);
    sigma.vas([S.tum])                      =1-delta([S.tum]);
end;

sigma.pla = 1 - sum( Q.lymph(org).*(1-sigma.vas(org)) ) /Q.lymph(pla);

%%% drug specific parameters starting below
%%%

ABC.tis      = S.initialize.org.NaN;
ABC.tis(org) = mAb.ABC.tis(org);
ABC.pla      = mAb.ABC.tis(pla);


if isfield(model,'variability_ABC') && strcmp(model.variability_ABC,'yes')
    %%variability for ABCs: sample from standard distribution with mean and
    %%CV [%] as in the article Shah/Betts (2012)with standard deviation std= CV *mu
    
    mu = mAb.ABC.tis;
    ABC_std(S.lun) = 4.95/100*0.1490;
    ABC_std(S.hea) = 3.57/100*0.1020;
    ABC_std(S.kid) = 3.10/100*0.1370;
    ABC_std(S.mus) = 5.76/100*0.0397;
    ABC_std(S.ski) = 5.67/100*0.1570;
    ABC_std(S.gut) = (1/2*(3.89/100+4.74/100))*0.0512; % average of small and large intestine
    ABC_std(S.spl) = 4.85/100*0.1280;
    ABC_std(S.liv) = 3.75/100*0.1210;
    ABC_std(S.bon) = 11.8/100*0.0727;
    ABC_std(S.adi) = 9.59/100*0.0478;
    
    
    std=S.initialize.org.NaN;
    ABC_std;
    std = ABC_std;
    
    
    
    %%transform mean and sd to the scale of the underlying normal distribution
    %%%mean of normal distribution
    log_mu(orgexTum) =  log(mu(orgexTum).^2./sqrt(mu(orgexTum).^2+std(orgexTum).^2));
    %%%standard deviation (sigma) of normal distribution
    log_sd(orgexTum) = sqrt(log(std(orgexTum).^2./mu(orgexTum).^2+1));
    for k=orgexTum
        %%lognormally distributed values of delta with mu and stdev
        ABC.tis(k)=exp((log_mu(k) +log_sd(k).* randn(1,1)));
    end
    
    ABC.tis(orgexTum)=ABC.tis(orgexTum);
    
end


ABC.pla      = mAb.ABC.tis(pla);



%%% estimation mode
%%%
if strcmp(Individual.estimation.modus,'on')
    sigma.vas(tum)= Individual.estimation.sigma_tum;
    ABC.tis(tum)  = Individual.estimation.ABC_tum;
end;

if strcmp(Individual.estimation.modus,'off')
    if isfield(model,'SCIDmice') && strcmp(model.SCIDmice,'yes')
        ABC.tis(tum)    =0.3556;
    end
    
    if isfield(model,'NudeMice') && strcmp(model.NudeMice,'yes')
        ABC.tis(tum)    = 0.2267;
    end
end


if isfield(model,'variability_ABC') && strcmp(model.variability_ABC,'yes')
    mu =ABC.tis(tum);
    std=10/100*mu;
    
    %%transform mean and sd to the scale of the underlying normal distribution
    %%%mean of normal distribution
    log_mu =  log(mu.^2./sqrt(mu.^2+std.^2));
    %%%standard deviation (sigma) of normal distribution
    log_sd = sqrt(log(std.^2./mu.^2+1));
    %%lognormally distributed values of delta with mu and stdev
    ABC.tis(tum)=exp((log_mu +log_sd.* randn(1,1)));
end




%%% define elimination-corrected tissue partition coefficients
eK.tis       = S.initialize.org.NaN;
eK.tis(org)  = ABC.tis(org)./(1-sigma.vas(org));

eK.pla       = ABC.pla./(1-sigma.pla);


%%% blood-to-plasma ratio assuming that mAb do not distribute or bind to
%%% erythrocytes
%%%
BP = (1-species.hct);

%%% tissue-specific extraction ratio
E.tis      = S.initialize.org.NaN;
E.tis(org) = 0;
E.pla      = 0;

%%% tissue plasma clearance
CLpla.tis      = S.initialize.org.NaN;
CLpla.tis(org) = 0;



if isfield(model,'SCIDmice') && strcmp(model.SCIDmice,'yes')
    CLpla.pla =  2.513577e-07; %OFV 2.306878e+00 ; this is the calculated CLPla for SCID mice not scaled to species.BW
end;

if isfield(model,'NudeMice') && strcmp(model.NudeMice,'yes')
    CLpla.pla =   2.645975e-07  ; %OFV 1.940170e-01;  CLPla for nude mice not scaled to species.BW
end;




%%%scaling of clearance with BW
CLpla.pla= CLpla.pla *(Individual.species.BW/0.025);

%%%  assuming 20% variability on CLPla: to be uniformly distributed in log
%%%  space
if isfield(model,'variability_CLpla') && strcmp(Individual.model.variability_CLpla,'yes')
    a=8/10;
    b=12/10;
    
    LA = log(a); LB=log(b);
    fac1=exp(LA + (LB-LA) * rand(size(CLpla.pla)));
    
    CLpla.pla  = fac1.* CLpla.pla;
end;


%%% determine plasma clearance corresponding to the different intrinsic
%%% tissue clearances
CLpla.tis(org) = E.tis(org).*Q.lymph(org).*(1-sigma.vas(org));


%%% partition coefficients
%%%
K.tis(org) = eK.tis(org)./(1-E.tis(org));
K.pla      = eK.pla./(1-E.pla);

%%% define intrinsic tissue clearance
%%%
CLint.tis      = S.initialize.org.NaN;
CLint.tis(org) = Q.lymph(org)./K.tis(org).*E.tis(org)./(1-E.tis(org));

%%% compute ABCs values and effective flows
Leff.lymph   = (1-sigma.vas).*Q.lymph;


%%% compute ABCs, eK and K with respect to interstitial volume
%%% based on volume scaling factors
%%%
SF = mAb.SF;
SF.int2tis(orgexTum) = species.fVtis.int(orgexTum);
SF.int2tis(tum)      = fVtis.int(tum);

SF.tis2int(orgexTum) = 1./SF.int2tis(orgexTum);
SF.tis2int(tum)      = 1./SF.int2tis(tum);


ABC.int           = S.initialize.org.NaN;
ABC.int(orgexTum) = SF.tis2int(orgexTum).*ABC.tis(orgexTum);
ABC.int(tum)      = SF.tis2int(tum).* ABC.tis(tum);

eK.int       = S.initialize.org.NaN;
eK.int(org)  = SF.tis2int(org).*eK.tis(org);

K.int        = S.initialize.org.NaN;
K.int(org)   = SF.tis2int(org).*K.tis(org);

E.int        = S.initialize.org.NaN;
E.int        = E.tis; % note: scaling factors cancel out!

CLint.int    = S.initialize.org.NaN;
CLint.int    = Q.lymph./K.int.*E.int./(1-E.int);




%%% initial condition of ODEs
%%%
X0 = zeros(1,S.maxIndex.all);


%%% initial condition for interstitial tumor volume
X0(S.V_tum) = V.int_intial(tum);



%%% -----------------------------------------------------------------------
%%% Assign model parameters
%%%



model.S             = S;
model.V.int         = V.int;
model.V.pla         = V.pla;
model.V.iv_bolus    = V.iv_bolus;
model.Q             = Q;
model.K             = K;
model.E             = E;
model.eK            = eK;
model.BP            = BP;
model.fLymph        = fLymph;
model.sigma.vas     = sigma.vas;
model.CLint         = CLint;
model.CLpla         = CLpla;
model.ABC           = ABC;
model.L             = Leff;
model.SF            = SF;
model.X0            = X0;



Individual.model = model;

%%% -----------------------------------------------------------------------
%%% double check, whether there is some parameter values resulting in
%%% NaN values in the ODE
%%%
dX = GenericPBPKmodel_RHS(0,Individual.model.X0',Individual);
if sum(isnan(dX))
    message = sprintf('Some parameter values of id=%d cause the RHS of the ODE to have NaN values!',Individual.id);
    GenericPBPKmodel_ReportErrorMessage(message);
end;


end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [X0] = GenericPBPKmodel_drugAdministration(X0,model,admin)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% time and dose are expected to be in internal units [min] and [nmol]

S = model.S;

for k=1:length(admin)
    
    dose = admin(k).dose;
    
    
    switch admin(k).route
        case 'iv_bolus';
            X0(S.bolus) = X0(S.bolus) + dose/model.V.iv_bolus;
            
            
        case 'iv_infusion';
            if dose>0 % start of iv infusion
                X0(S.IVbag)  = X0(S.IVbag) + dose;
                X0(S.IVrate) = X0(S.IVrate) + admin(k).infusionRate;
            else        % end of iv infusion
                X0(S.IVrate) = X0(S.IVrate) - admin(k).infusionRate;
            end;
            
        case 'po';
            %%% double check that p.o. dose = 0
            message = 'po administration of mAb not supported!';
            GenericPBPKmodel_ReportErrorMessage(message);
    end;
    
    
end;

if sum(isnan(X0))
    message = sprintf('Some parameter values of id=%d cause the initial condition X0 of the ODE to have NaN values!',Individual.id);
    GenericPBPKmodel_ReportErrorMessage(message);
end;


end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function dX = GenericPBPKmodel_RHS(t,X,Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% initialize output vector
dX = zeros(size(X));

%%% model and indexing
model = Individual.model; S = model.S;

%%% variables (always use row vector notation)
C_pla = X(S.C_pla)'; C_int = X(S.C_int)';

V_tum = X(S.V_tum)';%interstitial tumor volume

%%% tissue volumes, blood flows, endosomal clearance etc.
V_pla            = model.V.pla;
V_int            = model.V.int;
Q                = model.Q.lymph;
K                = model.K.int;
sigma.vas        = model.sigma.vas;
CLint            = model.CLint.int;
CLpla            = model.CLpla.pla;




%%% infusion
infusion_rate = X(S.IVrate);



%%% -----------------------------------------------------------------------
%%% START OF ODEs
%%%
dA_int = zeros(size(C_int));
org = S.organs; pla = S.pla;
tum=S.Tum;orgexTum=S.organsExTum;

%%%Tumor growth
%%%
kgrowth   =Individual.model.tumorgrowth;%6.35e-5;%[1/min] Urva et al. 2010
dV_tum    =+kgrowth*V_tum;



%%% interstitial spaces
dA_int(orgexTum) = Q(orgexTum).*( (1-sigma.vas(orgexTum))*C_pla - C_int(orgexTum)./K(orgexTum) );

dA_int(tum) = Q(tum).*( (1-sigma.vas(tum))*C_pla - C_int(tum)./K(tum) ) -dV_tum*C_int(tum);


%%% plasma
dA_pla = sum( Q(org).*C_int(org)./K(org) ) ...
    - sum( Q(org).*(1-sigma.vas(org))*C_pla ) - CLpla*C_pla ...
    + infusion_rate;




%%% drug amount in IVbag for infusion
dA_IVbag = -infusion_rate;

%%% drug amount metabolized or excreted
dA_metab = +CLpla*C_pla + sum( CLint(org).*C_int(org) );


%%%
%%% END OF ODEs
%%% -----------------------------------------------------------------------

%%% converting amounts to concentrations
dC_pla = dA_pla./V_pla;

dC_int = zeros(size(C_int));
dC_int(orgexTum) = dA_int(orgexTum)./V_int(orgexTum);

dC_int(tum)      = dA_int(tum)./V_tum;


%%% output vector (always in column vector notation)
dX(S.C_pla) = dC_pla';
dX(S.C_int) = dC_int';
dX(S.IVbag) = dA_IVbag';
dX(S.metab) = dA_metab';
dX(S.V_tum) = dV_tum';



end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function Individual = GenericPBPKmodel_determineStandartOutput(Individual,sim)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model    = Individual.model;
pred     = Individual.pred;  info = Individual.info;
compound = Individual.drug.listOfDrugs{1};
mAb      = Individual.drug.(compound);


S = model.S;



%%% -----------------------------------------------------------------------
%%% assign predictions (in internal units)
%%%
info.pred.t         = 'simulation time in [min]';
pred.t              = sim.t;



%%% initialize all compartments with NaN and assign only to those
%%% compartments values that are part of the model topology
initialize.NaN = ones(size(pred.t))*S.initialize.org.NaN;
org = S.organs; pla = S.pla;
tum=S.Tum;orgexTum=S.organsExTum;

sim.X_pla = sim.X(:,S.C_pla);
sim.X_int = sim.X(:,S.C_int);
sim.X_Vtum= sim.X(:,S.V_tum);

%%% tumor interstitial volume
%%%
pred.V.tum          =sim.X_Vtum;

%%% plasma concentration
%%%
info.pred.C.pla      = 'concentration in plasma [nmol/L]';
pred.C.pla           = sim.X_pla;

%%% interstitial concentrations
%%%
info.pred.C.int      = 'concentration in interstitial space [nmol/L]';
pred.C.int           = initialize.NaN;
pred.C.int(:,org)    = sim.X_int(:,org);

%%% tissue concentrations
%%%
info.pred.C.tis      = 'concentration in tissue space [nmol/L]';
pred.C.tis           = initialize.NaN;
pred.C.tis(:,org)    = pred.C.int(:,org)*diag(model.SF.int2tis(org));



%%% various amounts
%%%
info.pred.A.pla      = 'amount in plasma [nmol]';
pred.A.pla           = pred.C.pla*model.V.pla;

info.pred.A.int           = 'amount in interstitial space [nmol]';
pred.A.int                = initialize.NaN;
pred.A.int(:,orgexTum)    = pred.C.int(:,orgexTum)*diag(model.V.int(orgexTum));
pred.A.int(:,tum)         = pred.C.int(:,tum).*pred.V.tum;


info.pred.A.tis      = 'amount in tissue space [nmol]';
pred.A.tis           = pred.A.int;

info.pred.A.body     = 'total amount in the body in [nmol]';
pred.A.body          = sum(pred.A.int(:,org),2)+pred.A.pla;

info.pred.A.GItract  = 'remaining amount in the GI tract in [nmol]';
pred.A.GItract       = sim.X(:,S.GItract);

info.pred.A.IVbag    = 'remaining amount in the IVbag in [nmol]';
pred.A.IVbag         = sim.X(:,S.IVbag);

info.pred.IVrate     = 'infusion rate in [nmol/min]';
pred.IVrate          = sim.X(:,S.IVrate);

info.pred.A.metab    = 'amount metabolized in [nmol]';
pred.A.metab         = sim.X(:,S.metab);





%%% -----------------------------------------------------------------------
%%% determine standard output in [mg/L] or [mg]
%%%
stdout = Individual.stdout; stdout.T = Individual.species.T; T = stdout.T;
initialize.NaN = ones(size(pred.t))*T.initialize.tissueDB.NaN; SF = model.SF;

info.stdout.t         = 'simulation time in [min]';
stdout.t              = pred.t;

%%% tissue concentrations
%%%
info.stdout.C.tis     = 'concentration in tissue space [mg/L]';
stdout.C.tis          = initialize.NaN;
for k = intersect(T.tissueDB,S.organs)
    stdout.C.tis(:,k) = SF.nmol_to_mg*pred.C.tis(:,k);
end;

%%% plasma concentrations
%%%
info.stdout.C.pla     = 'concentration in plasma [mg/L]';
stdout.C.pla          = SF.nmol_to_mg*pred.C.pla;

%%% correct for residual blood
%%%
if isfield(stdout,'correct_for_residual_blood') && strcmp(stdout.correct_for_residual_blood,'yes')
    
    species = Individual.species;
    V.vas   = species.V.vas; V.tis = species.V.tis; hct = species.hct;
    
    V.vas   = species.V.res;
    
    for org = intersect(T.tissueDB,S.organs)
        stdout.C.tis(:,org) = ( V.tis(org).*stdout.C.tis(:,org) + (1-hct)*V.vas(org)*stdout.C.pla )./ (V.vas(org)+V.tis(org));
    end;
    
else
    stdout.correct_for_residual_blood = 'no';
end;


%%% various amounts
%%%
info.stdout.A.body    = 'total amount in the body in [mg]';
stdout.A.body         = SF.nmol_to_mg*pred.A.body;

info.stdout.A.GItract = 'remaining amount in the GI tract in [mg]';
stdout.A.GItract      = SF.nmol_to_mg*pred.A.GItract;

info.stdout.A.IVbag   = 'remaining amount in the IVbag in [mg]';
stdout.A.IVbag        = SF.nmol_to_mg*pred.A.IVbag;

info.stdout.IVrate    = 'infusion rate in [mg/min]';
stdout.IVrate         = SF.nmol_to_mg*pred.IVrate;

info.stdout.A.metab   = 'amount metabolized in [mg]';
stdout.A.metab        = SF.nmol_to_mg*pred.A.metab;


%%% final assignment
%%%
Individual.info   = info;
Individual.pred   = pred;
Individual.stdout = stdout;

end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_normalizedConcentrationPlot(Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

pred = Individual.pred; model = Individual.model;

if ~isfield(pred,'plot_normalized_conc') || ~strcmp(pred.plot_normalized_conc,'yes')
    return;
end;

t = pred.t; S = Individual.model.S; id = Individual.id;



initialize.NaN = ones(size(pred.t))*S.initialize.org.NaN;
org = S.organs; pla = S.pla;

%%% define original and normalized concentrations
%%%
C  = initialize.NaN;
nC = initialize.NaN;

C(:,org)  = pred.C.int(:,org);
C(:,pla)  = pred.C.pla;
for k = org
    nC(:,k) = C(:,k) / model.ABC.int(k) ;
end;
nC(:,pla)= C(:,pla);


figure(100+id); clf
%%% plot all concentrations in a single plot
%%%
subplot(1,2,1); hold on;
for org = S.allCmts
    plot(t,C(:,org),S.style.(S.name{org}),'LineWidth',1.3);
end;
title(sprintf('Individual(%d), drug %s',id,Individual.drug.listOfDrugs{1}),'Fontsize',15);
xlabel('t[min]'); ylabel('C [nM]');
xlim([0, t(end)]); ylim([1e-3 1.5]*max(max(C)));
set(gca,'YScale','log'); hold off;


%%% plot all normalized concentrations in a single plot
%%%
subplot(1,2,2); hold on;
for org = S.allCmts
    plot(t,nC(:,org),S.style.(S.name{org}),'LineWidth',1.3);
end;
title(sprintf('Normalized drug concentration'),'Fontsize',15);
legend(S.name(S.tissueDB),'Location','SouthEast');
xlabel('t[min]'); ylabel('C [nM]');
xlim([0, t(end)]); ylim([1e-3 1.5]*max(max(nC)));
set(gca,'YScale','log'); hold off;

end



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function Individual = GenericPBPKmodel_specificGraphicalOutput(Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




end

%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
