%%% Version: December 12th, 2015
%%%
%%% This function specifies a simplfied PBPK model for mAb with
%%% extravasation-limited tissue distribution, including ENDOGENOUS IgG
%%% but no target, parameterized in terms of interstitial volumes, with non-linear
%%% clearance from plasma and from interstitial fraction of organs(FcRn dependent
%%% clearance), KD Model for definition of fraction unbound to FcRn

%%% DETAILS:
%%% KD Model for nonlinear CL dependent on FcRn-IgG interaction
%%% with CL from interstitial organs (tight and leaky)
%%%
%%% For references and citation, please see MAIN script.
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function Individual = GenericPBPKmodel_mAb_PBPK_11CMT_extravasLim_int_3Drug_2KD_CLint(Individual)

%%% check whether drug class is really sMD
%%%
GenericPBPKmodel_checkForModelApplicability(Individual);

%%% define model specific parameters
%%%
Individual = GenericPBPKmodel_defineModelParameters(Individual);

%%% simulate PBPK model
%%%
model = Individual.model; dosingEvent = Individual.pred.dosingEvent;
X0 = model.X0; sim.t = []; sim.X = [];

for d=1:length(dosingEvent)
    
    %%% administration details and simulation time span
    admin = dosingEvent(d).admin;
    tspan = dosingEvent(d).tspan;
    
    %%% account for dosing
    X0 = GenericPBPKmodel_drugAdministration(X0,model,admin);
    
    %%% solve system of ODEs 'GenericPBPKmodel_RHS' with initial conditions 'X0'
    [t,X] =   ode15s(@GenericPBPKmodel_RHS,tspan,X0',[],Individual);%ode45(@GenericPBPKmodel_RHS,tspan,X0',[],Individual);%%
    
    %%% store current output in sim structure
    sim.t = [sim.t; t]; sim.X = [sim.X; X];
    
    X0 = X(end,:)';
    
end;

%%% determine standard simulation output
%%%
Individual = GenericPBPKmodel_determineStandartOutput(Individual,sim);

%%% normalized concentration plot
%%%
GenericPBPKmodel_normalizedConcentrationPlot(Individual);

%%% graphical output specific to this model

GenericPBPKmodel_specificGraphicalOutput(Individual);

end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_checkForModelApplicability(Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

numberOfDrugs = length(Individual.drug.listOfDrugs);
if ~ismember(numberOfDrugs,[1,2,3])
    message = sprintf('Number of drugs (k=%d) not compatable with model >>%s<<',...
        numberOfDrugs,Individual.model.type);
    GenericPBPKmodel_ReportErrorMessage(message);
end;

listOfAllowedCompounds = {'mAb','IVIG','IgGendo'};
for k = 1:numberOfDrugs
    
    compound = Individual.drug.listOfDrugs{k};
    drug     = Individual.drug.(compound);
    
    if ismember(drug.class,listOfAllowedCompounds)
        listOfAllowedCompounds = setdiff(listOfAllowedCompounds,drug.class);
    elseif ismember(drug.name,listOfAllowedCompounds)
        listOfAllowedCompounds = setdiff(listOfAllowedCompounds,drug.name);
    else
        message = sprintf('Drugs choosen for dosing not compatible with model >>%s<<',...
            Individual.model.type,Individual.drug.(compound).class);
        GenericPBPKmodel_ReportErrorMessage(message);
        
    end;
    
end;

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function Individual = GenericPBPKmodel_defineModelParameters(Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model = Individual.model;

%%% -----------------------------------------------------------------------
%%% Define indexing
%%%

%%% tissue topology indexing
%%%
S = Individual.species.T;

S.allCmts  = [S.lun S.adi S.hea S.kid S.mus S.bon S.ski S.gut S.spl S.liv S.pla];
S.organs   = [S.lun S.adi S.hea S.kid S.mus S.bon S.ski S.gut S.spl S.liv      ];
S.plasma   = [                                                            S.pla];
S.tissueDB = S.allCmts; % specify which tissues are part of the general tissue DB

S.visOrg      = [S.gut S.spl S.liv];
S.nonvisOrg   = [S.lun S.adi S.hea S.kid S.mus S.bon S.ski];

S.tight       = [S.adi S.bon S.mus S.ski];
S.leaky       = [S.liv S.hea S.lun S.kid S.spl S.gut];

S.maxIndex.org = max(S.allCmts);
S.initialize.org.NaN   = NaN*ones(1,S.maxIndex.org);

%%% compartment indexing
%%%
S.C_pla = 1;                      % total plasma compartment
S.bolus = S.C_pla;                % i.v. bolus administration
S.C_int = (1:S.maxIndex.org) + 1; % interstitial compartments

S.maxIndex.cmt = max(S.C_int);

%%% dosing and metabolism indexing
%%%
S.GItract = 1 + S.maxIndex.cmt; % gastro-intestinal tract (for po dosing)
S.IVbag   = 2 + S.maxIndex.cmt; % IVbag (for infusion dosing)
S.IVrate  = 3 + S.maxIndex.cmt; % IVbag (for infusion dosing)
S.metab   = 4 + S.maxIndex.cmt; % metabolized drug

S.maxIndex.dosing = 4 + S.maxIndex.cmt;

%%% additional indexing
%%%
S.maxIndex.each = 0 + S.maxIndex.dosing;

%%% drug indexing
S.mAb    = [1:S.maxIndex.each];
S.IgG    = [1:S.maxIndex.each] + S.maxIndex.each;
S.IgGendo= [1:S.maxIndex.each] + S.maxIndex.each + S.maxIndex.each;

%%% define range of indexes also based on drug name
%%%
for k = 1:length(Individual.drug.listOfDrugs)
    
    compound = Individual.drug.listOfDrugs{k};
    drug     = Individual.drug.(compound);
    if strcmp(drug.class,'mAb')
        S.(compound) = S.mAb;
    elseif strcmp(drug.name,'IVIG')
        S.(compound) = S.IgG;
    elseif strcmp(drug.name,'IgGendo')
        S.(compound) = S.IgGendo;
    end;
    
end;

%%% additional indexing
%%%
S.maxIndex.all = 3*S.maxIndex.each;


%%% -----------------------------------------------------------------------
%%% Define PBPK parameters
%%%

org = S.organs; pla = S.pla;
species  = Individual.species;

compound = Individual.drug.listOfDrugs{1};
mAb      = Individual.drug.(compound);

%%% tissue volumes
%%%
V.tis          = S.initialize.org.NaN;
V.tis(org)     = species.V.tis(org);

fVtis.int      = S.initialize.org.NaN;
fVtis.int(org) = species.fVtis.int(org);

V.int          = S.initialize.org.NaN;
V.int(org)     = fVtis.int(org) .* V.tis(org);

V.pla = (1-species.hct) * species.V.blood;

%%% volume for i.v. bolus dosing
V.iv_bolus = V.pla;

%%% blood flows (ensure closed circulatory system!)
%%%
Q.blo = S.initialize.org.NaN;




Q.blo(org)    = species.fQco(org) * species.co;
Q.blo(S.lun)  = species.co;

%%% peripheral plasma flows
Q.pla = (1-species.hct) * Q.blo;

%%% lymph flow as fraction of peripheral plasma and blood flow
fLymph = S.initialize.org.NaN;
fLymph(S.visOrg)    = 0.02;
fLymph(S.nonvisOrg) = 0.04;



Q.lymph = fLymph.*Q.pla;

Q.lymph(pla) = sum(Q.lymph(org));



%%% define vascular reflection coefficients
sigma.vas = S.initialize.org.NaN;

%%%%sigmas Brown Mouse (set manually, fraction (1-sigma.vas) is reduced by
%%%%factor 0.5 from Fronton Mouse and skin put to another group

sigma.vas([S.liv S.spl S.gut])            = 0.95;
sigma.vas([S.hea S.ski S.kid S.lun])      = 0.975;
sigma.vas([S.mus S.adi S.bon])            = 0.99;


sigma.pla = 1 - sum( Q.lymph(org).*(1-sigma.vas(org)) ) /Q.lymph(pla);

%%% drug specific parameters starting below
%%%

ABC.tis      = S.initialize.org.NaN;
ABC.tis(org) = mAb.ABC.tis(org);


ABC.pla      = mAb.ABC.tis(pla);

%%% define elimination-corrected tissue partition coefficients
eK.tis       = S.initialize.org.NaN;
eK.tis(org)  = ABC.tis(org)./(1-sigma.vas(org));
eK.pla       = ABC.pla./(1-sigma.pla);

%%% blood-to-plasma ratio assuming that mAb do not distribute or bind to
%%% erythrocytes
BP = (1-species.hct);


%%%%%% tissue-specific extraction ratio for two groups of tissues
E.tis      = S.initialize.org.NaN;

%final estimates with OFV= 4.798e-01 for KD_murine=4.8 and KD_human=4.8
%based on plasma data only
if isfield(model,'WTmice') && strcmp(model.WTmice,'yes')
    E.tis([S.tight])             = 1.880e-02;
    E.tis( [S.leaky])            = 3.019e-03  ;
end;

%final estimates with OFV= 2.352e+00 for KD_murine=4.8 and KD_human=4.8
%based on plasma data only
if isfield(model,'SCIDmice') && strcmp(model.SCIDmice,'yes')
    E.tis([S.tight])             =1.435e-05;
    E.tis( [S.leaky])            = 2.535e-06 ;
end



if strcmp(Individual.estimation.modus,'on')
    E.tis      = S.initialize.org.NaN;
    
    E_tight    = Individual.estimation.E_tight;
    E_leaky    = Individual.estimation.E_leaky;
    
    
    E.tis(S.tight)             = E_tight;
    E.tis(S.leaky)             = E_leaky;
end;



%%%%%% Define CL from plasma
fu=1;


if isfield(model,'WTmice') && strcmp(model.WTmice,'yes')
    CLpla.tis(pla) = 1/fu*8.37e-08  -  ...
        sum( sum (E.tis([S.tight]).* Q.lymph([S.tight]).*(1-sigma.vas([S.tight])))...
        + sum (E.tis([S.leaky]).* Q.lymph([S.leaky]).*(1-sigma.vas([S.leaky]))));
end

if isfield(model,'SCIDmice') && strcmp(model.SCIDmice,'yes')
    CLpla.tis(pla) = 1/fu*2.65e-07 -  ...
        sum( sum (E.tis([S.tight]).* Q.lymph([S.tight]).*(1-sigma.vas([S.tight])))...
        + sum (E.tis([S.leaky]).* Q.lymph([S.leaky]).*(1-sigma.vas([S.leaky]))));
end


%%%scaling of clearance with BW
CLpla.tis(pla) = CLpla.tis(pla) *(Individual.species.BW/0.025);

%%%check
if CLpla.tis(pla) < 0
    CLpla.tis(pla) = 0;
end;

%%%%%% steady state endogenous IgG level in plasma
if isfield(model,'WTmice') && strcmp(model.WTmice,'yes')
    IgGendoSS_pla      = 1.47e4;                  % in [nM] steady-state mouse IgG conc for mice
end


%%% account for SCID status
if isfield(model,'SCIDmice') && strcmp(model.SCIDmice,'yes')
    IgGendoSS_pla =66e-02 ;
end




%%%%%% tissue to interstitial volumes
SF   = mAb.SF;
SF.int2tis = species.fVtis.int;               %Umrechnung tissue conc to interstitial conc
SF.tis2int = 1./SF.int2tis;

ABC.int      = S.initialize.org.NaN;
ABC.int(org) = SF.tis2int(org).*ABC.tis(org);


%%%%%% steady state endogenous IgG level in tissues

IgGendoSS_int      = S.initialize.org.NaN;
IgGendoSS_int(org) = ABC.int(org)*IgGendoSS_pla;  %muss groesser als MAX sein damit fu nicht negativ


%%% FcRn binding capacity
%%%
FcRn.max=S.initialize.org.NaN;


%final estimates with OFV= 4.798e-01 for KD_murine=4.8 and KD_human=4.8 and
%based on plasma data only
if isfield(model,'WTmice') && strcmp(model.WTmice,'yes')
    FcRn.max([S.tight])             =4.736e+03;
    FcRn.max([S.leaky])             =7.710e+03  ;
end;


%final estimates with OFV= 2.352e+00 for KD_murine=4.8 and KD_human=4.8 and
%based on plasma data only
if isfield(model,'SCIDmice') && strcmp(model.SCIDmice,'yes')
    FcRn.max([S.tight])             = 2.122e-01;
    FcRn.max([S.leaky])             =3.504e-01;
end;



%%% estimation mode
if strcmp(Individual.estimation.modus,'on')
    FcRn.max      = S.initialize.org.NaN;
    
    FcRn_tight     = Individual.estimation.FcRn_tight;
    FcRn_leaky     = Individual.estimation.FcRn_leaky;
    
    FcRn.max([S.tight])           = FcRn_tight;
    FcRn.max(S.leaky)             = FcRn_leaky;
end;


if isfield(model,'KD48foralldrugs') && strcmp(model.KD48foralldrugs,'yes')
    
    KD_murine      =4.8; %[nM]
    KD_human       =4.8;
end

%%% Define fraction unbound IgG to FcRn for baseline endogenous IgG in
%%% organs
%%%
fu = S.initialize.org.NaN;

%%%define unbound FcRn for baseline endogenous IgG in WT mice in organs(murine)
FcRn_eff(org)= FcRn.max(org) - IgGendoSS_int(org) - KD_murine;

FcRn_u(org)  = 1/2* (FcRn_eff(org) + sqrt((FcRn_eff(org)).^2 +4*KD_murine*FcRn.max(org)));

%%%define fraction unbound to FcRn based on KD
fu(org)    = KD_murine./(KD_murine+FcRn_u(org));


%%% partition coefficients
%%%
K.tis(org) = eK.tis(org)./(1-E.tis(org));
K.pla      = 1;

%%% define intrinsic tissue clearance
%%%
CLint.tis = S.initialize.org.NaN;
CLint.tis(org) = Q.lymph(org)./K.tis(org).*E.tis(org)./(1-E.tis(org));


%%%change tissue to interstitial parameters

SF = mAb.SF;
SF.int2tis = species.fVtis.int;
SF.tis2int = 1./SF.int2tis;

K.int        = S.initialize.org.NaN;
K.int(org)   = SF.tis2int(org).*K.tis(org);


E.int        = S.initialize.org.NaN;
E.int(org)   = E.tis(org); % note: scaling factors cancel out!


CLint.int       = S.initialize.org.NaN;
CLint.int(org)  = Q.lymph(org)./K.int(org).*E.int(org)./(1-E.int(org));



%%% production rate of endogenous IgG
IgGproduction_rate =  (CLpla.tis(S.pla) + sum ((CLint.int(org).* ABC.int(org))))* IgGendoSS_pla;


if IgGproduction_rate < 0
    message= sprintf('WARNING IgGproduction_rate = %1.3e turns negative\n',IgGproduction_rate);
    GenericPBPKmodel_ReportErrorMessage(message);
end;

%%% account for FcRn status
if isfield(model,'FcRn_status') && strcmp(model.FcRn_status,'knock-out')
    FcRn.max([S.tight])             = 0;
    FcRn.max([S.leaky])             = 0;
end;



%%%%intrinsic Clearance unbound
CLint_u      =  S.initialize.org.NaN;
CLint_u(org) =  CLint.int(org)./fu(org) ;

%%%scaling of clearance with BW
CLint_u(org) = CLint_u(org) *(Individual.species.BW/0.025);


%%% compute ABCs values and effective flows
Leff.lymph   = (1-sigma.vas).*Q.lymph;

%%% initial condition of ODEs
%%% set steady state concentrations for endogenous IgG
IgGendo.X0               = zeros(size(S.IgGendo));
IgGendo.X0(S.C_pla)      = IgGendoSS_pla;
IgGendo.X0(S.C_int(org)) = ABC.int(org)*IgGendoSS_pla;


X0            = zeros(1,S.maxIndex.all);

X0(S.IgGendo) = IgGendo.X0;



%%% -----------------------------------------------------------------------
%%% Assign model parameters
%%%

model.S      = S;
model.V      = V;
model.Q      = Q;
model.K      = K;
model.E      = E;
model.eK     = eK;
model.BP     = BP;
model.fLymph = fLymph;
model.sigma  = sigma;
model.CLint  = CLint;
model.CLint_u= CLint_u;
model.CLpla  = CLpla;

model.FcRn               = FcRn;
model.IgGproduction_rate = IgGproduction_rate;
model.ABC                = ABC;
model.L                  = Leff;
model.SF                 = SF;
model.X0                 = X0;

model.KD_murine          = KD_murine;
model.KD_human           = KD_human;

Individual.model = model;

%%% -----------------------------------------------------------------------
%%% double check, whether there is some parameter values resulting in
%%% NaN values in the ODE
%%%
dX = GenericPBPKmodel_RHS(0,Individual.model.X0',Individual);
if sum(isnan(dX))
    message = sprintf('Some parameter values of id=%d cause the RHS of the ODE to have NaN values!',Individual.id);
    GenericPBPKmodel_ReportErrorMessage(message);
end;


end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [X0] = GenericPBPKmodel_drugAdministration(X0,model,admin)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% time and dose are expected to be in internal units [min] and [nmol]


S = model.S;

for k=1:length(admin)
    
    dose     = admin(k).dose; %hier dose in nmol ; wurde umgerechnet von mg in nmol
    
    
    compound = admin(k).drug;
    
    %%% part of vector X0_cmp corresponds to the compound being dosed
    X0_cmp = X0(S.(compound));
    
    switch admin(k).route
        
        
        case 'iv_bolus';
            X0_cmp(S.bolus) = X0_cmp(S.bolus) + dose/model.V.iv_bolus;
            
            
        case 'iv_infusion';
            if dose>0 % start of iv infusion
                X0_cmp(S.IVbag)  = X0_cmp(S.IVbag) + dose;
                X0_cmp(S.IVrate) = X0_cmp(S.IVrate) + admin(k).infusionRate;
            else      % end of iv infusion
                X0_cmp(S.IVrate) = X0_cmp(S.IVrate) - admin(k).infusionRate;
            end;
            
        case 'po';
            %%% double check that p.o. dose = 0
            message = 'po administration of mAb/pAb not supported!';
            GenericPBPKmodel_ReportErrorMessage(message);
            
    end;
    
    %%% re-assign the updated part to X0
    X0(S.(compound)) = X0_cmp;
    
end;

if sum(isnan(X0))
    message = sprintf('Some parameter values of id=%d cause the initial condition X0 of the ODE to have NaN values!',Individual.id);
    GenericPBPKmodel_ReportErrorMessage(message);
end;


end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function dX = GenericPBPKmodel_RHS(t,X,Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% initialize output vector
dX = zeros(size(X));

%%% model and indexing
model = Individual.model; S = model.S;

%%% variables (always use row vector notation)
mAb.X = X(S.mAb)'; IgG.X = X(S.IgG)';IgGendo.X = X(S.IgGendo)';

%%% assign mAb and (endogenous) IgG concentrations
mAb.C_pla = mAb.X(S.C_pla); mAb.C_int = mAb.X(S.C_int);
IgG.C_pla = IgG.X(S.C_pla); IgG.C_int = IgG.X(S.C_int);
IgGendo.C_pla = IgGendo.X(S.C_pla); IgGendo.C_int = IgGendo.X(S.C_int);

%%% tissue volumes, blood flows, endosomal clearance etc.
V_pla    = model.V.pla;
V_int    = model.V.int;
Q        = model.Q.lymph;
K        = model.K.int;
sigma    = model.sigma.vas;
CLpla    = model.CLpla;
CLint_u  = model.CLint_u;
FcRn     = model.FcRn.max;

IgG.production_rate = model.IgGproduction_rate;
KD_murine           = model.KD_murine;
KD_human            = model.KD_human;


%%% determine (int) clearance based on total IgG concentration i.e. mAb + IVIG +IgGendo concentration

mAb_int=  mAb.C_int(S.organs)  + IgGendo.C_int(S.organs);
IgG_int=  IgG.C_int(S.organs);

FcRneff1= KD_murine +mAb_int+IgG_int+KD_human-FcRn(S.organs);

FcRneff2= KD_murine.*IgG_int+mAb_int.*KD_human+KD_human*KD_murine-FcRn(S.organs).*KD_human-FcRn(S.organs).*KD_murine;

FcRneff3= -FcRn(S.organs).*KD_murine*KD_human;

p = ((3*FcRneff2)-(FcRneff1).^2)/3;
q = (2*(FcRneff1).^3)/27-(FcRneff1.*FcRneff2)/3+FcRneff3;

rho = sqrt(-(p.^3)/27);
phi = acos(-q./(2.*rho));

y1    = 2.*(rho.^(1/3)).*cos(phi./3);
FcRn_u(S.organs)= y1-(FcRneff1./3);



%%%define fraction unbound to FcRn based on KD
fu_mAb      = KD_murine./(KD_murine+FcRn_u(S.organs));
fu_IgG      = KD_human./(KD_human+FcRn_u(S.organs));

CLint_mAb = CLint_u(S.organs).*fu_mAb ;
CLint_IgG = CLint_u(S.organs).*fu_IgG ;

%%% infusion
mAb.infusion_rate = mAb.X(S.IVrate);
IgG.infusion_rate = IgG.X(S.IVrate);


%%% -----------------------------------------------------------------------
%%% START OF ODEs
%%%
mAb.VdC_int = zeros(size(mAb.C_int)); IgG.VdC_int = zeros(size(IgG.C_int)); org = S.organs;


mAb.VdC_int(org) = Q(org).*( (1-sigma(org))*mAb.C_pla - mAb.C_int(org)./K(org) )...
    - CLint_mAb.* mAb.C_int(org);


IgG.VdC_int(org) = Q(org).*( (1-sigma(org))*IgG.C_pla - IgG.C_int(org)./K(org) )...
    - CLint_IgG.* IgG.C_int(org);

IgGendo.VdC_int(org) = Q(org).*( (1-sigma(org))*IgGendo.C_pla - IgGendo.C_int(org)./K(org) )...
    - CLint_mAb.* IgGendo.C_int(org);


%%% plasma
mAb.VdC_pla = mAb.infusion_rate ...
    + sum( Q(org).*mAb.C_int(org)./K(org) ) ...
    - sum( Q(org).*(1-sigma(org))*mAb.C_pla ) - CLpla.tis(S.pla)*mAb.C_pla;

IgG.VdC_pla =  IgG.infusion_rate ...
    + sum( Q(org).*IgG.C_int(org)./K(org) ) ...
    - sum( Q(org).*(1-sigma(org))*IgG.C_pla ) - CLpla.tis(S.pla)*IgG.C_pla;

IgGendo.VdC_pla =  IgG.production_rate ...
    + sum( Q(org).*IgGendo.C_int(org)./K(org) ) ...
    - sum( Q(org).*(1-sigma(org))*IgGendo.C_pla ) - CLpla.tis(S.pla)*IgGendo.C_pla;

%%% drug amount in IVbag for infusion
mAb.dA_IVbag = -mAb.infusion_rate;
IgG.dA_IVbag = -IgG.infusion_rate;



mAb.dA_metab = +CLpla.tis(S.pla)*mAb.C_pla + sum (CLint_mAb.*mAb.C_int(org));
IgG.dA_metab = (+CLpla.tis(S.pla)*IgG.C_pla +  sum (CLint_IgG.*IgG.C_int(org)) );
IgGendo.dA_metab = +CLpla.tis(S.pla)*IgGendo.C_pla +  sum (CLint_mAb.*IgGendo.C_int(org)) - IgG.production_rate ;
%%%
%%% END OF ODEs
%%% -----------------------------------------------------------------------


%%% converting amounts to concentrations
mAb.dC_pla = mAb.VdC_pla./V_pla;
IgG.dC_pla = IgG.VdC_pla./V_pla;
IgGendo.dC_pla = IgGendo.VdC_pla./V_pla;

mAb.dC_int = zeros(size(mAb.C_int)); IgG.dC_int = zeros(size(IgG.C_int));IgGendo.dC_int = zeros(size(IgGendo.C_int));
mAb.dC_int(org) = mAb.VdC_int(org)./V_int(org);
IgG.dC_int(org) = IgG.VdC_int(org)./V_int(org);
IgGendo.dC_int(org) = IgGendo.VdC_int(org)./V_int(org);

%%% output vector (always in column vector notation)
mAb.dX = zeros(size(mAb.X)); IgG.dX = zeros(size(IgG.X));
mAb.dX(S.C_pla) = mAb.dC_pla';
mAb.dX(S.C_int) = mAb.dC_int';
mAb.dX(S.IVbag) = mAb.dA_IVbag';
mAb.dX(S.metab) = mAb.dA_metab';

IgG.dX(S.C_pla) = IgG.dC_pla';  %fehler da prodrate fuer alle
IgG.dX(S.C_int) = IgG.dC_int';
IgG.dX(S.IVbag) = IgG.dA_IVbag';
IgG.dX(S.metab) = IgG.dA_metab';

IgGendo.dX(S.C_pla) = IgGendo.dC_pla';
IgGendo.dX(S.C_int) =IgGendo.dC_int';
IgGendo.dX(S.metab) = IgGendo.dA_metab';

dX(S.mAb) = mAb.dX;
dX(S.IgG) = IgG.dX;
dX(S.IgGendo) = IgGendo.dX;

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function Individual = GenericPBPKmodel_determineStandartOutput(Individual,sim)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model    = Individual.model;
pred     = Individual.pred;  info = Individual.info;

S = model.S;

%%% initialize all compartments with NaN and assign only to those
%%% compartments values that are part of the model topology
initialize.NaN = ones(size(sim.t))*S.initialize.org.NaN;
org = S.organs;

list_of_drugs = {'mAb','IgG', 'IgGendo'};

for k=1:length(list_of_drugs)
    
    compound = list_of_drugs{k};
    
    %%% -----------------------------------------------------------------------
    %%% assign predictions (in internal units)
    %%%
    info.pred.t         = 'simulation time in [min]';
    pred.(compound).t       = sim.t;
    
    sim.(compound).X     = sim.X(:,S.(compound));
    sim.(compound).X_pla = sim.(compound).X(:,S.C_pla);
    sim.(compound).X_int = sim.(compound).X(:,S.C_int);
    
    %%% plasma concentration
    %%%
    info.pred.(compound).C.pla     = sprintf('concentration of %s in plasma [nmol/L]',compound);
    pred.(compound).C.pla          = sim.(compound).X_pla;
    
    %%% interstitial concentrations
    %%%
    info.pred.(compound).C.int     = sprintf('concentration of %s in interstitial space [nmol/L]',compound);
    pred.(compound).C.int          = initialize.NaN;
    pred.(compound).C.int(:,org)   = sim.(compound).X_int(:,org);
    
    %%% tissue concentrations
    %%%
    info.pred.(compound).C.tis     = sprintf('concentration of %s in tissue space [nmol/L]',compound);
    pred.(compound).C.tis          = initialize.NaN;
    pred.(compound).C.tis(:,org)   = pred.(compound).C.int(:,org)*diag(model.SF.int2tis(org));
    
    %%% various amounts
    %%%
    info.pred.(compound).A.pla     = sprintf('amount of %s in plasma [nmol]',compound);
    pred.(compound).A.pla          = pred.(compound).C.pla*model.V.pla;
    
    info.pred.(compound).A.int     = sprintf('amount of %s in interstitial space [nmol]',compound);
    pred.(compound).A.int          = initialize.NaN;
    pred.(compound).A.int(:,org)   = pred.(compound).C.int(:,org)*diag(model.V.int(org));
    
    info.pred.(compound).A.tis     = sprintf('amount of %s in tissue space [nmol]',compound);
    pred.(compound).A.tis          = pred.(compound).A.int;
    
    info.pred.(compound).A.body    = sprintf('total amount of %s in the body in [nmol]',compound);
    pred.(compound).A.body         = sum(pred.(compound).A.int(:,org),2)+pred.(compound).A.pla;
    
    info.pred.(compound).A.GItract = sprintf('remaining amount of %s in the GI tract in [nmol]',compound);
    pred.(compound).A.GItract      = sim.(compound).X(:,S.GItract);
    
    info.pred.(compound).A.IVbag   = sprintf('remaining amount of %s in the IVbag in [nmol]',compound);
    pred.(compound).A.IVbag        = sim.(compound).X(:,S.IVbag);
    
    info.pred.(compound).IVrate    = sprintf('%s infusion rate in [nmol/min]',compound);
    pred.(compound).IVrate         = sim.(compound).X(:,S.IVrate);
    
    info.pred.(compound).A.metab   = sprintf('amount of %s metabolized in [nmol]',compound);
    pred.(compound).A.metab        = sim.(compound).X(:,S.metab);
    
end;


%%% fraction unbound of total IgG
mAb_int  = pred.mAb.C.int;

IgG_int  = pred.IgG.C.int;
IgGendo_int  = pred.IgGendo.C.int;

mAb_tot  = mAb_int + IgGendo_int;

FcRn_max = model.FcRn.max;
KD_human    = model.KD_human;
KD_murine   = model.KD_murine;

max=repmat(FcRn_max,length(mAb_tot),1);

FcRneff1= KD_murine+mAb_tot+IgG_int+KD_human-max;


FcRneff2= KD_murine.*IgG_int+mAb_tot.*KD_human+KD_human*KD_murine-max.*KD_human-max.*KD_murine;

FcRneff3= -max.*KD_murine*KD_human;

p = ((3.*FcRneff2)-(FcRneff1).^2)./3;
q = (2.*(FcRneff1).^3)./27-(FcRneff1.*FcRneff2)./3+FcRneff3;


rho = sqrt(-(p.^3)./27);
phi = acos(-q./(2.*rho));

y1    = 2.*(rho.^(1/3)).*cos(phi./3);
FcRn_u= y1-(FcRneff1./3);



%%%define fraction unbound to FcRn based on KD
fu_mAb      = KD_murine./(KD_murine+FcRn_u);
fu_IgG      = KD_human./(KD_human+FcRn_u);



pred.mAb.fu = fu_mAb;
pred.IgG.fu = fu_IgG;








%%% -----------------------------------------------------------------------
%%% determine standard output in [mg/L] or [mg]
%%%
stdout = Individual.stdout; stdout.T = Individual.species.T; T = stdout.T;
initialize.NaN = ones(size(sim.t))*T.initialize.tissueDB.NaN; SF = model.SF;

%%% which compound to report on in the standard output,
%%% either 'mAb' or 'IgG
%%%
if strcmp (Individual.stdout.compound, 'IgG');
    
    compound = 'IgG';
else
    if strcmp (Individual.stdout.compound, 'IgGendo');
        compound = 'IgGendo';  %% default:%compound = 'mAb';
    else
        compound = 'mAb';
    end
end

info.stdout.compound  = 'compound shown in standard output';
stdout.compound       = compound;

info.stdout.t         = 'simulation time in [min]';
stdout.t              = pred.(compound).t;

%%% tissue concentrations
%%%
info.stdout.C.tis     = 'concentration in tissue space [mg/L]';
stdout.C.tis          = initialize.NaN;
for k = intersect(T.tissueDB,S.organs)
    stdout.C.tis(:,k) = SF.nmol_to_mg*pred.(compound).C.tis(:,k);
end;

%%% plasma concentrations
%%%
info.stdout.C.pla     = 'concentration in plasma [mg/L]';
stdout.C.pla  = SF.nmol_to_mg*pred.(compound).C.pla;

%%% correct for residual blood
%%%
if isfield(stdout,'correct_for_residual_blood') && strcmp(stdout.correct_for_residual_blood,'yes')
    
    species = Individual.species;
    V.vas   = species.V.vas; V.tis = species.V.tis; hct = species.hct;
    
    V.vas   = species.V.res;
    
    for org = intersect(T.tissueDB,S.organs)
        stdout.C.tis(:,org) = ( V.tis(org).*stdout.C.tis(:,org) + (1-hct)*V.vas(org)*stdout.C.pla )./( V.vas(org)+V.tis(org) );
    end;
    
else
    stdout.correct_for_residual_blood = 'no';
end;


%%% various amounts
%%%
info.stdout.A.body    = 'total amount in the body in [mg]';
stdout.A.body         = SF.nmol_to_mg*pred.(compound).A.body;

info.stdout.A.GItract = 'remaining amount in the GI tract in [mg]';
stdout.A.GItract      = SF.nmol_to_mg*pred.(compound).A.GItract;

info.stdout.A.IVbag   = 'remaining amount in the IVbag in [mg]';
stdout.A.IVbag        = SF.nmol_to_mg*pred.(compound).A.IVbag;

info.stdout.IVrate    = 'infusion rate in [mg/min]';
stdout.IVrate         = SF.nmol_to_mg*pred.(compound).IVrate;

info.stdout.A.metab   = 'amount metabolized in [mg]';
stdout.A.metab        = SF.nmol_to_mg*pred.(compound).A.metab;


%%% final assignment
%%%
Individual.info   = info;
Individual.pred   = pred;
Individual.stdout = stdout;

end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_normalizedConcentrationPlot(Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


if ~isfield(Individual.pred,'plot_normalized_conc') || ~strcmp(Individual.pred.plot_normalized_conc,'yes')
    return;
end;



compound = 'mAb';

pred = Individual.pred.(compound); model = Individual.model;

t = pred.t; S = Individual.model.S; id = Individual.id;

initialize.NaN = ones(size(pred.t))*S.initialize.org.NaN;
org = S.organs; pla = S.pla;

%%% define original and normalized concentrations
%%%
C  = initialize.NaN;
nC = initialize.NaN;

C(:,org)  = pred.C.int(:,org);
C(:,pla)  = pred.C.pla;
for k = org
    nC(:,k) = C(:,k) / model.ABC.int(k) ;
end;
nC(:,pla)= C(:,pla);


figure(100+id); clf
%%% plot all concentrations in a single plot
%%%
subplot(1,2,1); hold on;
for org = S.allCmts
    plot(t,C(:,org),S.style.(S.name{org}),'LineWidth',1.3);
end;
title(sprintf('Individual(%d), drug %s',id,Individual.drug.listOfDrugs{1}),'Fontsize',15);
xlabel('t[min]'); ylabel('C [nM]');
xlim([0, t(end)]); ylim([1e-3 1.5]*max(max(C)));
set(gca,'YScale','log'); hold off;


%%% plot all normalized concentrations in a single plot
%%%
subplot(1,2,2); hold on;
for org = S.allCmts
    plot(t,nC(:,org),S.style.(S.name{org}),'LineWidth',1.3);
end;
title(sprintf('Normalized drug concentration'),'Fontsize',15);
legend(S.name(S.tissueDB),'Location','SouthEast');
xlabel('t[min]'); ylabel('C [nM]');
xlim([0, t(end)]); ylim([1e-3 1.5]*max(max(nC)));
set(gca,'YScale','log'); hold off;

end



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function Individual = GenericPBPKmodel_specificGraphicalOutput(Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% define here specific output for this model

id = Individual.id;


%%% plot fraction unbound over time
plot_for_id = [id]; % example: plot_for_id = [1] change in virtual population to id+1 therefore write here [id] instead of a number [2]

if ~ismember(id,plot_for_id)
    return;
end;

model = Individual.model; S = model.S;

compound = 'mAb';
pred  = Individual.pred.(compound); t = pred.t/(60*24);

figure(200+id);
hold on;


for org=S.organs
    plot(t,pred.fu(:,org),S.style.(S.name{org}),'LineWidth',2);
end
xlabel('t [days]'); ylabel('fraction not bound to FcRn in endosomes');
legend(S.name(S.organs),'Location','best');
title(sprintf('ID = %d',Individual.id));


xlim([0 100]);
ylim([-0.1 1.1]);

hold on;



end


%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
