%%% Version: October 31st, 2016
%%%
%%% This function simulates the models of all individuals that comprise the
%%% virtual population and determines the objective function value (sum of
%%% squared error)
%%%
%%% For references and citation, please see MAIN script.
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


function [individual,evaluation] = GenericPBPKmodel_evaluatePBPKparameters_mAb_int(individual)


[individual,listOfIds] = GenericPBPKmodel_checkForEvaluationModusOn(individual);

%%% if there is nothing to evaluate --> return, otherwise report
%%%
if isempty(listOfIds.evaluation)
    evaluation = []; return;
else
    fprintf('\n   (*) Evaluate PBPK parameters(s) ...\n');
end;

%%% load experimental data
%%%
for id=listOfIds.expData
    individual(id) = GenericPBPKmodel_ExperimentalData(individual(id));
end;

%%% determine time points for simulation to compare to experimental data
%%%
for id=listOfIds.evaluation
    
    tpoints = [];
    for k = individual(id).evaluation.expIds
        tpoints = union(tpoints, individual(k).pred.t);
    end;
    individual(id).pred.timeSpan = tpoints';
    individual(id).pred.timeUnit = 'min';  %%% internal time unit
    
end;

%%% determine dosing events
%%%
for id=listOfIds.evaluation
    
    dosingEvent = GenericPBPKmodel_determineDosingEvents(individual(id));
    individual(id).pred.dosingEvent = dosingEvent;
    
end;


[output] = determine_residuals(individual,listOfIds);

evaluation.listOfIds = listOfIds;
evaluation.output    = output;

%%% report about results
%%%
fprintf('\n   Results of the evaluation process:\n');
fprintf('\n   (a) sum of squared error: %2.6e',evaluation.output);
fprintf('\n\n')


end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [individual, listOfIds] = GenericPBPKmodel_checkForEvaluationModusOn(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

listOfIds.evaluation = [];

N_ids = length(individual);
for id = 1:N_ids
    
    %%% report about characteristics of the individual
    individual(id).id = id;
    
    %%% check, if field 'estimation.modus' exists and is equal to 'on'
    %%%
    if isfield(individual(id),'evaluation') && isfield(individual(id).evaluation,'modus') && strcmp(individual(id).evaluation.modus,'on')
        
        listOfIds.evaluation = [listOfIds.evaluation id];
        
        %%% check, if expIds really only contains ids which are part of the
        %%% virtual population
        %%%
        if ~all(ismember(individual(id).evaluation.expIds,[1:N_ids]))
            message = sprintf('individual(%d).evaluation.expIds contains ids that are not part of the virtual population!',id);
            GenericPBPKmodel_ReportErrorMessage(message);
        end;
        
        %%% check, if expIds really only contains ids with experimental data
        %%%
        for expId = individual(id).evaluation.expIds
            
            if ~strcmp(individual(expId).model.type,'ExperimentalData')
                message = sprintf('individual(%d).evaluation.expIds contains ids that do not contain experimental data!',id);
                GenericPBPKmodel_ReportErrorMessage(message);
            end;
            
        end;
        %%% field 'estimation.modus' does not exist or is equal to 'off'
        %%%
    else
        individual(id).evaluation.modus = 'off';
    end;
    
end;

%%% compile list of ids with experimental data
%%%
listOfIds.expData = [];
for id = 1:N_ids
    
    if strcmp(individual(id).model.type,'ExperimentalData')
        listOfIds.expData = [listOfIds.expData id];
    end;
    
end;
end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [res,n] = determine_residuals(individual,listOfIds)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% simulate individual models
%%%
for id=listOfIds.evaluation
    
    individual(id) = eval(sprintf('GenericPBPKmodel_%s(individual(id))',individual(id).model.type));
    
end;


%%% compare predictions to experimental data based on standard output,
%%% i.e., vascular plasma and tissue concentration
%%%


%%%log transform data and predictions when using no weighing scheme for SSE
%%%
logtransformation=1;
res = 0; n = 0;

for id=listOfIds.evaluation
   
    S = individual(id).model.S;
    pla = S.pla;
    
    if logtransformation==1;
        t_pred = log(individual(id).stdout.t);
    else
        t_pred = individual(id).stdout.t;
    end
 
    for k=individual(id).evaluation.expIds;

        if logtransformation==1;
            t_exp = log(individual(k).stdout.t) ;
 
        else
            t_exp = individual(k).stdout.t;
        end;
        [~,ind_pred,ind_exp] = intersect(t_pred,t_exp);
 
        %%% compute residuals for plasma
        %%%
        if logtransformation==1;
            C_pred  = log(individual(id).stdout.C.pla(ind_pred));
            C_exp   = log(individual( k).stdout.C.pla(ind_exp));
            [ind,~] = find(~isnan(C_exp)); % only none NaN values
        else
            C_pred  = individual(id).stdout.C.pla(ind_pred);
            C_exp   = individual( k).stdout.C.pla(ind_exp);
            [ind,~] = find(~isnan(C_exp)); % only none NaN values
        end;
        
        
        
        res     = res + compute_residuals(C_pred(ind),C_exp(ind));
        n       = n + length(ind);
        
        
    end;
    
    
    
end
end






%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function res = compute_residuals(C_pred,C_exp)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


logtransformation=1;


if logtransformation==1;
    res = sum( ( (C_pred-C_exp) ).^2 );
else
    res = sum( ( (C_pred-C_exp)./C_pred ).^2 );
end


end











%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

