%%% Version: Saskia May 29th, 2015
%%%
%%% This function compiles the reference Individuals, combinding anatomical
%%% and physiological data, drug-specific data and study information
%%%
%%% For references and citation, please see MAIN script.
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function individual = GenericPBPKmodel_assignSpeciesData(individual)

fprintf('\n   (a) Assigning anatomical and physiological data');

%%% load species database
%%%
mammals = GenericPBPKmodel_speciesDatabase;

message = 'no error';

for id = 1:length(individual)
    
    species = individual(id).species;
    
    switch species.type
        
        case 'human';
            
            %%% check for age
            if ismember(species.age,[0,1,5,10,15,35])
                subtype = sprintf('human%d',species.age);
            else
                message = 'Unknown age for human'; GenericPBPKmodel_ReportErrorMessage(message);
            end;
            
            %%% check for sex
            if ismember(species.sex,{'male','female'})
                subtype = sprintf('%s%s',subtype,species.sex(1));
            else
                message = 'Unknown sex for human'; GenericPBPKmodel_ReportErrorMessage(message);
            end;
            
            individual(id).species = mammals.human.(subtype);
            
            
            %%% assign manually set species data, e.g., in the virtual population,
            %%% back to the species structure
            individual(id).species = GenericPBPKmodel_assignPreviousValuesBack(individual(id).species,species);
            
            %%% Check if BW and BH correspond to reference values
            ref.BW = mammals.human.(subtype).BW;
            ref.BH = mammals.human.(subtype).BH;
            
            if (species.BW ~= ref.BW) || (species.BH ~= ref.BH)
                
                %%% check, whether there is a scaling method pre-defined
                %%% in the virtual population
                individual(id) = check_for_scaling_method(individual(id));
                
                %%% scale individual according to the scaling method
                individual(id).species.BWref = ref.BW;
                individual(id).species.BHref = ref.BH;
                individual(id) = GenericPBPKmodel_scaleIndividual(individual(id));
            end;
            
            
        case 'rat';
            
            %%% check for sex
            if ~ismember(species.sex,{'male'})
                message = 'Unsupported sex for rat'; GenericPBPKmodel_ReportErrorMessage(message);
            end;
            
            %%% check for BW
            if ismember(species.BW,[0.250, 0.475])
                subtype = sprintf('rat%d',1000*species.BW);
                
                individual(id).species = mammals.rat.(subtype);
                
                %%% assign manually set species data, e.g., in the virtual population,
                %%% back to the species structure
                individual(id).species = GenericPBPKmodel_assignPreviousValuesBack(individual(id).species,species);
                
            else
                
                fprintf('default reference rat subtype 250g\n');
                subtype='rat250';
                individual(id).species = mammals.rat.(subtype);
                
                
                %%% assign manually set species data, e.g., in the virtual population,
                %%% back to the species structure
                individual(id).species = GenericPBPKmodel_assignPreviousValuesBack(individual(id).species,species);
                
                %%% Check if BW and BH correspond to reference values
                ref.BW = mammals.rat.(subtype).BW;
                ref.BH = mammals.rat.(subtype).BH;
                if (species.BW ~= ref.BW)
                    
                    %%% check, whether there is a scaling method pre-defined
                    %%% in the virtual population
                    individual(id) = check_for_scaling_method(individual(id));
                    
                    %%% scale individual according to the scaling method
                    individual(id).species.BWref = ref.BW;
                    individual(id) = GenericPBPKmodel_scaleIndividual(individual(id));
                end
                
            end;
            
            
            
        case 'mouse';
            
            %%% check for sex
            if ~ismember(species.sex,{'male'})
                message = 'Unsupported sex for mouse'; GenericPBPKmodel_ReportErrorMessage(message);
            end;
            
            %%% check for BW
            if strcmp(species.source,'Brown et al (1997)') && ismember(species.BW,[0.025, 0.040])
                subtype = sprintf('mouse%d',1000*species.BW);
                
                individual(id).species = mammals.mouse.(subtype);
                
                %%% assign manually set species data, e.g., in the virtual population,
                %%% back to the species structure
                individual(id).species = GenericPBPKmodel_assignPreviousValuesBack(individual(id).species,species);
                
                
                
            elseif strcmp(species.source,'Baxter et al (1994)') && (species.BW==0.022)
                subtype = sprintf('mouseBax');
                
                individual(id).species = mammals.mouse.(subtype);
                
                %%% assign manually set species data, e.g., in the virtual population,
                %%% back to the species structure
                individual(id).species = GenericPBPKmodel_assignPreviousValuesBack(individual(id).species,species);
                
                
                
                
            elseif strcmp(species.source,'Fronton et al (2014)') && (species.BW==0.025)
                subtype = sprintf('mouseFaH');
                
                individual(id).species = mammals.mouse.(subtype);
                
                %%% assign manually set species data, e.g., in the virtual population,
                %%% back to the species structure
                individual(id).species = GenericPBPKmodel_assignPreviousValuesBack(individual(id).species,species);
                
                
                
                
            else
                fprintf('default reference mouse subtype 25g\n');
                subtype='mouse25'; %%%default mouse 25g
                individual(id).species = mammals.mouse.(subtype);
                
                
                %%% assign manually set species data, e.g., in the virtual population,
                %%% back to the species structure
                individual(id).species = GenericPBPKmodel_assignPreviousValuesBack(individual(id).species,species);
                
                %%% Check if BW and BH correspond to reference values
                ref.BW = mammals.mouse.(subtype).BW;
                ref.BH = mammals.mouse.(subtype).BH;
                if (species.BW ~= ref.BW)
                    
                    %%% check, whether there is a scaling method pre-defined
                    %%% in the virtual population
                    individual(id) = check_for_scaling_method(individual(id));
                    
                    %%% scale individual according to the scaling method
                    individual(id).species.BWref = ref.BW;
                    individual(id) = GenericPBPKmodel_scaleIndividual(individual(id));
                end
                
            end;
            
            
        otherwise
            message = 'Unknown species';
    end;
    
    
    %%% crete info string
    %%%
    species = individual(id).species;
    if strcmp(species.type,'human')
        individual(id).details = sprintf('%s, %s, %dy, %1.1fkg, %1.2fm [%s]',species.type,species.sex,species.age,species.BW,species.BH,individual(id).color);
    else
        individual(id).details = sprintf('%s, %dg [%s]',species.type,1e3*species.BW,individual(id).color);
    end;
    
    
end;

if ~strcmp(message,'no error')
    GenericPBPKmodel_ReportErrorMessage(message);
end;



end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function Individual = check_for_scaling_method(Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% assign default scaling method, if not speciefied otherwise

if isfield(Individual.species,'scalingMethod')
    
    %%% scaling method pre-specified. Check, whether it exists.
    if ~ismember(Individual.species,{'LBW','BW'}) %true if individual.species kein LBW oder BW enthalet dann error message
        message('Unknown scaling method!');
        GenericPBPKmodel_ReportErrorMessage(message);
    end;
    %%% scaling method not pre-specified. Assign default method depending
    %%% on species type.
else
    switch Individual.species.type
        
        case 'human';
            Individual.species.scalingMethod = 'LBW';
        otherwise
            Individual.species.scalingMethod = 'BW';
    end;
end;


end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function newStruc = GenericPBPKmodel_assignPreviousValuesBack(newStruc,oldStruc)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% assign values defined, e.g., in the virtual population, back to the
%%% species structure

speciesfields = fieldnames(oldStruc);

for k = 1:length(speciesfields)
    
    field = speciesfields{k};
    newStruc.(field) = oldStruc.(field);
    
end;



end


%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++





