%%% Version: October 31st, 2014
%%%
%%% UPDATE:
%%% (i) typo corrected: 'neutal' to 'neutral' in 'fraction_neutral(drug,pH)'
%%% ii) 22 Januar 2015, corrected ABCs

%%% This function determines tissue partition coefficients for different
%%% drug classes (sMD, mAb).
%%%
%%% For references and citation, please see MAIN script.
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function K = GenericPBPKmodel_TissuePartitionCoefficients(Individual,drug,method)


switch method.approach
    
    case 'RodgersRowland';
        
        K = GenericPBPKmodel_K_tis_up_RodgersRowland(Individual,drug);
        
    case 'ShahBetts';
        
        K = GenericPBPKmodel_ABC_ShahBetts(Individual,drug,method);
        
    otherwise
        GenericPBPKmodel_ReportErrorMessage('Unknown method to predict partition coefficients!');
end;


end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function K_tis_up = GenericPBPKmodel_K_tis_up_RodgersRowland(Individual,drug)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% Uses method published by Rodgers and Rowland to determine
%%% tissue-to-unbound plasma partition coefficients.

if ~strcmp(drug.class,'sMD')
    GenericPBPKmodel_ReportErrorMessage('RodgersRowland method only valid for small molecule drugs!');
end;

species  = Individual.species; T = species.T;

hct      = species.hct;   % hematocrit values
fVtis    = species.fVtis; % volume fractions of interstitial and cellular space
r        = species.r;     % tissue-to-plasma ratios of binding proteins

fuP      = drug.fuP;          % fraction unbound in plasma
K_ery_up = drug.K_ery_up;
logPow   = drug.logPow;
logPvow  = drug.logPvow;

BP = hct*fuP*K_ery_up + (1-hct);

if (isnan(logPow))
    error('Octanol-to-water partition coefficient undefined!');
end
% estimate logPvow based on logPow according to Poulin & Theil
if (isnan(logPvow))
    logPvow = 1.115*logPow-1.35;
end

KA_TPR    = T.initialize.tissueDB.NaN;

% ionization effects
pH_p = 7.4; pH_iw = 7.0; pH_e  = 7.22;

fnC  = fraction_neutral(drug,pH_iw);
fnE  = fraction_neutral(drug,pH_e);
fnP  = fraction_neutral(drug,pH_p);

% neutral lipids-to-water partition coefficient
Pow         = 10^logPow;
Knlw        = T.initialize.tissueDB.NaN;
Knlw(:)     = Pow;
Knlw(T.adi) = 10^logPvow;

% approximate neutral phospholipids-to-water partition coefficient
Knpw = 0.3*Knlw+0.7;

switch drug.subclass
    case {'mod. to strong base','zwitter ion'} % binding to acidic phospholipids (APmt)
        KA_TPR(:) = 0;
        KA_AP  = (K_ery_up - fnP/fnE*fVtis.wic(T.ery) - fnP*Pow*fVtis.nlt(T.ery)...
            - fnP*(0.3*Pow+0.7)*fVtis.npt(T.ery)) * fnE/((1-fnE)*fnP)/fVtis.APmt(T.ery);
    case {'neutral'}      % binding to lipoproteins (Ltp)
        KA_TPR = (1/fuP -1 -fnP*Pow*fVtis.nlt(T.pla) ...
            -fnP*(0.3*Pow+0.7)*fVtis.npt(T.pla))*r.Ltp;
        KA_AP  = 0;
    case {'acid','weak base'} % binding to albumin (Atp)
        KA_TPR = (1/fuP -1 -fnP*Pow*fVtis.nlt(T.pla) ...
            -fnP*(0.3*Pow+0.7)*fVtis.npt(T.pla))*r.Atp;
        KA_AP  = 0;
    otherwise
        GenericPBPKmodel_ReportErrorMessage('Unknown drug subclass type!');
end

K_tis_up = fVtis.wex +fnP/fnC*fVtis.wic + fnP*Knlw.*fVtis.nlt ...
    +fnP*Knpw.*fVtis.npt + fnP*(1-fnC)/fnC*KA_AP*fVtis.APmt + KA_TPR;

b = [T.art T.ven T.blo];
K_tis_up(b)     = BP/fuP;   % blood-to-unbound plasma partition coefficient
K_tis_up(T.pla) = 1/fuP;    % plasma-to-unbound plasma partition coefficient
K_tis_up(T.ery) = K_ery_up; % erythrocyte-to-unbound plasma partition coefficient

%%% -----------------------------------------------------------------------
    function fn = fraction_neutral(drug,pH)
        
        % pka  = pka values in decreasing order
        % pH   = pH value of the environment
        
        pKa = drug.pKa;
        
        switch drug.subclass
            case 'acid'
                fn = 1/(1+10^(pH-pKa(1)));
            case {'mod. to strong base','weak base'}
                fn = 1/(1+10^-(pH-pKa(1)));
            case 'neutral'
                fn = 1;
            case 'diprotonic-acid'
                fn = 1/(1+10^(pH-pKa(1))+10^(2*pH-pKa(1)-pKa(2)));
            case 'diprotonic-base'
                fn = 1/(1+10^-(pH-pKa(2))+ 10^-(2*pH-pKa(1)-pKa(2)));
            case 'zwitter ion'
                fn = 1/(1+10^(pH-pKa(2))+ 10^-(pH-pKa(1)));
            otherwise
                GenericPBPKmodel_ReportErrorMessage('Unknown drug subclass type!');
        end
        
    end % of fn = fraction_neutral(drug,pH)

end % of K_tis_up = GenericPBPKmodel_K_tis_up_RodgersRowland(Individual,drug)



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function ABC_tis = GenericPBPKmodel_ABC_ShahBetts(Individual,drug,method)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% Uses method published by Rodgers and Rowland to determine
%%% tissue-to-unbound plasma partition coefficients.

if ~ismember(drug.class,{'mAb','pAb','ADC'})
    GenericPBPKmodel_ReportErrorMessage('ShahBetts method only applicable to monoclonal antibodies!');
end;

species = Individual.species; T = species.T;

%%% antibody biodistribution coefficients (ABCs) from
%%% Shah & Betts, mAbs 5:2, 297?305; March/April 2013, Table 1
%%%
ABC_exp = T.initialize.tissueDB.NaN;

%%% values are given in unit [%], so finally divide by 100 to obtain
%%% fractions
ABC_exp(T.lun) = 14.9;
ABC_exp(T.hea) = 10.2;
ABC_exp(T.kid) = 13.7;
ABC_exp(T.mus) = 3.97;
ABC_exp(T.ski) = 15.7;
ABC_exp(T.gut) = 1/2*(5.22+5.03); % average of small and large intestine
ABC_exp(T.spl) = 12.8;
ABC_exp(T.liv) = 12.1;
ABC_exp(T.bon) = 7.27;
ABC_exp(T.adi) = 4.78;
ABC_exp(T.pla) = 100;

%%% convert to fractions
ABC_exp = ABC_exp/100;

%%% depending on correction for residual blood, either ABC_exp or
%%% corrected are assigned
correct_ABC_for_residual_blood = 0;

if correct_ABC_for_residual_blood
    
    
    species  = Individual.species;
    V.vas    = species.V.vas; V.tis = species.V.tis; hct = species.hct;
    
    %%% ASSUMPTION: residual blood equal to fac * vascular blood ...
    %fac = 0.01;
    fac=0.5;
    
    f_resblo = fac* V.vas ./ (V.vas+V.tis);
    
    ABC_tis  = ( ABC_exp - (1-hct)*f_resblo ) ./ (1-f_resblo);
    
    ABC_tis(T.pla) = 1;
    
else
    
    ABC_tis = ABC_exp;
    
end;

end

%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

