%%% Version: Mar 10th, 2017
%%%
%%%
%%% This script specifies all inputs for the simulation of the
%%% pharmacokinetics of different mAbs in a virtual population
%%% to be specified.
%%%

%%% For reference, please cite:
%%%
%%% --> For small molecule drugs:
%%%
%%%     W. Huisinga, A. Solms, L. Fronton, S. Pilari,
%%%     Modeling Interindividual Variability in Physiologically Based
%%%     Pharmacokinetics and Its Link to Mechanistic Covariate Modeling,
%%%     CPT: Pharmacometrics & Systems Pharmacology (2012) 1, e5;
%%%     doi:10.1038/psp.2012.3
%%%
%%% --> For monoclonal antibodies:
%%%
%%%     L. Fronton, S. Pilari, W. Huisinga
%%%     Monoclonal antibody disposition: A simplified PBPK model and its
%%%     implications for the derivation & interpretation of classical
%%%     compartment models,
%%%     J. Pharmacokinet Pharmacodyn (2014)
%%%     doi:10.1007/s10928-014-9349-1
%%%
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%

clear all; close all;

fprintf('\n\n Start simulation ...');


%%% =================================================================
%%% Specify dosing, including the following information
%%%
%%% 'drug'      sMD or mAb (see below)
%%% 'route'     of adminstration: iv bolus, iv infusion or per oral
%%% 'dose'      of the drug, specified in units specified by ...
%%% 'doseUnit'  unit of the dose in 'mg' or 'mg/kg BW', 'mg/kg LBW', 'mg/m2 BSA'
%%% 'time'      of drug administration in units specified by ...
%%% 'timeUnit'  unit of time in 'min', 'h', 'd' (day), 'w' (week)
%%% 'repeats'   single dose (=1) or multiple dosing (>1)
%%% 'Tinf'      duration of infusion (if applicable)
%%% 'interval'  dosing interval (multiple dosing case, if applicable)
%%%
%%% SMALL MOLECULE DRUGS:
%%% further subdivided into moderate-to-strong bases,
%%% weak bases, neutrals, acids and zwitterions. This property determines
%%% the method to a-priori predict the tissue partition coefficients.
%%%
%%% Moderate-to-strong bases: Amitriptyline, Caffeine,
%%% Diltiazem, Diphenhydramine, Imipramine, Lidocaine, Metoprolol,
%%% Pindolol, Sildefanil, Theophylline
%%%
%%% Weak bases: Alprazolam, Diazepam, Flunitrazepam, Midazolam, Triazolam
%%%
%%% Acids: Amobarbital, Diclofenac, Hexobarbital, Ibuprofen, Methohexital,
%%% Thiopental, Tolbutamide, Valproate, Warfarin
%%%
%%% test drugs: drugA (similar to Warfarin), drugB (similar to Lidocaine)
%%%
%%% MONOCLONAL ANTIBODIES:
%%% Zalutumumab, mAb7E3
%%%

single_dose.drug      = [];
single_dose.route     = [];                         % 'iv_bolus', 'iv_infusion', 'po'
single_dose.dose      = [];                         % in [mg] or [mg]/per BSD (body size descriptor)'
single_dose.doseUnit  ='mg';                        % 'mg' or 'mg/kg BW', 'mg/kg LBW', 'mg/m2 BSA'
single_dose.time      = 0;                          % time of first administration
single_dose.timeUnit  = 'min';
single_dose.repeats   = 1;                          % number of administration events
single_dose.Tinf      = NaN;
single_dose.interval  = NaN;

dosing       = single_dose;
dosing.drug  = 'mAbMOPC21';
dosing.dose  = 0.0038;
dosing.route = 'iv_bolus';



%%% =================================================================
%%% Load characteristics of reference individuals
%%%
%%% The following reference individuals are available
%%%
%%%     reference.human.male_35y_70kg_176cm
%%%     reference.human.female_35y_60kg_163cm
%%%     reference.human.male_15y_54kg_167cm
%%%     reference.human.female_15y_53kg_161cm
%%%     reference.human.unisex_10y_32kg_138cm
%%%     reference.human.unisex_5y_19kg_109cm
%%%
%%%     reference.rat.male_250g
%%%     reference.rat.male_475g
%%%
%%%     reference.mouse.male_25g
%%%     reference.mouse.male_40g
%%%     reference.mouse.male_22g_Bax
%%%     reference.mouse.male_22g_Fro

reference = GenericPBPKmodel_referenceIndividual();


%%% =================================================================
%%% Generate virtual population based on antropometric data.
%%% Approaches for generating the population:




population.how2generate = 'mAb_PBPK_ModelComparison_InfluenceTumor_SCID';

individual = GenericPBPKmodel_virtualPopulation(population,dosing,reference);




%%% =================================================================
%%% Assign anatomical & physiolocial species data and assign
%%% drug-specific data relevant for the chosen dosing
%%%

individual = GenericPBPKmodel_assignSpeciesData(individual);
T = GenericPBPKmodel_tissueIndexing;

individual = GenericPBPKmodel_assignDrugData(individual);


%%% =================================================================
%%% Random effects
%%% State factor of perturbation, i.e., if facX = 2, then the perturbed
%%% value of a quantity X will ly between 1/(1+facX) * X and (1+facX) * X
%%%


k=0;
randomEffects.mAb.facX.V_blood  =12/10*k;
randomEffects.mAb.facX.hct      =11/10*k;
randomEffects.mAb.facX.Dose     =0*k;
individual = GenericPBPKmodel_randomEffects(individual,randomEffects);


%%% =================================================================
%%% Simulate PBPK model for all individuals
%%%
%%% Note: If timeSpan consists only of initial and final time, i.e.,
%%%     timeSpan = [T0 Tend],
%%% then Tend will be automatically adapted, if final dosing event is
%%% at a time later than Tend

pred.timeSpan = [0 10];          % simulation time span
pred.timeUnit = 'd';             % unit of timeSpan

individual    = GenericPBPKmodel_simulatePBPKmodels(individual,pred);


%%% =================================================================
%%% Data analysis and descriptive statistics of individuals and
%%% the population
%%%

stats = GenericPBPKmodel_dataAnalysis(individual);


%%% =================================================================
%%% Graphical output. Specify below, which plots to be shown
%%%

ShowGraphics.plasma.allIndividuals = 'yes';     % 'yes' or 'no'
ShowGraphics.plasma.YScale         = 'log';     % 'log' or 'lin'
ShowGraphics.plasma.figNr          = 10;

allIds = 1:length(individual);
ShowGraphics.allTissues.individual = [allIds];   % specify id of individuals, e.g. [1 2 7]; [] = 'No plot'
% refer to specifications for virtual poulation for referencing of default individuals
ShowGraphics.allTissues.YScale     = 'log';     % 'log' or 'lin'
ShowGraphics.allTissues.figNr      = 11;
ShowGraphics.allTissues.oneFigOnly = 'yes';     % use same figure for all individuals 'yes'/'no'

ShowGraphics.percentiles.plasma    = 'no';     % applies only to a virtual population of size 10 or larger
ShowGraphics.percentiles.YScale    = 'log';
ShowGraphics.percentiles.figNr     = 25;

ShowGraphics.lowerLimitY           = 1e-1;      % sets the lower value of the Y axis

ShowGraphics.TestForLossOfMass.id    = [1];      % for testing purposes.
ShowGraphics.TestForLossOfMass.figNr = 40;

GenericPBPKmodel_graphicalOutput(individual,ShowGraphics);


fprintf('Simulation done\n')
fprintf('Compute Plots\n')

%%%compute percentiles
showplasma =1;
showtissues=1;

if showplasma
    
    figure(25); clf
    
    %%% Plot percentiles for venous blood concentrations for entire population
    %%% Only applicable for population of size>=10
    
    T = individual(1).stdout.T;  %figNr  = ShowGraphics.percentiles.figNr ;
    compound = individual(1).drug.listOfDrugs{1};
    
    switch individual(1).drug.(compound).class
        
        
        case 'mAb'
            %%% scale time from [min] to [day]
            ShowGraphics.SF.time = 1/(60*24);
            ShowGraphics.unit.time = 'day';
            
            %%% leave concentration in [mg]/L
            ShowGraphics.SF.conc = 1;
            ShowGraphics.unit.conc = 'mg/L';
            
            %%% uncomment below in case you want [nmol/L]
            ShowGraphics.SF.conc = individual(1).drug.(compound).SF.mg_to_nmol;
            ShowGraphics.unit.conc = 'nmol/L';
            
        case 'pAb' %e.g. IVIG
            %%% scale time from [min] to [day]
            ShowGraphics.SF.time = 1/(60*24);
            ShowGraphics.unit.time = 'day';
            
            %%% leave concentration in [mg]/L
            ShowGraphics.SF.conc = 1;
            ShowGraphics.unit.conc = 'mg/L';
            
            %%% uncomment below in case you want [nmol/L]
            ShowGraphics.SF.conc = individual(1).drug.(compound).SF.mg_to_nmol;
            ShowGraphics.unit.conc = 'nmol/L';
            
        otherwise
            
            %%% leave time in [min]
            ShowGraphics.SF.time = 1;
            ShowGraphics.unit.time = 'min';
            
            %%% leave concentration in [mg/L]
            ShowGraphics.SF.conc = 1;
            ShowGraphics.unit.conc = 'mg/L';
            
    end;
    
    ShowGraphics.upperLimitY = 10*ShowGraphics.lowerLimitY;
    
    
    %%% all plasma concentration-time profiles
    %%%
    tspan = individual(1).stdout.t* ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    hold on;
    for id=1:length(individual)
        if ~strcmp(individual(id).model(1).type,'ExperimentalData')
            
            t     = individual(id).stdout.t* ShowGraphics.SF.time;
            C.pla = individual(id).stdout.C.pla.*ShowGraphics.SF.conc;
            color =individual(id).color;
            
            plot(tspan(ind),C.pla(ind),color,'LineWidth',1.3);
        else
            
            t     = individual(id).stdout.t* ShowGraphics.SF.time;
            C.pla = individual(id).stdout.C.pla.*ShowGraphics.SF.conc;
            color1= char(individual(id).color);
            
            ind = find(isfinite(C.pla)); %plot experimental data
            plot(t(ind),C.pla(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
            
            %%%pred.errorbar in [nM]
            errorbar_up   =  individual(id).pred.errorbar.up.C.pla;
            errorbar_down =  individual(id).pred.errorbar.down.C.pla;
            errorbar(t(ind),C.pla(ind),errorbar_down(ind),errorbar_up(ind),'ko','LineWidth',3);
            
        end;
    end;
    
    set(gca,'Layer','top');
    title('plasma');
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    set(gca,'box','on')
    
    xlim([tspan(1), tspan(end)]);
    ymin = ShowGraphics.lowerLimitY;
    ymax = max((individual(2).stdout.C.pla.*ShowGraphics.SF.conc));
    if ymax<=ymin
        fprintf('\n'); fprintf(2,'>>> ShowGraphics.lowerLimitY lowered, since it was smaller than Cmax! <<< \n');
        ymin = 1e-3*ymax;
    end;
    grid off; ylim([ymin ymax]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    set(gca,'xticklabel',{[]})
    
    hold on;
    
    ax = gca;
    set(ax,'YTick',[ 1e-1 1e0 1e1 1e2 1000])

    
    set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('plasma','FontSize', 28);
    
    % % % Saving eps with matlab
    filename='Fig15a';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
    
end





%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% Tissues
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

if showtissues
    
    ymin=10^-1;
    ymax=10^3;
    
    %%% Plot percentiles for venous blood concentrations for entire population
    %%% Only applicable for population of size>=10
    
    T = individual(1).stdout.T;  %figNr  = ShowGraphics.percentiles.figNr ;
    
    
    %%%  tissue concentration-time profiles
    %%%
    
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    
    C.hea=individual(id).stdout.C.tis(:,T.hea).*ShowGraphics.SF.conc;
    C.mus=individual(id).stdout.C.tis(:,T.mus).*ShowGraphics.SF.conc;
    C.kid=individual(id).stdout.C.tis(:,T.kid).*ShowGraphics.SF.conc;
    C.lun=individual(id).stdout.C.tis(:,T.lun).*ShowGraphics.SF.conc;
    C.adi=individual(id).stdout.C.tis(:,T.adi).*ShowGraphics.SF.conc;
    C.bon=individual(id).stdout.C.tis(:,T.bon).*ShowGraphics.SF.conc;
    C.ski=individual(id).stdout.C.tis(:,T.ski).*ShowGraphics.SF.conc;
    C.spl=individual(id).stdout.C.tis(:,T.spl).*ShowGraphics.SF.conc;
    C.liv=individual(id).stdout.C.tis(:,T.liv).*ShowGraphics.SF.conc;
    C.gut=individual(id).stdout.C.tis(:,T.gut).*ShowGraphics.SF.conc;
    C.tum=individual(id).stdout.C.tis(:,T.tum).*ShowGraphics.SF.conc;
    
    hold on
    
    
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    %%% Heart
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    figure(26); clf
    hold on
    for id=1:length(individual)
        if ~strcmp(individual(id).model(1).type,'ExperimentalData')
            
            t    = individual(id).stdout.t* ShowGraphics.SF.time;
            C.hea=individual(id).stdout.C.tis(:,T.hea).*ShowGraphics.SF.conc;
            color=individual(id).color;
            
            plot(tspan(ind),C.hea(ind),color,'LineWidth',1.3);
            
        else
            t      = individual(id).stdout.t* ShowGraphics.SF.time;
            C.hea  =individual(id).stdout.C.tis(:,T.hea).*ShowGraphics.SF.conc;
            color1 = char(individual(id).color);
            ind = find(isfinite(C.hea)); %plot experimental data
            plot(t(ind),C.hea(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
            
            %%pred.errorbar in [nM]
            pred=individual(id).pred;
            errorbar_up   =  pred.errorbar.up.C.tis(:,T.hea);
            errorbar_down =  pred.errorbar.down.C.tis(:,T.hea);
            errorbar(t(ind),C.hea(ind),errorbar_down(ind),errorbar_up(ind),'ko');
            
        end;
    end
    set(gca,'Layer','top');
    title('heart');
    
    xlim([tspan(1), tspan(end)]);
    grid off; ylim([ymin ymax]);
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    box on
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    set(gca,'xticklabel',{[]})
    
      set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('heart','FontSize', 28);
    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig15c';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
    
    hold on
    
    
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    %%% Muscle
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    
    
    figure(27)
    hold on
    for id=1:length(individual)
        if ~strcmp(individual(id).model(1).type,'ExperimentalData')
            
            t    = individual(id).stdout.t* ShowGraphics.SF.time;
            C.mus=individual(id).stdout.C.tis(:,T.mus).*ShowGraphics.SF.conc;
            color=individual(id).color;
            
            
            plot(tspan(ind),C.mus(ind),color,'LineWidth',1.3);
            
        else
            
            t       = individual(id).stdout.t* ShowGraphics.SF.time;
            C.mus   =individual(id).stdout.C.tis(:,T.mus).*ShowGraphics.SF.conc;
            color1  = char(individual(id).color);
            ind     = find(isfinite(C.mus)); %plot experimental data
            plot(t(ind),C.mus(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
            
            %%Caution: pred.errorbar in [nM]
            pred=individual(id).pred;
            errorbar_up   =  pred.errorbar.up.C.tis(:,T.mus);
            errorbar_down =  pred.errorbar.down.C.tis(:,T.mus);
            errorbar(t(ind),C.mus(ind),errorbar_down(ind),errorbar_up(ind),'ko');
            
            
        end;
    end
    set(gca,'Layer','top');
    title('muscle');
    
    xlim([tspan(1), tspan(end)]);
    grid off; ylim([ymin ymax]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    set(gca,'xticklabel',{[]})
    box on
    
      set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('muscle','FontSize', 28);
    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig15e';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
    
    
    hold on
    
    
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    %%% Kidney
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    %%%
    figure(28)
    hold on
    for id=1:length(individual)
        if ~strcmp(individual(id).model(1).type,'ExperimentalData')
            
            t    = individual(id).stdout.t* ShowGraphics.SF.time;
            C.kid=individual(id).stdout.C.tis(:,T.kid).*ShowGraphics.SF.conc;
            color=individual(id).color;
            
            plot(tspan(ind),C.kid(ind),color,'LineWidth',1.3);
            
        else
            
            t      = individual(id).stdout.t* ShowGraphics.SF.time;
            C.kid  =individual(id).stdout.C.tis(:,T.kid).*ShowGraphics.SF.conc;
            color1 = char(individual(id).color);
            ind    = find(isfinite(C.kid)); %plot experimental data
            plot(t(ind),C.kid(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
            
            %%% pred.errorbar in [nM]
            pred=individual(id).pred;
            errorbar_up   =  pred.errorbar.up.C.tis(:,T.kid);
            errorbar_down =  pred.errorbar.down.C.tis(:,T.kid);
            errorbar(t(ind),C.kid(ind),errorbar_down(ind),errorbar_up(ind),'ko');
            
            
        end;
    end
    set(gca,'Layer','top');
    title('kidney');
    
    xlim([tspan(1), tspan(end)]);
    grid off; ylim([ymin ymax]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    box on
    
    set(gca,'xticklabel',{[]},'yticklabel',{[]});
    
      set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('kidney','FontSize', 28);
    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig15d';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
    
    hold on
    
    
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    %%% Lung
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    figure(29);
    hold on
    for id=1:length(individual)
        if ~strcmp(individual(id).model(1).type,'ExperimentalData')
            
            t    = individual(id).stdout.t* ShowGraphics.SF.time;
            C.lun=individual(id).stdout.C.tis(:,T.lun).*ShowGraphics.SF.conc;
            color=individual(id).color;
            
            
            plot(tspan(ind),C.lun(ind),color,'LineWidth',1.3);
            
        else
            
            t      = individual(id).stdout.t* ShowGraphics.SF.time;
            C.lun  =individual(id).stdout.C.tis(:,T.lun).*ShowGraphics.SF.conc;
            color1 = char(individual(id).color);
            ind    = find(isfinite(C.lun)); %plot experimental data
            plot(t(ind),C.lun(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
            
            %%Caution: pred.errorbar in [nM]
            pred=individual(id).pred;
            errorbar_up   =  pred.errorbar.up.C.tis(:,T.lun);
            errorbar_down =  pred.errorbar.down.C.tis(:,T.lun);
            errorbar(t(ind),C.lun(ind),errorbar_down(ind),errorbar_up(ind),'ko');
            
            
        end;
    end
    set(gca,'Layer','top');
    title('lung');
    
    xlim([tspan(1), tspan(end)]);
    ymin_lung = ShowGraphics.lowerLimitY;
    ymax_lung = max((individual(2).stdout.C.pla.*ShowGraphics.SF.conc));
    if ymax_lung<=ymin_lung
        fprintf('\n'); fprintf(2,'>>> ShowGraphics.lowerLimitY lowered, since it was smaller than Cmax! <<< \n');
        ymin_lung = 1e-3*ymax_lung;
    end;
    grid off; ylim([ymin_lung ymax_lung]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    set(gca,'xticklabel',{[]},'yticklabel',{[]});
    box on
    
      set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('lung','FontSize', 28);
    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig15b';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
    
    
    hold on
    
    
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    %%% Skin
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    
    figure(32)
    hold on
    for id=1:length(individual)
        if ~strcmp(individual(id).model(1).type,'ExperimentalData')
            
            t    = individual(id).stdout.t* ShowGraphics.SF.time;
            C.ski=individual(id).stdout.C.tis(:,T.ski).*ShowGraphics.SF.conc;
            color=individual(id).color;
            
            plot(tspan(ind),C.ski(ind),color,'LineWidth',1.3);
            
        else
            
            t      = individual(id).stdout.t* ShowGraphics.SF.time;
            C.ski  =individual(id).stdout.C.tis(:,T.ski).*ShowGraphics.SF.conc;
            color1 = char(individual(id).color);
            ind    = find(isfinite(C.ski)); %plot experimental data
            plot(t(ind),C.ski(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
            
            %%% pred.errorbar in [nM]
            pred=individual(id).pred;
            errorbar_up   =  pred.errorbar.up.C.tis(:,T.ski);
            errorbar_down =  pred.errorbar.down.C.tis(:,T.ski);
            errorbar(t(ind),C.ski(ind),errorbar_down(ind),errorbar_up(ind),'ko');
            
            
        end;
    end
    set(gca,'Layer','top');
    title('skin');
    
    xlim([tspan(1), tspan(end)]);
    grid off; ylim([ymin ymax]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    set(gca,'xticklabel',{[]},'yticklabel',{[]});
    box on
    
      set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('skin','FontSize', 28);
    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig15f';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
    
    hold on
    
    
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    %%% Gut
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    
    figure(33)
    hold on
    for id=1:length(individual)
        if ~strcmp(individual(id).model(1).type,'ExperimentalData')
            
            t    = individual(id).stdout.t* ShowGraphics.SF.time;
            C.gut= individual(id).stdout.C.tis(:,T.gut).*ShowGraphics.SF.conc;
            color= individual(id).color;
            
            plot(tspan(ind),C.gut(ind),color,'LineWidth',1.3);
            
        else
            
            t      = individual(id).stdout.t* ShowGraphics.SF.time;
            C.gut  = individual(id).stdout.C.tis(:,T.gut).*ShowGraphics.SF.conc;
            color1 = char(individual(id).color);
            ind    = find(isfinite(C.gut)); %plot experimental data
            plot(t(ind),C.gut(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
            
            %%pred.errorbar in [nM]
            pred=individual(id).pred;
            errorbar_up   =  pred.errorbar.up.C.tis(:,T.gut);
            errorbar_down =  pred.errorbar.down.C.tis(:,T.gut);
            errorbar(t(ind),C.gut(ind),errorbar_down(ind),errorbar_up(ind),'ko');
            
            
        end;
    end
    set(gca,'Layer','top');
    title('gut');
    
    xlim([tspan(1), tspan(end)]);
    grid off; ylim([ymin ymax]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    set(gca,'xticklabel',{[]},'yticklabel',{[]});
    box on
    
      set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('gut','FontSize', 28);
    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig15h';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
    
    hold on
    
    
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    %%% Spleen
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind   = 1:length(tspan);
    
    
    figure(34)
    hold on
    for id=1:length(individual)
        if ~strcmp(individual(id).model(1).type,'ExperimentalData')
            t    = individual(id).stdout.t* ShowGraphics.SF.time;
            C.spl=individual(id).stdout.C.tis(:,T.spl).*ShowGraphics.SF.conc;
            color=individual(id).color;
            
            plot(tspan(ind),C.spl(ind),color,'LineWidth',1.3);
            
        else
            
            t      = individual(id).stdout.t* ShowGraphics.SF.time;
            C.spl  =individual(id).stdout.C.tis(:,T.spl).*ShowGraphics.SF.conc;
            color1 = char(individual(id).color);
            ind    = find(isfinite(C.spl)); %plot experimental data
            plot(t(ind),C.spl(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
            
            %%% pred.errorbar in [nM]
            pred=individual(id).pred;
            errorbar_up   =  pred.errorbar.up.C.tis(:,T.spl);
            errorbar_down =  pred.errorbar.down.C.tis(:,T.spl);
            errorbar(t(ind),C.spl(ind),errorbar_down(ind),errorbar_up(ind),'ko');
            
            
        end;
    end
    set(gca,'Layer','top');
    title('spleen');
    
    xlim([tspan(1), tspan(end)]);
    grid off; ylim([ymin ymax]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    set(gca,'YTick',[ 1e-1 1e0 1e1 1e2 1000])
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    
    
    box on
    
      set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('spleen','FontSize', 28);
   
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig15i';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
    hold on
    
    
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    %%% Liver
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind = 1:length(tspan);

    
    figure(35)
    hold on
    for id=1:length(individual)
        if ~strcmp(individual(id).model(1).type,'ExperimentalData')
            
            t    = individual(id).stdout.t* ShowGraphics.SF.time;
            C.liv= individual(id).stdout.C.tis(:,T.liv).*ShowGraphics.SF.conc;
            color= individual(id).color;
            
            plot(tspan(ind),C.liv(ind),color,'LineWidth',1.3);
            
        else
            
            t      = individual(id).stdout.t* ShowGraphics.SF.time;
            C.liv  =individual(id).stdout.C.tis(:,T.liv).*ShowGraphics.SF.conc;
            color1 = char(individual(id).color);
            ind    = find(isfinite(C.liv)); %plot experimental data
            plot(t(ind),C.liv(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
            
            %%pred.errorbar in [nM]
            pred=individual(id).pred;
            errorbar_up   =  pred.errorbar.up.C.tis(:,T.liv);
            errorbar_down =  pred.errorbar.down.C.tis(:,T.liv);
            errorbar(t(ind),C.liv(ind),errorbar_down(ind),errorbar_up(ind),'ko');
            
            
        end;
    end
    set(gca,'Layer','top');
    title('liver');
    
    xlim([tspan(1), tspan(end)]);
    grid off; ylim([ymin ymax]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
     set(gca,'YTick',[ 1e-1 1e0 1e1 1e2 1000])
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    set(gca,'xticklabel',{[]});
    
    box on
    
      set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('liver','FontSize', 28);

    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig15g';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
    
    hold on
    
    
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    %%% Tumor
    %%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind = 1:length(tspan);
    
    
    
    figure(36)
    hold on
    for id=1:length(individual)
        if ~strcmp(individual(id).model(1).type,'ExperimentalData')
            
            t    = individual(id).stdout.t* ShowGraphics.SF.time;
            C.tum= individual(id).stdout.C.tis(:,T.tum).*ShowGraphics.SF.conc;
            color= individual(id).color;

            plot(tspan(ind),C.tum(ind),color,'LineWidth',1.3);
        else
            t      = individual(id).stdout.t* ShowGraphics.SF.time;
            C.tum  = individual(id).stdout.C.tis(:,T.tum).*ShowGraphics.SF.conc;
            color1 = char(individual(id).color);

            ind = find(isfinite(C.tum)); %plot experimental data
            plot(t(ind),C.tum(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color1,'MarkerSize',10);
            
            %%pred.errorbar in [nM]
            pred=individual(id).pred;
            errorbar_up   =  pred.errorbar.up.C.tis(:,T.tum);
            errorbar_down =  pred.errorbar.down.C.tis(:,T.tum);
            errorbar(t(ind),C.tum(ind),errorbar_down(ind),errorbar_up(ind),'ko');
            
            
        end;
    end
    set(gca,'Layer','top');
    title('tumour');
    
    xlim([tspan(1), tspan(end)]);
    grid off; ylim([ymin ymax]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    
    
    lw = 2;   %line width
    fs = 18;  %fontsize
    
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',fs);
    set(get(gca,'xlabel'),'Fontsize',fs);
    set(get(gca,'ylabel'),'Fontsize',fs);
    set(legend,'Fontsize',fs);
    set(gca,'yticklabel',{[]});
    box on
    
      set(gca,'FontSize',24); %%%Fontsize for axis ticks
    title('tumour','FontSize', 28);
   
    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig15j';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
   
    
    
    
end

