%%% Version: October 31st, 2016
%%%
%%% This script specifies all inputs for the simulation of the
%%% pharmacokinetics of different mAbs in a virtual population
%%% to be specified.
%%%
%%% For reference, please cite:
%%%
%%% --> For small molecule drugs:
%%%
%%%     W. Huisinga, A. Solms, L. Fronton, S. Pilari,
%%%     Modeling Interindividual Variability in Physiologically Based
%%%     Pharmacokinetics and Its Link to Mechanistic Covariate Modeling,
%%%     CPT: Pharmacometrics & Systems Pharmacology (2012) 1, e5;
%%%     doi:10.1038/psp.2012.3
%%%
%%% --> For monoclonal antibodies:
%%%
%%%     L. Fronton, S. Pilari, W. Huisinga
%%%     Monoclonal antibody disposition: A simplified PBPK model and its
%%%     implications for the derivation & interpretation of classical
%%%     compartment models,
%%%     J. Pharmacokinet Pharmacodyn (2014)
%%%     doi:10.1007/s10928-014-9349-1
%%%
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%

clear all; close all;


%%% =================================================================
%%% Specify dosing, including the following information
%%%
%%% 'drug'      sMD or mAb (see below)
%%% 'route'     of adminstration: iv bolus, iv infusion or per oral
%%% 'dose'      of the drug, specified in units specified by ...
%%% 'doseUnit'  unit of the dose in 'mg' or 'mg/kg BW', 'mg/kg LBW', 'mg/m2 BSA'
%%% 'time'      of drug administration in units specified by ...
%%% 'timeUnit'  unit of time in 'min', 'h', 'd' (day), 'w' (week)
%%% 'repeats'   single dose (=1) or multiple dosing (>1)
%%% 'Tinf'      duration of infusion (if applicable)
%%% 'interval'  dosing interval (multiple dosing case, if applicable)
%%%
%%% SMALL MOLECULE DRUGS:
%%% further subdivided into moderate-to-strong bases,
%%% weak bases, neutrals, acids and zwitterions. This property determines
%%% the method to a-priori predict the tissue partition coefficients.
%%%
%%% Moderate-to-strong bases: Amitriptyline, Caffeine,
%%% Diltiazem, Diphenhydramine, Imipramine, Lidocaine, Metoprolol,
%%% Pindolol, Sildefanil, Theophylline
%%%
%%% Weak bases: Alprazolam, Diazepam, Flunitrazepam, Midazolam, Triazolam
%%%
%%% Acids: Amobarbital, Diclofenac, Hexobarbital, Ibuprofen, Methohexital,
%%% Thiopental, Tolbutamide, Valproate, Warfarin
%%%
%%% test drugs: drugA (similar to Warfarin), drugB (similar to Lidocaine)
%%%
%%% MONOCLONAL ANTIBODIES:


single_dose.drug      = [];
single_dose.route     = [];                         % 'iv_bolus', 'iv_infusion', 'po'
single_dose.dose      = [];                         % in [mg] or [mg]/per BSD (body size descriptor)'
single_dose.doseUnit  ='mg/kg BW';                  % 'mg' or 'mg/kg BW', 'mg/kg LBW', 'mg/m2 BSA'
single_dose.time      = 0;                          % time of first administration
single_dose.timeUnit  = 'd';
single_dose.repeats   = 1;                          % number of administration events
single_dose.Tinf      = NaN;                        % duration of iv infusion
single_dose.interval  = NaN ;                        % multiple dosing interval

dosing       = single_dose;                          % dose in [mg/kg BW], administration at t=0, time in [h]
dosing.drug  = 'mAb7E3';
dosing.dose  =8;
dosing.route = 'iv_bolus';


IVIG_therapy = 1;
if IVIG_therapy
    dosing(2)       = dosing(1);
    dosing(2).drug  = 'IVIG';
    dosing(2).dose  =1000;
end;

%%% =================================================================
%%% Load characteristics of reference individuals
%%%
%%% The following reference individuals are available
%%%
%%%     reference.human.male_35y_70kg_176cm
%%%     reference.human.female_35y_60kg_163cm
%%%     reference.human.male_15y_54kg_167cm
%%%     reference.human.female_15y_53kg_161cm
%%%     reference.human.unisex_10y_32kg_138cm
%%%     reference.human.unisex_5y_19kg_109cm
%%%
%%%     reference.rat.male_250g
%%%     reference.rat.male_475g
%%%
%%%     reference.mouse.male_25g
%%%     reference.mouse.male_40g
%%%     reference.mouse.male_22g_Bax
%%%     reference.mouse.male_22g_Fro

reference = GenericPBPKmodel_referenceIndividual();


%%% =================================================================
%%% Generate virtual population based on antropometric data.
%%% Approaches for generating the population:



population.how2generate ='mAb_PBPK_Modelcomparison_WTmice_differentKD_inset';



individual = GenericPBPKmodel_virtualPopulation(population,dosing,reference);




%%% =================================================================
%%% Assign anatomical & physiolocial species data and assign
%%% drug-specific data relevant for the chosen dosing
%%%

individual = GenericPBPKmodel_assignSpeciesData(individual);
T = GenericPBPKmodel_tissueIndexing;

individual = GenericPBPKmodel_assignDrugData(individual);


%%% =================================================================
%%% Random effects
%%% State factor of perturbation, i.e., if facX = 2, then the perturbed
%%% value of a quantity X will ly between 1/(1+facX) * X and (1+facX) * X
%%%



k=0;
randomEffects.mAb.facX.V_blood  =12/10*k;
randomEffects.mAb.facX.hct      =11/10*k;
randomEffects.mAb.facX.Dose     =0*k;
individual = GenericPBPKmodel_randomEffects(individual,randomEffects);




%%% =================================================================
%%% Simulate PBPK model for all individuals
%%%
%%% Note: If timeSpan consists only of initial and final time, i.e.,
%%%     timeSpan = [T0 Tend],
%%% then Tend will be automatically adapted, if final dosing event is
%%% at a time later than Tend

pred.timeSpan = [0 60];          % simulation time span
pred.timeUnit = 'd';             % unit of timeSpan

individual    = GenericPBPKmodel_simulatePBPKmodels(individual,pred);


%%% =================================================================
%%% Data analysis and descriptive statistics of individuals and
%%% the population
%%%

stats = GenericPBPKmodel_dataAnalysis(individual);


%%% =================================================================
%%% Graphical output. Specify below, which plots to be shown
%%%

ShowGraphics.plasma.allIndividuals = 'yes';     % 'yes' or 'no'
ShowGraphics.plasma.YScale         = 'log';     % 'log' or 'lin'
ShowGraphics.plasma.figNr          = 10;

allIds = 1:length(individual);
ShowGraphics.allTissues.individual = [allIds];   % specify id of individuals, e.g. [1 2 7]; [] = 'No plot'
% refer to specifications for virtual poulation for referencing of default individuals
ShowGraphics.allTissues.YScale     = 'log';     % 'log' or 'lin'
ShowGraphics.allTissues.figNr      = 11;
ShowGraphics.allTissues.oneFigOnly = 'yes';     % use same figure for all individuals 'yes'/'no'

ShowGraphics.percentiles.plasma    = 'no';     % applies only to a virtual population of size 10 or larger
ShowGraphics.percentiles.YScale    = 'log';
ShowGraphics.percentiles.figNr     = 25;

ShowGraphics.lowerLimitY           = 1e-1;      % sets the lower value of the Y axis

ShowGraphics.TestForLossOfMass.id    = [];      % for testing purposes.





fprintf('Simulation done\n')

fprintf('Generate Figure\n')
GenerateFigure=1;

if GenerateFigure
    
    figure25=figure(25); clf
    
    axes2 = axes('Parent',figure25,...
        'XGrid','off','YGrid','off');
    
    %%% Plot percentiles for venous blood concentrations for entire population
    %%% Only applicable for population of size>=10
    
    T = individual(1).stdout.T;  %figNr  = ShowGraphics.percentiles.figNr ;
    
    compound = individual(1).drug.listOfDrugs{1};
    
    
    switch individual(1).drug.(compound).class
      
        case 'mAb'
            %%% scale time from [min] to [day]
            ShowGraphics.SF.time = 1/(60*24);
            ShowGraphics.unit.time = 'day';
            
            %%% leave concentration in [mg]/L
            ShowGraphics.SF.conc = 1;
            ShowGraphics.unit.conc = 'mg/L';
            
            %%% uncomment below in case you want [nmol/L]
            ShowGraphics.SF.conc = individual(1).drug.(compound).SF.mg_to_nmol;
            ShowGraphics.unit.conc = 'nmol/L';
            
        case 'pAb' %e.g. IVIG
            %%% scale time from [min] to [day]
            ShowGraphics.SF.time = 1/(60*24);
            ShowGraphics.unit.time = 'day';
            
            %%% leave concentration in [mg]/L
            ShowGraphics.SF.conc = 1;
            ShowGraphics.unit.conc = 'mg/L';
            
            %%% uncomment below in case you want [nmol/L]
            ShowGraphics.SF.conc = individual(1).drug.(compound).SF.mg_to_nmol;
            ShowGraphics.unit.conc = 'nmol/L';
            
        otherwise
            
            %%% leave time in [min]
            ShowGraphics.SF.time = 1;
            ShowGraphics.unit.time = 'min';
            
            %%% leave concentration in [mg/L]
            ShowGraphics.SF.conc = 1;
            ShowGraphics.unit.conc = 'mg/L';
            
    end;
    
    ShowGraphics.upperLimitY = 10*ShowGraphics.lowerLimitY;
    
    %%% combine all plasma concentration-time profiles in a single matrix
    %%%
    
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    
    
    hold on
    for id=1:length(individual)
        
        t = individual(id).stdout.t*ShowGraphics.SF.time;
        C_pla = (individual(id).stdout.C.pla*ShowGraphics.SF.conc);
        
        hold on
        color = char(individual(id).color);
        
        if ~strcmp(individual(id).model(1).type,'ExperimentalData')
            
            plot(t,C_pla,color,'LineWidth',1.3); 
        else
            ind = find(isfinite(C_pla));
            plot(t(ind),C_pla(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color,'MarkerSize',7);
            
            
        end;
    end
    
    
    set(gca,'Layer','top');
    title('WT mice, mAb 7E3+IVIG');
    xlabel('time [days]');
    ylabel('mAb plasma concentration [nM]');
    xlim([tspan(1), 60]);
    lw = 2;   %line width
    fs = 18;  %fontsize
    h = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
      set(get(gca,'title'),'Fontsize',22);
    set(get(gca,'xlabel'),'Fontsize',22);
    set(get(gca,'ylabel'),'Fontsize',22);
    set(legend,'Fontsize',fs);
    grid off;
    ylim([1e0 1e4]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    box on
    
    
    axes4 = axes('Parent',figure25,'Position',[0.65 0.65 0.2 0.2],...
        'LineWidth',2,'FontSize',14,'XGrid','off','YGrid','off');
    
    
    hold('all')
    
    %%%Zoomed graphics
    
    tspan = individual(1).stdout.t*ShowGraphics.SF.time;
    ind   = 1:length(tspan);
    hold on
    for id=1:length(individual)
        
        t = individual(id).stdout.t*ShowGraphics.SF.time;
        C_pla = (individual(id).stdout.C.pla*ShowGraphics.SF.conc);
        hold on
        color = char(individual(id).color);
        
        if ~strcmp(individual(id).model(1).type,'ExperimentalData')
            plot(t,C_pla,color,'LineWidth',1.3);
        else
            ind = find(isfinite(C_pla));
            plot(t(ind),C_pla(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color,'MarkerSize',7);
        end;
    end
    
    xlim([tspan(1), 1]);
    ylim([1e2 1.5e3]);
    set(gca,'YScale',ShowGraphics.percentiles.YScale);
    box on
    lw = 1;   %line width
    fs = 18;  %fontsize
    h  = findobj(gca,'Type','line');
    set(h,'LineWidth',lw);
    set(gca,'FontSize',fs);
    set(gca,'LineWidth',lw);
    set(get(gca,'title'),'Fontsize',22);
    set(get(gca,'xlabel'),'Fontsize',22);
    set(get(gca,'ylabel'),'Fontsize',22);
    set(legend,'Fontsize',fs);
       grid off;
    box on;
    
    hold('all')
    
    % % % Saving eps with matlab and then marking pdf and eps and png with system commands
    filename='Fig10';
    print(gcf, '-depsc2','-loose',[filename,'.eps']);
    
end






