%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%------------------------------------------------------------------------
%% DESCRIPTION:
%%%------------------------------------------------------------------------
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% This function simulates viral dynamics under zidovudine monotherapy. The
% inputs for the code include fitness costs and resistance factors for
% different mutant genotypes and drug efficacy. The code computes the time
% profiles of the mutant genotypes, mutations and selective advantages.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% For reference please cite

% Estimating HIV-1 Fitness Characteristics from Cross-sectional Genotype
% Data, Sathej Gopalakrishnan, Hesam Montazeri, Stephan Menz, Niko
% Beerenwinkel and Wilhelm Huisinga (Submitted)

% Details of the viral dynamics model and mutation models found there-in

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function ZDV_Simulation

clear all;
close all
clc

epsilon = 0.75; %Drug efficacy of ZDV on wild-type
DrugConcFold = epsilon/(1-epsilon); % C/IC50 for the wild-type assuming an Emax kind of model

% -------------------------------------------------------------------------
% Average Waiting times for each site of mutation computed from CT-CBN
% -------------------------------------------------------------------------

lambda = textread('ZDV_lambdas');
T_data = [1/lambda(1); 1/lambda(2); 1/lambda(2) + 1/lambda(3); 1/lambda(1) + 1/lambda(4) + 1/lambda(5); 1/lambda(1) + 1/lambda(5); 1/lambda(2) + 1/lambda(3) + 1/lambda(6)];
T_data = T_data./min(T_data); % Normalize by time to fastest mutation
T_stat = T_data;

% -------------------------------------------------------------------------

% Key to numbering of mutations

% 1 : M41L 
% 2 : D67N 
% 3 : K70R 
% 4 : L210W 
% 5 : T215Y 
% 6 : T219Q

% -------------------------------------------------------------------------
%                              Mutant Genotypes
% -------------------------------------------------------------------------

% Mut1  : {2} 
% Mut2  : {2,3} 
% Mut3  : {2,3,6} 
% Mut4  : {1} 
% Mut5  : {1,5}
% Mut6  : {1,4,5}
% Mut7  : {1,2,4,5} 
% Mut8  : {1,2,3,4,5,6}

% Read binary representation of mutational genotype lattice

Lattice = textread('ZDV_lattice');
MutGen = Lattice([1:7 10 16],1:end-1);

% Resistance factors estimated from ICBN model

%     1.0471 : WT 
%     1.9953 : Mut1 
%     6.4565 : Mut2
%    11.2202 : Mut3
%     2.2387 : Mut4 
%     4.1687 : Mut5
%    11.4815 : Mut6 
%    23.4423 : Mut7
%   263.0268 : Mut8

% Resistances re-defined to enter the model appropriately
par.Res = [1/1.0471, 1/1.9953, 1/6.4565, 1/11.2202, 1/2.2387, 1/4.1687, 1/11.4815, 1/23.4423, 1/263.0268]';
par.Res = par.Res./par.Res(1); % Normalize so that Res(WT) = 1
Res = par.Res;

% Vector of fitness costs

s = [0
    0.165954384293497
    0.285405096083664
    0.226620692402823
    0.252913478543702
    0.078281673272994
    0.202769378770693
    0.065317521794638
    0.058682713610012];

% -------------------------------------------------------------------------
% -------------Infection Model Parameters (in vivo) -----------------------
% -------------------------------------------------------------------------

% T-cells

par.lambda_T    = 2e9;
par.delta_T     = 0.02;
par.delta_PIC_T = 0.35;
par.beta_T_WT   = 8e-12;
par.rho_rev     = 0.33;
par.CL_T_WT     = par.beta_T_WT*(1/par.rho_rev - 1);
par.delta_T1    = 0.02;
par.k_T         = 0.35;
par.delta_T2    = 1;
par.CL_VI       = 23;
par.CL_VNI      = 23;
par.b           = 1;
par.q           = 1;
par.rho_PR      = 0.67;
par.Nhat_T      = 1000;
par.N_T         = par.b*par.q*par.rho_PR*par.Nhat_T;
par.N_T_NI      = par.Nhat_T - par.N_T;

% Macrophages

par.lambda_M    = 6.9e7;
par.beta_M_WT   = 1e-14;
par.Nhat_M      = 100;
par.delta_M     = 0.0069;
par.delta_M1    = 0.0069;
par.delta_M2    = 0.09;
par.k_M         = 0.07;
par.delta_PIC_M = 0.0035;
par.N_M         = par.b*par.q*par.rho_PR*par.Nhat_M;
par.N_M_NI      = par.Nhat_M - par.N_M;
par.CL_M_WT     = par.beta_M_WT*(1/par.rho_rev - 1);

% Latent cells

par.p           = 8e-6;
par.alpha       = 1e-3;
par.delta_TL    = 1e-4;

% -------------------------------------------------------------------------
%      Mutation Probabilities, similar as in Stilianakis et al, 1998
% -------------------------------------------------------------------------

par.mu0 = 3.4e-5;
par.mu1 = par.mu0/3;  % Probability of G -> A mutations
par.mu2 = par.mu1/2;  % Probability of other non-transversion mutations
par.mu3 = par.mu1/10; % Probability of transversion mutations

% -------------------------------------------------------------------------
% Defining a mutation transition rate matrix with nucleotide specific
% mutation rates
% -------------------------------------------------------------------------

par.A = sparse(length(par.Res),length(par.Res));

for i = 1:length(MutGen)
    for j = (i+1):length(MutGen)
        if sum(xor(MutGen(i,:),MutGen(j,:))) == 1
            if find(xor(MutGen(i,:),MutGen(j,:))) == 5
                par.A(i,j)     = (par.mu3/2)*(1-par.mu0)^5;
                par.A(j,i)     = (par.mu3/2)*(1-par.mu0)^5;
                
            elseif find(xor(MutGen(i,:),MutGen(j,:))) == 1
                par.A(i,j) = par.mu3*(1-par.mu0)^5;
                par.A(j,i) = par.mu3*(1-par.mu0)^5;
                
            elseif ((find(xor(MutGen(i,:),MutGen(j,:))) == 2)&&(MutGen(i,2) == 1))
                par.A(i,j) = par.mu2*(1-par.mu0)^5;
            elseif ((find(xor(MutGen(i,:),MutGen(j,:))) == 2)&&(MutGen(i,2) == 0))
                par.A(j,i) = par.mu1*(1-par.mu0)^5;
                
            elseif ((find(xor(MutGen(i,:),MutGen(j,:))) == 3) &&(MutGen(i,3) == 1))
                par.A(i,j) =  par.mu1*(1-par.mu0)^5;
            elseif ((find(xor(MutGen(i,:),MutGen(j,:))) == 3) &&(MutGen(i,3) == 0))
                par.A(j,i) = par.mu2*(1-par.mu0)^5;
                
            elseif find(xor(MutGen(i,:),MutGen(j,:))) == 4
                par.A(i,j) = par.mu3*(1-par.mu0)^5;
                par.A(j,i) = par.mu3*(1-par.mu0)^5;
                
            elseif find(xor(MutGen(i,:),MutGen(j,:))) == 6
                par.A(i,j) = par.mu3*(1-par.mu0)^5;
                par.A(j,i) = par.mu3*(1-par.mu0)^5;
            end
            
        end
        if sum(xor(MutGen(i,:),MutGen(j,:))) == 2
            if ((find(xor(MutGen(i,:),MutGen(j,:)) == 1, 1,'last')) < 7 & (MutGen(i,1:6) == [1 1 1 1 1 1]))
                par.A(i,j) = (par.mu1/2)*(1-par.mu0)^4;
            elseif ((find(xor(MutGen(i,:),MutGen(j,:)) == 1, 1,'last')) < 7 & (MutGen(j,1:6) == [1 1 1 1 1 1]))
                par.A(j,i) = (par.mu2/2)*(1-par.mu0)^4;
            end
        end
    end
end

% -------------------------------------------------------------------------
% -------------- Pre-treatment Steadystate Computation---------------------
% -------------------------------------------------------------------------

tspan     = [0 5000];
Xinit     = zeros(7*length(Res)+2,1);
Xinit(1,1)= par.lambda_T/par.delta_T; % Initial TU
Xinit(2,1)= par.lambda_M/par.delta_M; % Initial MU
Xinit(8,1)= 1; % Initial infecting virus

par.epsilon = 0; % No drug

par.beta_T  = zeros(length(Res),1);
par.CL_T    = zeros(length(Res),1);
par.beta_M  = zeros(length(Res),1);
par.CL_M    = zeros(length(Res),1);

par.beta_T  = (1-s).*par.beta_T_WT;
par.CL_T    = par.beta_T_WT*(1/par.rho_rev - (1-s));
par.beta_M  = (1-s).*par.beta_M_WT;
par.CL_M    = par.beta_M_WT*(1/par.rho_rev - (1-s));

[t Xss] = ode15s(@Kleist_Menz_Huisinga_ODE,tspan,Xinit,[],par);

% -------------------------------------------------------------------------
% ------------------- AFTER DRUG TREATMENT WITH ZDV -----------------------
% -------------------------------------------------------------------------

par.epsilon = epsilon;

par.beta_T  = zeros(length(Res),1);
par.CL_T    = zeros(length(Res),1);
par.beta_M  = zeros(length(Res),1);
par.CL_M    = zeros(length(Res),1);

F = DrugConcFold.*par.Res;

par.CL_T(:,1) = par.beta_T_WT.*(1/(par.rho_rev) - (1-s)./(par.rho_rev + (1 - par.rho_rev)./(1./(1+F))));
par.beta_T(:,1) = par.beta_T_WT.*(1-s)./(par.rho_rev + (1 - par.rho_rev)./(1./(1+F)));
par.CL_M(:,1) = par.beta_M_WT.*(1/(par.rho_rev) - (1-s)./(par.rho_rev + (1 - par.rho_rev)./(1./(1+F))));
par.beta_M(:,1) = par.beta_M_WT.*(1-s)./(par.rho_rev + (1 - par.rho_rev)./(1./(1+F)));

% -------------------------------------------------------------------------
%                               Solve the ODE system
% -------------------------------------------------------------------------

tspan = [0:1:1000];
[t X] = ode15s(@Kleist_Menz_Huisinga_ODE,tspan,Xss(end,:),[],par);

% -------------------------------------------------------------------------
% Compute Selective Advantages of different mutants relative to WT
% (normalized to 1) and also to compare amongst specific mutants
% -------------------------------------------------------------------------

DrugEff_fac_eta = 1./(par.rho_rev + (1-par.rho_rev)./(1./(1+F))); % 1 - eta
SelectiveAdvantage = ((1- s(2:9)).*(DrugEff_fac_eta(2:9)))./(DrugEff_fac_eta(1));

% Compute total viral load

V_tot = X(:,8)+X(:,9); % Wild-type VI and VNI

% Adding all the VI
for i = 15:6:(15 + 6*(length(par.Res)-2))
    V_tot = V_tot + X(:,i);
end

% Adding all the VNI
for i = (6*length(par.Res) + 4):1:(7*length(par.Res) + 2)
    V_tot = V_tot + X(:,i);
end

figure(1)
semilogy(t,V_tot,'b-');xlim([0 270]);ylim([1e7 1e12]);
xlabel('Time (days)'); ylabel('Viral load (in numbers)');

% -------------------------------------------------------------------------
%                Compute abundance of different mutations
% -------------------------------------------------------------------------

Mutation1 = zeros(length(t),1);
Mutation2 = zeros(length(t),1);
Mutation3 = zeros(length(t),1);
Mutation4 = zeros(length(t),1);
Mutation5 = zeros(length(t),1);
Mutation6 = zeros(length(t),1);

for i = 2:length(MutGen)
    if MutGen(i,1) == 1
        Mutation1 = Mutation1 + X(:,15 + 6*(i-2)) + X(:,15+6*(length(MutGen)-2)+i-1);
    end
    if MutGen(i,2) == 1
        Mutation2 = Mutation2 + X(:,15 + 6*(i-2))+ X(:,15+6*(length(MutGen)-2)+i-1);
    end
    if MutGen(i,3) == 1
        Mutation3 = Mutation3 + X(:,15 + 6*(i-2))+ X(:,15+6*(length(MutGen)-2)+i-1);
    end
    if MutGen(i,4) == 1
        Mutation4 = Mutation4 + X(:,15 + 6*(i-2))+ X(:,15+6*(length(MutGen)-2)+i-1);
    end
    if MutGen(i,5) == 1
        Mutation5 = Mutation5 + X(:,15 + 6*(i-2))+ X(:,15+6*(length(MutGen)-2)+i-1);
    end
    if MutGen(i,6) == 1
        Mutation6 = Mutation6 + X(:,15 + 6*(i-2))+ X(:,15+6*(length(MutGen)-2)+i-1);
    end
end

% -------------------------------------------------------------------------
%           Compute fractions of abundance of different mutations
% -------------------------------------------------------------------------

Mutation1_fraction = Mutation1./V_tot;
Mutation2_fraction = Mutation2./V_tot;
Mutation3_fraction = Mutation3./V_tot;
Mutation4_fraction = Mutation4./V_tot;
Mutation5_fraction = Mutation5./V_tot;
Mutation6_fraction = Mutation6./V_tot;

WildType = (X(:,8) + X(:,9))./V_tot;

figure(2)
plot(t,Mutation3_fraction,'b-');xlim([0 400]);ylim([0 1]);xlabel('Time(days)');ylabel('Fractional abundance of mutation');title('K70R');


%--------------------------------------------------------------------------
%----------------- Computing mechanistic waiting times---------------------
% Detection criteria: Total viral load > 500 copies of RNA per ml and a
% mutation constitutes >20% of the total viral abundance
%--------------------------------------------------------------------------
%--------------------------------------------------------------------------

% Mutation 1

for i = 1:length(tspan)
    if (Mutation1_fraction(i) > 0.2) && (V_tot(i) > 1.2e8)
        Rebound_Mutation1 = tspan(i);
        break;
    end
end

% Mutation 2

for i = 1:length(tspan)
    if (Mutation2_fraction(i) > 0.2) && (V_tot(i) > 1.2e8)
        Rebound_Mutation2 = tspan(i);
        break;
    end
end

% Mutation 3

for i = 1:length(tspan)
    if (Mutation3_fraction(i) > 0.2) && (V_tot(i) > 1.2e8)
        Rebound_Mutation3 = tspan(i);
        break;
    end
end

% Mutation 4

for i = 1:length(tspan)
    if (Mutation4_fraction(i) > 0.2) && (V_tot(i) > 1.2e8)
        Rebound_Mutation4 = tspan(i);
        break;
    end
end

% Mutation 5

for i = 1:length(tspan)
    if (Mutation5_fraction(i) > 0.2) && (V_tot(i) > 1.2e8)
        Rebound_Mutation5 = tspan(i);
        break;
    end
end

% Mutation 6

for i = 1:length(tspan)
    if (Mutation6_fraction(i) > 0.2) && (V_tot(i) > 1.2e8)
        Rebound_Mutation6 = tspan(i);
        break;
    end
end

% -------------------------------------------------------------------------
%  Computing waiting time to observe a mutation at each of the sites from
%                                  the model
% -------------------------------------------------------------------------

T_pred = zeros(6,1);
T_pred(1) = Rebound_Mutation1;
T_pred(2) = Rebound_Mutation2;
T_pred(3) = Rebound_Mutation3;
T_pred(4) = Rebound_Mutation4;
T_pred(5) = Rebound_Mutation5;
T_pred(6) = Rebound_Mutation6;

T_pred = T_pred./min(T_pred); % Normalize by smallest T_pred, as in data
T_mech = T_pred;

% Compare mechanistic and statistical waiting times

WaitingTimes = [T_stat T_mech]


end



function dX = Kleist_Menz_Huisinga_ODE(t,X,par)

par.Res = par.Res;

dX = zeros(7*length(par.Res)+2,1);

TU = X(1);
MU = X(2);
T1_Mut = zeros(length(par.Res),1);
T2_Mut = zeros(length(par.Res),1);
TL_Mut = zeros(length(par.Res),1);
M1_Mut = zeros(length(par.Res),1);
M2_Mut = zeros(length(par.Res),1);
VI_Mut = zeros(length(par.Res),1);
VNI_Mut = zeros(length(par.Res),1);

T1_Mut(1) = X(3);
T2_Mut(1) = X(4);
TL_Mut(1) = X(5);
M1_Mut(1) = X(6);
M2_Mut(1) = X(7);
VI_Mut(1) = X(8);
VNI_Mut(1) = X(9);
j = 1;
for i = 2:length(par.Res)
    T1_Mut(i) = X(9+j);
    T2_Mut(i) = X(10+j);
    TL_Mut(i) = X(11+j);
    M1_Mut(i) = X(12+j);
    M2_Mut(i) = X(13+j);
    VI_Mut(i) = X(14+j);
    j = j+6;
end

j = 15+6*(length(par.Res)-2)+1;
for i = 2:length(par.Res)
    VNI_Mut(i) = X(j);
    j = j+1;
end

T1 = T1_Mut;
T2 = T2_Mut;
TL = TL_Mut;
M1 = M1_Mut;
M2 = M2_Mut;
VI = VI_Mut;
VNI = VNI_Mut;

dTU        = par.lambda_T + par.delta_PIC_T*sum(T1) - par.delta_T*TU - TU*(sum(par.beta_T.*VI));
dMU        = par.lambda_M + par.delta_PIC_M*sum(M1) - par.delta_M*MU - MU*(sum(par.beta_M.*VI));
dT1_Mut(1) = par.beta_T(1)*VI_Mut(1)*TU - (par.delta_T1 + par.delta_PIC_T + par.k_T)*T1_Mut(1);
dM1_Mut(1) = par.beta_M(1)*VI_Mut(1)*MU - (par.delta_M1 + par.delta_PIC_M + par.k_M)*M1_Mut(1);
dVI_Mut(1) = par.N_T*T2_Mut(1) + par.N_M*M2_Mut(1) - par.CL_VI*VI_Mut(1) - (par.CL_T(1)*TU + par.beta_T(1)*TU + par.CL_M(1)*MU + par.beta_M(1)*MU)*VI_Mut(1);
dVNI_Mut(1)= par.N_T_NI.*T2_Mut(1) + par.N_M_NI*M2_Mut(1) - par.CL_VNI*VNI_Mut(1);

for i = 2:length(par.Res)
    dT1_Mut(i)  = par.beta_T(i)*VI_Mut(i)*TU - (par.delta_T1 + par.delta_PIC_T + par.k_T)*T1_Mut(i);
    dM1_Mut(i)  = par.beta_M(i)*VI_Mut(i)*MU - (par.delta_M1 + par.delta_PIC_M + par.k_M)*M1_Mut(i);
    dVI_Mut(i)  = par.N_T*T2_Mut(i) + par.N_M*M2_Mut(i) - par.CL_VI*VI_Mut(i) - (par.CL_T(i)*TU + par.beta_T(i)*TU + par.CL_M(i)*MU + par.beta_M(i)*MU)*VI_Mut(i);
    dVNI_Mut(i) = par.N_T_NI*T2_Mut(i) + par.N_M_NI*M2_Mut(i) - par.CL_VNI*VNI_Mut(i);
    
end

% -------------------------------------------------------------------------
%         Reactions that form T2 (incorporating the mutations)
% -------------------------------------------------------------------------

for i = 1:length(par.Res)
    
    dT2_Mut(i) = (1-par.p)*par.k_T*par.A(i,:)*T1_Mut + (1-par.p)*((1-par.mu0)^6*par.k_T*T1_Mut(i)) + par.alpha*TL_Mut(i) - par.delta_T2*T2_Mut(i);
    dTL_Mut(i) = par.p*par.k_T*par.A(i,:)*T1_Mut + par.p*((1-par.mu0)^6*par.k_T*T1_Mut(i)) - par.alpha*TL_Mut(i) - par.delta_TL*TL_Mut(i);
    dM2_Mut(i) = par.k_M*par.A(i,:)*M1_Mut + ((1-par.mu0)^6*par.k_M*M1_Mut(i)) - par.delta_M2*M2_Mut(i);
    
end

dX(1) = dTU;dX(2) = dMU;dX(3) = dT1_Mut(1); dX(4) = dT2_Mut(1); dX(5) = dTL_Mut(1);dX(6) = dM1_Mut(1);dX(7) = dM2_Mut(1);dX(8) = dVI_Mut(1);dX(9) = dVNI_Mut(1);

j = 1;
for i = 2:length(par.Res)
    dX(j+9) = dT1_Mut(i);
    dX(j+10) = dT2_Mut(i);
    dX(j+11) = dTL_Mut(i);
    dX(j+12) = dM1_Mut(i);
    dX(j+13) = dM2_Mut(i);
    dX(j+14) = dVI_Mut(i);
    j = j+6;
end

j = 15+6*(length(par.Res)-2)+1;
for i = 2:length(par.Res)
    dX(j) = dVNI_Mut(i);
    j = j+1;
end

end