%%% 
%%% This function specifies the graphical output related to the entire
%%% virtual population. Graphical output specific to some model should be
%%% coded in the corresponding model file.
%%% 
%%% Version: February 10th, 2014. 
%%% For references and citation, please see MAIN script.
%%% 
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the 
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported 
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function [] = GenericPBPKmodel_graphicalOutput(individual,ShowGraphics)


%%% set output units depending on drug class
%%%
ShowGraphics = GenericPBPKmodel_setOutputUnits(individual,ShowGraphics);

%%% plot plasma concentration of entire population
%%%
if strcmp(ShowGraphics.allindividual.plasma,'yes')
    GenericPBPKmodel_plasmaConcentration(individual,ShowGraphics)
end;

%%% plot concentration-time profiles of all tissues for specified
%%% individuals
%%%
if ~isempty(ShowGraphics.allTissues.individual)
    GenericPBPKmodel_plotAllTissues(individual,ShowGraphics)
end;

%%% plot normalized concenration-time profiles for specified individuals
%%%
if ~isempty(ShowGraphics.normConc.individual)
    GenericPBPKmodel_normalizedConcentration(individual,ShowGraphics)
end;

%%% plot 5th,25th, 50th, 75th and 95th percentile of concentration-time
%%% profiles of entire population (if larger than 9 individuals)
%%%
if strcmp(ShowGraphics.percentiles.plasma,'yes') && (length(individual)>=10)
    GenericPBPKmodel_precentiles(individual,ShowGraphics)
end;
    
%%% plot total amount over time for debugging
%%%
if ~isempty(ShowGraphics.TestForLossOfMass.id)
    GenericPBPKmodel_checkForConservationOfAmounts(individual,ShowGraphics)
end;

fprintf('\n');

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function ShowGraphics = GenericPBPKmodel_setOutputUnits(individual,ShowGraphics)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    
%%% set default units depending on drug class
switch individual(1).drug.class

    case 'sMD'
        %%% scale time from [min] to [h]
        ShowGraphics.SF.time = 1/60; 
        ShowGraphics.unit.time = 'h'; 
        
        %%% leave concentration in [mg]/L
        ShowGraphics.SF.conc = 1; 
        ShowGraphics.unit.conc = 'mg/L'; 
        
    case 'mAb'
        %%% scale time from [min] to [day]
        ShowGraphics.SF.time = 1/(60*24); 
        ShowGraphics.unit.time = 'day'; 
        
        %%% leave concentration in [mg]/L
        ShowGraphics.SF.conc = 1; 
        ShowGraphics.unit.conc = 'mg/L'; 
        
        %%% uncomment below in case you want [nmol/L]
        % ShowGraphics.SF.conc = individual(1).drug.SF.mg_to_nmol; 
        % ShowGraphics.unit.conc = 'nmol/L'; 
        
        
    otherwise
        
        %%% leave time in [min] 
        ShowGraphics.SF.time = 1; 
        ShowGraphics.unit.time = 'min'; 
        
        %%% leave concentration in [mg]/L
        ShowGraphics.SF.conc = 1; 
        ShowGraphics.unit.conc = 'mg/L'; 
        
end;

ShowGraphics.upperLimitY = 10*ShowGraphics.lowerLimitY;

end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_plasmaConcentration(individual,ShowGraphics)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    
%%% plot plasma concentrations for entire population

T = individual(1).T; figNr = ShowGraphics.allindividual.figNr; 
SF = ShowGraphics.SF; unit = ShowGraphics.unit;

%%% default limits of y axis
%%%
ymin = ShowGraphics.lowerLimitY;
ymax = ShowGraphics.upperLimitY;

figure(figNr); clf; hold on;
for id=1:length(individual)
    
    t = individual(id).stdout.t*SF.time;
    C = individual(id).stdout.C.tis*SF.conc;
    color = char(individual(id).color);
    
    if ~strcmp(individual(id).model(1).type,'ExperimentalData')
        plot(t,C(:,T.pla),color,'LineWidth',1.3);
        xlim([0, t(end)]); ylim([ymin ymax]);
    else
        ind = find(isfinite(C(:,T.pla)));
        plot(t(ind),C(ind,T.pla),color);
        plot(t(ind),C(ind,T.pla),'d','MarkerEdgeColor','k','MarkerFaceColor','g','MarkerSize',7);
        xlim(individual(id).stdout.limits.t(T.pla,:)*SF.time), ylim([ymin ymax]);
    end;
    
    if ~isnan(max(C(:,T.pla)))
        ymax = max([ymax, max(C(:,T.pla))]);
    end;
    
end;
hold off; grid on; 

title(sprintf('Compound = %s',individual(1).drug.name));
xlabel(sprintf('t[%s]',unit.time)); 
ylabel(sprintf('C [%s] in venous plasma',unit.conc)); ylim([ymin 1.1*ymax]); 
set(gca,'YScale',ShowGraphics.allindividual.YScale);

fprintf('\n');

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_plotAllTissues(individual,ShowGraphics)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% Give detailed plots for all tissues for specified individuals -----

T = individual(1).T; figNr = ShowGraphics.allTissues.figNr; 
SF = ShowGraphics.SF; unit = ShowGraphics.unit;

if sum(ismember(ShowGraphics.allTissues.individual,1:length(individual))==0)
    GenericPBPKmodel_ReportErrorMessage('ShowGraphics.allTissues.individual contains invalid ids!');
end;

%%% default limits of y axis
%%%
ymin = ShowGraphics.lowerLimitY;
ymax = ShowGraphics.upperLimitY;

figure(figNr); clf;
for id = ShowGraphics.allTissues.individual
    
    hold on;
    t = individual(id).stdout.t*SF.time;
    C = individual(id).stdout.C.tis*SF.conc;
    color = char(individual(id).color);
    
    %%% determine max concentration value for all non NaN concentration
    %%% time profiles
    %%%
    nonNaNtissues = find(isfinite(max(C)));
    if ~isempty(nonNaNtissues)
        ymax = max(ymax,max(max(C(:,nonNaNtissues))));  % take max of previous ymax and current one
    end;
    
    %%% adjust ymin, if too small
    %%%
    if ymax<=ymin
        fprintf('\n'); fprintf(2,'>>> ShowGraphics.lowerLimitY lowered, since it was smaller than Cmax! <<< \n');
        ymin = 1e-3*ymax;
    end;
    
    for k = 1:12
        
        tissues = [T.ven T.adi T.bon T.bra T.gut T.hea T.kid T.liv T.lun T.mus T.ski T.spl];
        
        subplot(3,4,k); hold on;
        tis = tissues(k);
        
        if ~strcmp(individual(id).model(1).type,'ExperimentalData')
            plot(t,C(:,tis),color,'LineWidth',1.1);
            xlim([0, t(end)]); ylim([ymin ymax]);
        else
            ind = find(isfinite(C(:,tis)));
            plot(t(ind),C(ind,tis),color);
            plot(t(ind),C(ind,tis),'d','MarkerEdgeColor','k','MarkerFaceColor','g','MarkerSize',7)
            xlim(individual(id).stdout.limits.t(tis,:)*SF.time), ylim([ymin ymax]);
        end;
        xlabel(sprintf('t[%s]',unit.time));
        ylabel(sprintf('C [%s] in %s',unit.conc,T.name{tis})); ylim([ymin 1.1*ymax]);
        set(gca,'YScale',ShowGraphics.allTissues.YScale);
        
    end;
    subplot(3,4,2); title(sprintf('drug %s',individual(id).drug.name),'Fontsize',15);
    set(gca,'YScale',ShowGraphics.allTissues.YScale);
    
    if strcmp(ShowGraphics.allTissues.oneFigOnly,'no') && id~=ShowGraphics.allTissues.individual(end)
        hold off;
        figNr = figNr + 1;
        figure(figNr); clf;
    end;
    
    
end;


end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_normalizedConcentration(individual,ShowGraphics)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% Plot all tissues for specified individuals in a single plot (not sclaed
%%% and scaled with the partition coefficient
    
T = individual(1).T; figNr = ShowGraphics.normConc.figNr; 
SF = ShowGraphics.SF; unit = ShowGraphics.unit;

if sum(ismember(ShowGraphics.normConc.individual,1:length(individual))==0)
    GenericPBPKmodel_ReportErrorMessage('ShowGraphics.individual.allTissues contains invalid ids!');
end;

for id = ShowGraphics.normConc.individual
    
    if strcmp(individual(id).model(1).type,'ExperimentalData')
        GenericPBPKmodel_ReportErrorMessage('ShowGraphics.individual.allTissues not defined for experimental data!');
    end;


    figure(figNr); clf;

    %%% -------------------------------------------------------------------
    %%% plot all concentration-time profiles in a single plot
    %%%
    
    subplot(1,2,1); hold on;
    t = individual(id).stdout.t*SF.time;
    C = individual(id).stdout.C.tis*SF.conc;
    
    %%% default limits of y axis
    %%%
    ymin = ShowGraphics.lowerLimitY;
    ymax = ShowGraphics.upperLimitY;
    
    %%% determine max concentration value for all non NaN concentration
    %%% time profiles
    nonNaNtissues = find(isfinite(max(C)));
    if ~isempty(nonNaNtissues)
        ymax = max(ymax,max(max(C(:,nonNaNtissues))));  % take max of previous ymax and current one
    end;
    
    for tis = T.tissueDB
        plot(t,C(:,tis),T.style.(T.name{tis}),'LineWidth',1.3);
    end;
    title(sprintf('individual(%d), drug %s',id,individual(id).drug.name),'Fontsize',15);
    xlabel(sprintf('t[%s]',unit.time)); xlim([0, t(end)]); 
    ylabel(sprintf('C [%s] in %s',unit.conc,T.name{tis})); ylim([ymin 1.1*ymax]);    
    set(gca,'YScale',ShowGraphics.allTissues.YScale); hold off;
    
    
    %%% -------------------------------------------------------------------
    %%% plot all normalized concentration-time profiles in a single plot
    %%%
    subplot(1,2,2); hold on;
    
    nC = individual(id).stdout.nC.tis*SF.conc;
    
    %%% default limits of y axis
    %%%
    ymin = ShowGraphics.lowerLimitY;
    ymax = ShowGraphics.upperLimitY;

    %%% determine max concentration value for all non NaN concentration
    %%% time profiles
    nonNaNtissues = find(isfinite(max(nC)));
    if ~isempty(nonNaNtissues)
        ymax = max(ymax,max(max(nC(:,nonNaNtissues))));  % take max of previous ymax and current one
    end;
    
    for tis = T.tissueDB
        plot(t,nC(:,tis),T.style.(T.name{tis}),'LineWidth',1.3);
    end;
    title(sprintf('Normalized drug concentration'),'Fontsize',15);
    legend(T.name(T.tissueDB),'Location','SouthEast');
    xlabel(sprintf('t[%s]',unit.time)); xlim([0, t(end)]); 
    ylabel(sprintf('nC [%s] in %s',unit.conc,T.name{tis})); ylim([ymin 1.1*ymax]);    
    set(gca,'YScale',ShowGraphics.allTissues.YScale); hold off;

    %%% plot profiles of for next individual in new figure
    %%%
    figNr = figNr + 1;
    
end;

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_precentiles(individual,ShowGraphics)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% Plot percentiles for venous blood concentrations for entire population
%%% Only applicable for population of size>=10

T = individual(1).T;  figNr = ShowGraphics.percentiles.figNr;
SF = ShowGraphics.SF; unit = ShowGraphics.unit;

if length(individual(1).study.observationTime)==2
    beep;
    fprintf(2,'   For the percentile plot to work you have to enforce identical\n');
    fprintf(2,'   time vectors for all individuals. Suggestion: set \n\n');
    fprintf(2,'     study.observationTime = [0:dt:tend] \n\n');
    fprintf(2,'   with, e.g, dt=1 and tend=''observation time span'' in the main script.\n\n');
    return;
end;

%%% determine sex of individual for coloring purpose
%%%
BW = NaN*ones(1,length(individual)); BH = NaN*ones(1,length(individual));
ind_male = []; ind_female = []; ind_uni = [];

for id=1:length(individual)
    BW(id) = individual(id).BW; BH(id) = individual(id).BH;
    switch individual(id).sex
        case 'male';   
            ind_male = [ind_male, id];
        case 'female'; 
            ind_female = [ind_female, id];
        case 'uni';    
            ind_uni = [ind_uni, id];
    end;
end;

figure(figNr); clf;

%%% plot BW versus BH for all individulas
%%%
subplot(1,2,1);
plot(BW(ind_male),BH(ind_male),'ro',BW(ind_female),BH(ind_female),'bo',BW(ind_uni),BH(ind_uni),'ko');
xlabel('BW'); ylabel('BH');
fprintf('\n\n   Legend Fig.%d. Left: male (r), female (b), uni (k). Right: 5th, 25th. 50th, 75th and 95th percentiles\n\n',ShowGraphics.percentiles.figNr);

tspan = individual(1).stdout.t*SF.time;
if length(tspan)<5
    fprintf(' Only %d time points specified in study.observationTime! Please increase number for more meaningful graphical representation!');
    beep;
end;

%%% combine all plasma concentration-time profiles in a single matrix
%%%
ind = 1:length(tspan);
C_pla = NaN*ones(length(individual),length(individual(1).stdout.t(ind)));
for id=1:length(individual)
    C_pla(id,:) = individual(id).stdout.C.tis(ind,T.pla)';
end;

subplot(1,2,2);
percentile = prctile(C_pla,[5 25 50 75 95])';

hold on;
h = area(tspan(ind),[percentile(:,1)'; diff(percentile,1,2)']');
set(get(h(1),'Children'),'FaceColor',[1 1 1],'LineStyle','none');
set(get(h(2),'Children'),'FaceColor',[.9 .9 .9],'LineStyle','none');
set(get(h(3),'Children'),'FaceColor',[.8 .8 .8],'LineStyle','none');
set(get(h(4),'Children'),'FaceColor',[.8 .8 .8],'LineStyle','none');
set(get(h(5),'Children'),'FaceColor',[.9 .9 .9],'LineStyle','none');

plot(tspan(ind),percentile(:,3),'r-',tspan(ind),percentile(:,[1 5]),'k-',...
    tspan(ind),percentile(:,[2 4]),'k--','LineWidth',1.3);
set(gca,'Layer','top');

title(sprintf('Compound = %s',individual(id).drug.name));
xlabel(sprintf('t[%s]',unit.time)); 
ylabel(sprintf('C [%s] in venous plasma',unit.conc));
xlim([0, tspan(end)]);

ymin = ShowGraphics.lowerLimitY;
ymax = max(max(C_pla));
if ymax<=ymin
    fprintf('\n'); fprintf(2,'>>> ShowGraphics.lowerLimitY lowered, since it was smaller than Cmax! <<< \n');
    ymin = 1e-3*ymax;
end;
grid on; ylim([ymin ymax]);
set(gca,'YScale',ShowGraphics.percentiles.YScale);


end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_checkForConservationOfAmounts(individual,ShowGraphics)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% Give detailed plots for amounts (total, metabolized and sum) for specified individuals -----

T = individual(1).T; figNr = ShowGraphics.TestForLossOfMass.figNr; 
SF = ShowGraphics.SF; unit = ShowGraphics.unit;

if sum(ismember(ShowGraphics.TestForLossOfMass.id,1:length(individual))==0)
    GenericPBPKmodel_ReportErrorMessage('ShowGraphics.TestForLossOfMass.id contains invalid ids!');
end;

figure(figNr); clf; hold on;
for id = ShowGraphics.TestForLossOfMass.id
    
    t = individual(id).stdout.t*SF.time;
    A.body    = individual(id).stdout.A.body;
    A.metab   = individual(id).stdout.A.metab;
    A.GItract = individual(id).stdout.A.GItract;
    A.IVbag   = individual(id).stdout.A.IVbag;
    A.all     = A.body + A.metab + A.GItract + A.IVbag;
    color = char(individual(id).color);
    
    plot(t,A.all/A.all(1),color,'LineWidth',1.3);
    xlabel(sprintf('t[%s]',unit.time)); 
    ylabel('rel change in (A_{body}+A_{metab}+A_{GItract}+A_{IVbag})');
    xlim([0, t(end)/60]); ylim([0.1 10]);
    set(gca,'YScale','log');
    
    title('If you do not see a straight line (for single dose), check your model equations!');
    
    
end;

end

%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++



