%%% 
%%% This function specifies a lumped PBPK model for mAb with
%%% extravasation-limited tissue distribution and no target
%%% 
%%% Version: February 10th, 2014. 
%%% For references and citation, please see MAIN script.
%%% 
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the 
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported 
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function individual = GenericPBPKmodel_mAb_lumpedPBPK_xCMT_extravasationLimited(individual)

%%% check whether drug class is really mAb
%%%
if ~strcmp(individual.drug.class,'mAb')
    message = sprintf('Model >>mAb_PBPK_11CMT_extravasationLimited<< not defined for drug type %s!',individual.drug.class);
    GenericPBPKmodel_ReportErrorMessage(message);
end;

%%% define model specific parameters
%%%
individual = GenericPBPKmodel_defineModelParameters(individual);

%%% simulate PBPK model 
%%%
model = individual.model; S = model.S; study = individual.study;
X0 = model.X0; sim.t = []; sim.X = []; 

for d=1:study.numberOfDosing
    
    %%% account for dosing for each new dosing interval
    X0(S.bolus)   = X0(S.bolus) + study.bolus.dose*model.SF.mg_to_nmol/model.V.iv_bolus;
    X0(S.IVbag)   = study.infusion.dose*model.SF.mg_to_nmol; 

    %%% solve system of ODEs 'GenericPBPKmodel_RHS' with initial conditions 'X0'
    [t,X] = ode15s(@GenericPBPKmodel_RHS,study.observationTime,X0',[],individual);
    
    %%% transform relative to absolute time and modify last time point
    %%% to allow for trough value measurement
    Tend  = max(study.observationTime);
    t = t + (d-1)*Tend; 
    if d~=study.numberOfDosing
        t(end)  = t(end) - 1e-10; 
    end;
    
    %%% store current output in sim structure
    sim.t = [sim.t; t];
    sim.X = [sim.X; X];

    X0 = X(end,:)'; 

end;

%%% determine standart simulation output
%%%
individual = GenericPBPKmodel_determineStandartOutput(individual,sim);

%%% graphical output specific to this model
%%%
GenericPBPKmodel_specificGraphicalOutput(individual);

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_defineModelParameters(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


model   = individual.model;

%%% -----------------------------------------------------------------------
%%% starting point of the lumped PBPK model is the default 
%%% well-stirred vascular&tissue model with 13 compartments
%%% This part is identical to the corresponding part in 
%%% GenericPBPK_sMD_wellstirredTissueModel.m
%%%
%%% -----------------------------------------------------------------------

%%% tissue indexing for detailed PBPK model that is to be lumped
%%%
T = individual.T;

T.allTis      = [T.lun T.adi T.hea T.kid T.mus T.bon T.ski T.gut T.spl T.liv T.pla];
T.allTisExPla = [T.lun T.adi T.hea T.kid T.mus T.bon T.ski T.gut T.spl T.liv      ];
T.tissueDB    = T.allTis; % specify which tissues are part of the general tissue DB

T.visTis      = [T.gut T.spl T.liv];
T.nonvisTis   = [T.lun T.adi T.hea T.kid T.mus T.bon T.ski];

T.initialize.tis.NaN   = NaN*ones(1,max(T.allTis));

%%% -----------------------------------------------------------------------
%%% Define PBPK parameters 
%%%

a   = T.allTis;
mAb = individual.drug; 

%%% tissue volumes 
%%%
V.tis      = T.initialize.tis.NaN;
V.tis(a)   = individual.V.tis(a);

%%% blood flows (ensure closed circulatory system!)
%%%
Q.blo    = T.initialize.tis.NaN;
Q.blo(T.allTisExPla) = individual.Q.blo(T.allTisExPla);

Q.pla    = (1-individual.hct)*Q.blo; % peripheral plasma flow

%%% lymph flow as fraction of peripheral plasma and blood flow
%%%
fLymph = T.initialize.tis.NaN;
fLymph(T.visTis)    = 0.02;
fLymph(T.nonvisTis) = 0.04;

Q.lymph = fLymph.*Q.pla;
Q.lymph(T.pla) = sum(Q.lymph(T.allTisExPla));

%%% blood-to-plasma ratio assuming that mAb do not distribute or bind to
%%% erythrocytes
%%%
BP = (1-individual.hct);

%%% define vascular reflection coefficints
%%%
sigma.vas       = T.initialize.tis.NaN;
sigma.vas([T.liv T.spl T.gut])       = 0.90;
sigma.vas([T.hea T.kid T.lun])       = 0.95;
sigma.vas([T.mus T.ski T.adi T.bon]) = 0.98;
sigma.vas(T.pla) = 1 - sum( Q.lymph(T.allTisExPla).*(1-sigma.vas(T.allTisExPla)) ) /Q.lymph(T.pla); 


%%% define elimination-corrected tissue partition coefficients
eK.tis = T.initialize.tis.NaN;

useABCs = 0;
if isfield(model,'useABCs') && strcmp(model.useABCs,'yes')
    useABCs = 1;
    fprintf(2,'   >> ABCs from Shah and Betts 2013 used <<'); fprintf('\n');
end;

if useABCs ==1
    method.name = 'ShahBetts'; drug = individual.drug;
    ABC = GenericPBPKmodel_TissuePartitionCoefficients(individual,drug,method);
    eK.tis(T.allTis) = ABC(T.allTis)./(1-sigma.vas(T.allTis));
else
    eK.tis(T.hea) = 2.322;
    eK.tis(T.kid) = 2.576;
    eK.tis(T.liv) = 1.324;
    eK.tis(T.lun) = 2.152;
    eK.tis(T.gut) = 0.623;
    eK.tis(T.spl) = 0.303;
    eK.tis(T.ski) = 6.270;
    eK.tis([T.mus T.adi T.bon])  = 1.695;
end;
eK.tis(T.pla) = 1/(1-sigma.vas(T.pla));


%%% tissue-specific extraction ratio
E.tis = T.initialize.tis.NaN;
E.tis(T.allTis) = 0;

%%% tissue plasma clearance
CLpla.tis = T.initialize.tis.NaN;
CLpla.tis(T.allTis) = 0;

scenario = 7;
if isfield(model,'scenario')
    scenario = model.scenario;
    fprintf(2,'   >> scenario No.%d used <<',scenario); fprintf('\n');
end;

switch scenario
    case 2;
        E.tis([T.adi T.liv T.mus T.spl]) = 0.029;
    case 4;
        E.tis(T.ski) = 0.095;
    case 5;
        E.tis(T.mus) = 0.14;
    case 6;
        E.tis(T.liv) = 0.042;
    case 7;
        CLpla.tis(T.pla) = 9.0278e-8; % 9.0278e-08 L/min = 9.e-8 * 1e3 * 60*24 mL/day = 0.13 mL/day (as in the article)
    case 8;
        E.tis(T.spl) = 0.5;
        CLpla.tis(T.pla) = 9.0278e-8 - E.tis(T.spl)*Q.lymph(T.spl)*(1-sigma.vas(T.spl));
    otherwise
        GenericPBPKmodel_ReportErrorMessage('Unknown scenario!');
end; 
CLpla.tis(T.allTisExPla) = E.tis(T.allTisExPla).*Q.lymph(T.allTisExPla).*(1-sigma.vas(T.allTisExPla));


%%% elimination corrected partition coefficients
%%%
K.tis = eK.tis./(1-E.tis); 

%%% define intrinsic tissue clearance
%%%
CLint.tis = T.initialize.tis.NaN;
CLint.tis = Q.lymph./K.tis.*E.tis./(1-E.tis);

%%% account for FcRn status 
if strcmp(model.FcRn_status,'knock-out')
    
    %%% define knock-out extraction ratio and clearance
    %%% Note: elimination corrected partition coefficients
    %%% (and thus also the ABCs) are assumed to be the
    %%% same as for the wild-type (supported by Shah & Betts, 2013)
    facE = 70;
    E.tis = facE*E.tis./(1-E.tis + facE*E.tis);  
    K.tis = eK.tis./(1-E.tis);
    CLint.tis = Q.lymph./K.tis.*E.tis./(1-E.tis);

    if scenario~=1
        GenericPBPKmodel_ReportErrorMessage('Knock-out case has been implemented only for scenario 1!')
    end;
    
    CLpla.tis = 23*CLpla.tis;
end;

%%% double check the p.o. dose = 0
if individual.study.po.dose>0
    GenericPBPKmodel_ReportErrorMessage('po administration of mAb not supported!');
end;



%%% -----------------------------------------------------------------------
%%% Determine lumped PBPK model parameter values based on the above 
%%% parameter values of the detailed PBPK model
%%%
%%% -----------------------------------------------------------------------


%%% -----------------------------------------------------------------------
%%% Define indexing of lumped PBPK model
%%%

lumping = model.lumping; 

%%% Check, whether lumping contains exactly the same tissues as the above
%%% details PBPK model
%%%
tissues = [];
for n = 1:length(lumping)
    for tis = lumping{n}
        tissues = union(tissues,tis);
    end;
end
if ~isempty(setdiff(T.allTis,tissues))
    message = 'Tissues listed in >>lumping<< do not match the detailed PBPK model to be lumped!';
    GenericPBPKmodel_ReportErrorMessage(message);
end;

S.maxIndex.tis = length(lumping); 
S.allTis = 1:S.maxIndex.tis;

%%% determine lumped compartments containing ven, liv and kid 
for n=S.allTis
    if ismember(T.pla,lumping{n}), S.Cen = n; end;
end;
S.allExCen = setdiff(S.allTis,S.Cen);
S.bolus    = S.Cen; % i.v. bolus administration 

S.initialize.tis.NaN   = NaN*ones(1,S.maxIndex.tis);

%%% dosing and metabolism indexing

S.GItract = 1 + S.maxIndex.tis; % gastro-intestinal tract (for po dosing)
S.IVbag   = 2 + S.maxIndex.tis; % IVbag (for infusion dosing)
S.metab   = 3 + S.maxIndex.tis; % metabolized drug

S.maxIndex.dosing = 3 + S.maxIndex.tis;

%%% additional indexing
%%%

S.maxIndex.all = 0 + S.maxIndex.dosing;

%%% ODE indexing (for relative indexing)
%%%
S.C_lump = 1:S.maxIndex.tis; % all tissues


%%% -----------------------------------------------------------------------
%%% Define lumped PBPK model parameters 
%%%

V.lump      = S.initialize.tis.NaN; 
Q.lump      = S.initialize.tis.NaN;
sigma.lump  = S.initialize.tis.NaN;
E.lump      = S.initialize.tis.NaN;
CLint.lump  = S.initialize.tis.NaN;
CLpla.lump  = S.initialize.tis.NaN;
eK.lump     = S.initialize.tis.NaN;
K.lump      = S.initialize.tis.NaN;

for n = S.allExCen
    
    tis = lumping{n};
    V.lump(n)   = sum(V.tis(tis)); 
    Q.lump(n)   = sum(Q.lymph(tis));
    
    sigma.lump(n) = 1 - sum( Q.lymph(tis).*(1-sigma.vas(tis)) )/Q.lump(n);
    eK.lump(n)    = sum( V.tis(tis).*eK.tis(tis).*(1-sigma.vas(tis)) )/( V.lump(n).*(1-sigma.lump(n)) );
    E.lump(n)     = 1/(Q.lump(n)*(1-sigma.lump(n))) * sum( Q.lymph(tis).*(1-sigma.vas(tis)).*E.tis(tis) );
    K.lump(n)     = eK.lump(n)/(1-E.lump(n));
    CLint.lump(n) = Q.lump(n)/K.lump(n)*E.lump(n)/(1-E.lump(n));
    CLpla.lump(n) = sum( CLpla.tis(tis) );

end;

tis = lumping{S.Cen};
V.lump(S.Cen)   = sum( V.tis(tis) );
Q.lump(S.Cen)   = sum(Q.lump(S.allExCen));
sigma.lump(S.Cen)  = 1 - sum( Q.lump(S.allExCen).*(1-sigma.lump(S.allExCen)) )/Q.lump(S.Cen);
if length(lumping) == 1
    sigma.lump(S.Cen) = 0;
end;
eK.lump(S.Cen)  = sum( V.tis(tis).*(1-sigma.vas(tis)).*eK.tis(tis) )/( V.lump(S.Cen).*(1-sigma.lump(S.Cen)) );
E.lump(S.Cen)   = 1/(Q.lump(S.Cen)*(1-sigma.lump(S.Cen))) * sum( Q.lymph(tis).*(1-sigma.vas(tis)).*E.tis(tis) );
K.lump(S.Cen)   = eK.lump(S.Cen)/(1-E.lump(S.Cen));
CLpla.lump(S.Cen) = sum( CLpla.tis(tis) );

%%% volume for i.v. bolus dosing
%%%
V.iv_bolus = V.lump(S.Cen);

%%% initial condition of ODE
%%%
X0 = zeros(1,S.maxIndex.all);


%%% -----------------------------------------------------------------------
%%% Assign model parameters 
%%%

model.S     = S;
model.T     = T;
model.V     = V;
model.Q     = Q;
model.K     = K;
model.eK    = eK;
model.CLint = CLint;
model.CLpla = CLpla;
model.E     = E;
model.sigma = sigma;
model.BP    = BP;
model.SF    = mAb.SF;
model.X0    = X0;

individual.model = model;
end



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function dX = GenericPBPKmodel_RHS(t,X,individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% initialize output vector
dX = zeros(size(X));

%%% model and indexing
model = individual.model; S = model.S;

%%% variables (always use row vector notation)
C_lump    = X(S.C_lump)';

%%% tissue volumes, blood flows, extraction ratios, clearance etc.
V     = model.V.lump;
Q     = model.Q.lump; 
sigma = model.sigma.lump;
K     = model.K.lump;
eK    = model.eK.lump;
CLint = model.CLint.lump;
CLpla = model.CLpla.lump;
infusion = individual.study.infusion;

infusion_rate = 0;
if (infusion.tend > 0) && (t<=infusion.tend)
    infusion_rate = model.SF.mg_to_nmol * infusion.dose/infusion.tend;
end;

%%% -----------------------------------------------------------------------
%%% START OF ODEs
%%%
VdC_lump = zeros(size(C_lump));

C_pla = C_lump(S.Cen)/( (1-sigma(S.Cen))*eK(S.Cen) );

if length(S.allTis) > 1

    %%% all compartment except the central
    Cmt = S.allExCen;
    
    VdC_lump(Cmt)   = Q(Cmt).*( (1-sigma(Cmt)).*C_pla - C_lump(Cmt)./K(Cmt) ) ...
                     -CLint(Cmt).*C_lump(Cmt);
                 
    %%% central compartment             
    VdC_lump(S.Cen) = sum( Q(Cmt).*C_lump(Cmt)./K(Cmt) ) ...
                     -sum( Q(Cmt).*(1-sigma(Cmt)).*C_pla )...
                     -CLpla(S.Cen)*C_pla + infusion_rate; 

else
    
    %%% central compartment
    VdC_lump(S.Cen)  = -CLpla(S.Cen)*C_pla + infusion_rate;

end;

%%% drug amount in IVbag for infusion
dA_IVbag = -infusion_rate;

%%% metabolized and excreted compound (mass)
dA_metab = sum( CLint(S.allExCen).*C_lump(S.allExCen) ) + CLpla(S.Cen)*C_pla; 

%%%
%%% END OF ODEs 
%%% -----------------------------------------------------------------------


%%% converting amounts to concentrations
dC_lump = zeros(size(C_lump));
dC_lump(S.allTis) = VdC_lump(S.allTis)./V(S.allTis);

%%% output vector (always in column vector notation)
dX(S.C_lump) = dC_lump';
dX(S.IVbag)  = dA_IVbag';
dX(S.metab)  = dA_metab';

end



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_determineStandartOutput(individual,sim)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model = individual.model; info = individual.info; S = model.S; 
pred  = individual.pred; 

%%% assign predicted profiles
%%%
info.pred.t         = 'simulation time in [min]';
pred.t              = sim.t;

info.pred.C.lump    = 'concentration-time profile in lumped tissue space [nmol/L]';
pred.C.lump         = sim.X(:,S.C_lump);

info.pred.A.tis     = 'amount in tissue space [mg]';
pred.A.tis          = pred.C.lump(:,S.allTis)*diag(model.V.lump(S.allTis));

info.pred.A.body    = 'total amount in the body in [nmol]';
pred.A.body         = sum(pred.A.tis,2);

info.pred.A.GItract = 'remaining amount in the GI tract in [nmol]';
pred.A.GItract      = sim.X(:,S.GItract);

info.pred.A.IVbag   = 'remaining amount in the IVbag in [nmol]';
pred.A.IVbag        = sim.X(:,S.IVbag);

info.pred.A.metab   = 'amount metabolized in [nmol]';
pred.A.metab        = sim.X(:,S.metab);

%%% transform lumped concentrations back to original concentrations
%%% initialize all compartments with NaN and assign only to those
%%% compartments values that are part of the model topology
initialize.NaN = ones(size(pred.t))*model.T.initialize.tis.NaN;

info.pred.C.tis = 'concentration-time profile in tissue space [nmol/L]';
pred.C.tis = initialize.NaN;
for n = S.allTis
    
    ABC_lump = model.eK.lump(n)*(1-model.sigma.lump(n));
    nC  = pred.C.lump(:,n)/ABC_lump;
    for tis = model.lumping{n}
        ABC_tis = (1-model.sigma.vas(tis)).*model.eK.tis(tis);
        pred.C.tis(:,tis) = ABC_tis*nC;
    end;
    
end; 

%%% determine standard output in [mg/L] or [mg]
%%%
stdout = individual.stdout; T = stdout.T;
initialize.NaN = ones(size(pred.t))*T.initialize.tissueDB.NaN; SF = model.SF; 

info.stdout.t         = 'simulation time in [min]';
stdout.t              = pred.t;

info.stdout.C.tis     = 'concentration-time profile in tissue space [mg/L]';
stdout.C.tis          = initialize.NaN;
for k = intersect(T.tissueDB,model.T.tissueDB)
    stdout.C.tis(:,k) = SF.nmol_to_mg*pred.C.tis(:,k);
end;

stdout.C.tis(:,T.pla) = SF.nmol_to_mg*pred.C.tis(:,T.pla);
stdout.C.tis(:,T.blo) = stdout.C.tis(:,T.pla)*model.BP;
stdout.C.tis(:,[T.ven T.art]) = stdout.C.tis(:,T.blo)*[1 1];

info.stdout.nC.tis    = 'normalized concentration-time profile (nC.tis = C.tis/eK.tis) in tissue space [mg/L] ';
stdout.nC.tis         = initialize.NaN;
for k = intersect(T.tissueDB,model.T.tissueDB)
    stdout.nC.tis(:,k)  = stdout.C.tis(:,k) / ( model.eK.tis(k).*(1-model.sigma.vas(k)) ) ;
end;
stdout.nC.tis(:,[T.ven T.art]) = stdout.C.tis(:,T.blo)*([1 1]/model.eK.tis(T.blo));

if model.correct_for_residual_blood 
    resBlo = individual.fVtis.res; resPla = (1-individual.hct)*resBlo;
    for k = intersect(T.tissueDB,setdiff(model.T.tissueDB,[T.pla,T.blo,T.ven,T.art]))
        stdout.C.tis(:,k) = (1-resBlo(k))*stdout.C.tis(:,k) + resPla(k)*stdout.C.tis(:,model.T.pla);
    end;
end;

info.stdout.A.body    = 'total amount in the body in [mg]';
stdout.A.body         = SF.nmol_to_mg*pred.A.body;

info.stdout.A.GItract = 'remaining amount in the GI tract in [mg]';
stdout.A.GItract      = SF.nmol_to_mg*pred.A.GItract;

info.stdout.A.IVbag   = 'remaining amount in the IVbag in [mg]';
stdout.A.IVbag        = SF.nmol_to_mg*pred.A.IVbag;

info.stdout.A.metab   = 'amount metabolized in [mg]';
stdout.A.metab        = SF.nmol_to_mg*pred.A.metab;


individual.info   = info;
individual.pred   = pred;
individual.stdout = stdout;

end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_specificGraphicalOutput(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model = individual.model; 

CLpla_tot = sum(model.CLpla.lump(model.S.allTis));


end



%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

